<?php

declare(strict_types=1);

namespace App\Services\Credit;

use App\Repositories\Contracts\Credit\CreditRepositoryInterface;
use Illuminate\Database\Eloquent\Model;

class CreditService
{
    public function __construct(
        private CreditRepositoryInterface $creditRepository
    ) {}

    /**
     * Get the base price per credit
     */
    public function getCreditPrice(): float
    {
        return $this->creditRepository->getCreditPrice();
    }

    /**
     * Get the credit currency
     */
    public function getCreditCurrency(): string
    {
        return $this->creditRepository->getCreditCurrency();
    }

    /**
     * Check if bulk discounts are enabled
     */
    public function areBulkDiscountsEnabled(): bool
    {
        return $this->creditRepository->areBulkDiscountsEnabled();
    }

    /**
     * Get all bulk discount tiers
     */
    public function getBulkDiscountTiers(): array
    {
        return $this->creditRepository->getBulkDiscountTiers();
    }

    /**
     * Calculate the best discount percentage for a given quantity
     */
    public function calculateDiscountPercentage(int $quantity): float
    {
        return $this->creditRepository->calculateDiscountPercentage($quantity);
    }

    /**
     * Calculate the total price for a given quantity of credits
     */
    public function calculateTotalPrice(int $quantity): array
    {
        return $this->creditRepository->calculateTotalPrice($quantity);
    }

    /**
     * Get the minimum credits per purchase limit
     */
    public function getMinCreditsPerPurchase(): int
    {
        return $this->creditRepository->getMinCreditsPerPurchase();
    }

    /**
     * Get the maximum credits per purchase limit
     */
    public function getMaxCreditsPerPurchase(): int
    {
        return $this->creditRepository->getMaxCreditsPerPurchase();
    }

    /**
     * Check if credit expiry is enabled
     */
    public function isCreditExpiryEnabled(): bool
    {
        return $this->creditRepository->isCreditExpiryEnabled();
    }

    /**
     * Get credit expiry period in days
     */
    public function getCreditExpiryDays(): int
    {
        return $this->creditRepository->getCreditExpiryDays();
    }

    /**
     * Check if credit refunds are enabled
     */
    public function areCreditRefundsEnabled(): bool
    {
        return $this->creditRepository->areCreditRefundsEnabled();
    }

    /**
     * Check if credit transfers are enabled
     */
    public function areCreditTransfersEnabled(): bool
    {
        return $this->creditRepository->areCreditTransfersEnabled();
    }

    /**
     * Validate quantity against limits
     */
    public function validateQuantityLimits(int $quantity): void
    {
        $this->creditRepository->validateQuantityLimits($quantity);
    }

    /**
     * Get pricing tiers for display (useful for frontend)
     */
    public function getPricingTiers(): array
    {
        return $this->creditRepository->getPricingTiers();
    }

    /**
     * Format currency amount for display
     */
    public function formatCurrency(float $amount): string
    {
        return $this->creditRepository->formatCurrency($amount);
    }

    /**
     * Calculate price breakdown with detailed information
     */
    public function getDetailedPricing(int $quantity): array
    {
        $pricing = $this->calculateTotalPrice($quantity);

        return [
            'base_info' => [
                'credit_price' => $this->getCreditPrice(),
                'currency' => $this->getCreditCurrency(),
                'formatted_credit_price' => $this->formatCurrency($this->getCreditPrice()),
            ],
            'quantity_info' => [
                'requested_quantity' => $quantity,
                'min_quantity' => $this->getMinCreditsPerPurchase(),
                'max_quantity' => $this->getMaxCreditsPerPurchase(),
                'is_valid_quantity' => $quantity >= $this->getMinCreditsPerPurchase() &&
                    $quantity <= $this->getMaxCreditsPerPurchase(),
            ],
            'discount_info' => [
                'bulk_discounts_enabled' => $this->areBulkDiscountsEnabled(),
                'available_tiers' => $this->getBulkDiscountTiers(),
                'applied_discount_percentage' => $pricing['discount_percentage'],
            ],
            'pricing' => $pricing,
            'formatted_pricing' => [
                'subtotal' => $this->formatCurrency($pricing['subtotal']),
                'discount_amount' => $this->formatCurrency($pricing['discount_amount']),
                'total' => $this->formatCurrency($pricing['total']),
                'savings' => $this->formatCurrency($pricing['savings']),
            ],
            'features' => [
                'expiry_enabled' => $this->isCreditExpiryEnabled(),
                'expiry_days' => $this->getCreditExpiryDays(),
                'refunds_enabled' => $this->areCreditRefundsEnabled(),
                'transfers_enabled' => $this->areCreditTransfersEnabled(),
            ],
        ];
    }

    // Merged methods from UserCreditHistoryService
    /**
     * Get all credit history records with optional filters
     */
    public function getAll(array $args = [])
    {
        return $this->creditRepository->getAll($args);
    }

    /**
     * Find credit history record by ID
     */
    public function findById(int $id): ?Model
    {
        return $this->creditRepository->findById($id);
    }

    /**
     * Find credit history record by UUID
     */
    public function findByUuid(string $uuid): ?Model
    {
        return $this->creditRepository->findByUuid($uuid);
    }

    /**
     * Create a new credit history record
     */
    public function create(array $data): Model
    {
        return $this->creditRepository->create($data);
    }

    /**
     * Update an existing credit history record
     */
    public function update(Model $model, array $data): bool
    {
        return $this->creditRepository->update($model, $data);
    }

    /**
     * Update credit history record by UUID
     */
    public function updateByUuid(string $uuid, array $data): bool
    {
        return $this->creditRepository->updateByUuid($uuid, $data);
    }

    /**
     * Delete a credit history record
     */
    public function delete(Model $model): bool
    {
        return $this->creditRepository->delete($model);
    }

    /**
     * Delete credit history record by ID
     */
    public function deleteById(int $id): bool
    {
        return $this->creditRepository->deleteById($id);
    }

    /**
     * Delete credit history record by UUID
     */
    public function deleteByUuid(string $uuid): bool
    {
        return $this->creditRepository->deleteByUuid($uuid);
    }
}
