<?php

namespace App\Services\Job;

use App\Repositories\Contracts\Job\JobRepositoryInterface;
use App\Repositories\Contracts\Location\CityRepositoryInterface;
use App\Repositories\Contracts\Location\StateRepositoryInterface;
use App\Repositories\Contracts\Location\CountryRepositoryInterface;

class JobService
{
    protected $jobRepository;

    public function __construct(
        JobRepositoryInterface $jobRepository,
        protected CityRepositoryInterface $cityRepository,
        protected StateRepositoryInterface $stateRepository,
        protected CountryRepositoryInterface $countryRepository
    ) {
        $this->jobRepository = $jobRepository;
    }

    public function getAll($args = [])
    {
        // Convert state slug to ID
        if (!empty($args['state']) && !is_numeric($args['state'])) {
            $state = $this->stateRepository->findBySlug($args['state']);
            if ($state) {
                $args['state'] = $state->id;
            }
        }

        // Convert country slug to ID
        if (!empty($args['country']) && !is_numeric($args['country'])) {
            $country = $this->countryRepository->findBySlug($args['country']);
            if ($country) {
                $args['country'] = $country->id;
            }
        }

        // Convert city slug to ID
        if (!empty($args['city']) && !is_numeric($args['city'])) {

            if (is_numeric($args['country'])) {
                $city = $this->cityRepository->findByCountryAndName($args['country'], $args['city']);
                if ($city) {
                    $args['city'] = $city->id;
                } else {
                    $args['city'] = null;
                }
            } else {
                $args['city'] = null;
            }
        }

        // Sanitize salary filters
        $args = $this->sanitizeSalaryFilters($args);

        return $this->jobRepository->getAll($args);
    }

    public function find($id)
    {
        return $this->jobRepository->find($id);
    }

    public function findBySlug($slug)
    {
        return $this->jobRepository->findBySlug($slug);
    }

    public function findByUuid(string $uuid)
    {
        return $this->jobRepository->findByUuid($uuid);
    }

    public function create(array $data)
    {
        return $this->jobRepository->create($data);
    }

    public function update($id, array $data)
    {
        return $this->jobRepository->update($id, $data);
    }

    public function updateByUuid(string $uuid, array $data)
    {
        return $this->jobRepository->updateByUuid($uuid, $data);
    }

    public function delete($id)
    {
        return $this->jobRepository->delete($id);
    }

    public function getFeaturedJobs($args = [])
    {
        return $this->jobRepository->getFeaturedJobs($args);
    }

    public function getRelatedJobs($args = [])
    {
        return $this->jobRepository->getRelatedJobs($args);
    }

    public function getJobList($args = [], $perPage = 15)
    {
        $paginatedJobs = $this->getAll($args)
            ->has('author')
            ->orderBy('created_at', 'desc')
            ->paginate(
                $perPage,
                pageName: 'job-page',
            );

        return $this->sortFeaturedFirst($paginatedJobs);
    }

    /**
     * Sanitize and validate salary filter parameters
     *
     * @param array $args The filter arguments
     * @return array The sanitized arguments
     */
    protected function sanitizeSalaryFilters(array $args): array
    {
        // Sanitize min_salary
        if (isset($args['min_salary'])) {
            $args['min_salary'] = $this->sanitizeSalaryValue($args['min_salary']);
        }

        // Sanitize max_salary
        if (isset($args['max_salary'])) {
            $args['max_salary'] = $this->sanitizeSalaryValue($args['max_salary']);
        }

        // Ensure logical consistency: min_salary should not be greater than max_salary
        if (!empty($args['min_salary']) && !empty($args['max_salary'])) {
            if ($args['min_salary'] > $args['max_salary']) {
                // Swap values if min is greater than max
                [$args['min_salary'], $args['max_salary']] = [$args['max_salary'], $args['min_salary']];
            }
        }

        return $args;
    }

    /**
     * Sort paginated collection to show featured jobs first
     *
     * @param mixed $paginator The paginated collection
     * @return mixed The paginator with sorted items
     */
    protected function sortFeaturedFirst($paginator)
    {
        // Sort the collection items to show featured jobs first, then by created_at desc
        $sortedItems = $paginator->getCollection()->sortByDesc(function ($job) {
            // Featured jobs get priority (1), then sort by created_at timestamp
            return [
                $job->is_featured ? 1 : 0,
                $job->created_at->timestamp
            ];
        })->values();

        // Replace the collection in the paginator with the sorted one
        $paginator->setCollection($sortedItems);

        return $paginator;
    }

    /**
     * Sanitize individual salary value
     *
     * @param mixed $value The salary value to sanitize
     * @return int|null The sanitized salary value or null if invalid
     */
    protected function sanitizeSalaryValue($value): ?int
    {
        // Remove if empty or non-numeric
        if (empty($value) || !is_numeric($value)) {
            return null;
        }

        // Convert to integer and ensure it's positive
        $salary = (int) $value;

        return $salary > 0 ? $salary : null;
    }
}
