<?php

namespace App\Services\Lead;

use App\DTOs\Lead\LeadAddress\CreateLeadAddressDTO;
use App\Models\Lead\LeadAddress;
use App\Services\Location\CityService;
use App\Services\Location\CountryService;
use App\Services\Location\StateService;

class LeadAddressService
{
    public function __construct(
        protected CountryService $countryService,
        protected StateService $stateService,
        protected CityService $cityService
    ) {}

    /**
     * Create a lead address from DTO
     */
    public function create(CreateLeadAddressDTO $dto): LeadAddress
    {
        // Resolve location IDs if names are provided
        $this->resolveLocationIds($dto);

        return LeadAddress::create($dto->toArray());
    }

    /**
     * Create lead address from request data with location name resolution
     */
    public function createFromLocationData(int $leadId, array $locationData): ?LeadAddress
    {
        // Skip if no meaningful location data
        if (empty($locationData['address']) && empty($locationData['lat']) && empty($locationData['lng'])) {
            return null;
        }

        // Build DTO data
        $dtoData = [
            'lead_id' => $leadId,
            'address_line_1' => $locationData['address'] ?? null,
            'latitude' => isset($locationData['lat']) ? (float) $locationData['lat'] : null,
            'longitude' => isset($locationData['lng']) ? (float) $locationData['lng'] : null,
        ];

        // Add location names if provided
        if (! empty($locationData['city'])) {
            $dtoData['city_name'] = $locationData['city'];
        }

        if (! empty($locationData['state'])) {
            $dtoData['state_name'] = $locationData['state'];
        }

        if (! empty($locationData['state_code'])) {
            $dtoData['state_code'] = $locationData['state_code'];
        }

        if (! empty($locationData['country'])) {
            $dtoData['country_name'] = $locationData['country'];
        }

        if (! empty($locationData['country_code'])) {
            $dtoData['country_code'] = $locationData['country_code'];
        }

        $dto = CreateLeadAddressDTO::fromArray($dtoData);

        return $this->create($dto);
    }

    /**
     * Resolve location IDs from location names in DTO
     */
    protected function resolveLocationIds(CreateLeadAddressDTO $dto): void
    {
        // Only resolve if we have location names and don't already have IDs
        if (! $dto->hasLocationNames()) {
            return;
        }

        // Handle Country
        if ($dto->country_name && $dto->country_code && ! $dto->country_id) {
            $country = $this->countryService->findOrCreateByCodeAndName(
                $dto->country_code,
                $dto->country_name
            );
            $dto->country_id = $country->id;
        }

        // Handle State (requires country_id)
        if ($dto->state_name && $dto->country_id && ! $dto->state_id) {
            $state = $this->stateService->findOrCreateByNameAndCountry(
                $dto->state_name,
                $dto->country_id,
                $dto->state_code ?? null
            );
            $dto->state_id = $state->id;
        }

        // Handle City (requires state_id and country_id)
        if ($dto->city_name && $dto->state_id && $dto->country_id && ! $dto->city_id) {
            $city = $this->cityService->findOrCreateByNameAndState(
                $dto->city_name,
                $dto->state_id,
                $dto->country_id
            );
            $dto->city_id = $city->id;
        }
    }
}
