<?php

namespace App\Services\Lead;

use App\DTOs\Lead\LeadForm\CreateLeadFormDTO;
use App\DTOs\Lead\LeadForm\UpdateLeadFormDTO;
use App\DTOs\Lead\LeadForm\LeadFormFilterDTO;
use App\Models\Lead\LeadForm;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;

class LeadFormService
{
    /**
     * Get filtered and paginated lead forms
     */
    public function getFiltered(LeadFormFilterDTO $filterDTO): LengthAwarePaginator
    {
        $query = LeadForm::query();

        // Apply filters
        if ($filterDTO->search) {
            $query->where(function ($q) use ($filterDTO) {
                $q->where('title', 'like', '%' . $filterDTO->search . '%')
                  ->orWhere('description', 'like', '%' . $filterDTO->search . '%');
            });
        }

        if ($filterDTO->is_active !== null) {
            $query->where('is_active', $filterDTO->is_active);
        }

        if ($filterDTO->match_criteria) {
            $query->where('match_criteria', $filterDTO->match_criteria);
        }

        if ($filterDTO->reception_mode) {
            $query->where('reception_mode', $filterDTO->reception_mode);
        }

        if ($filterDTO->created_by) {
            $query->where('created_by', $filterDTO->created_by);
        }

        // Date filters
        if ($filterDTO->created_from) {
            $query->whereDate('created_at', '>=', $filterDTO->created_from);
        }
        if ($filterDTO->created_to) {
            $query->whereDate('created_at', '<=', $filterDTO->created_to);
        }

        // Load relationships
        $relations = $filterDTO->getRelationsToLoad();
        if (!empty($relations)) {
            $query->with($relations);
        }

        // Load counts if requested
        if ($filterDTO->with_counts) {
            $query->withCount($filterDTO->getCountsToLoad());
        }

        // Apply sorting
        $query->orderBy($filterDTO->order_by, $filterDTO->order_direction);

        // Paginate
        return $query->paginate($filterDTO->per_page);
    }

    /**
     * Create a new lead form
     */
    public function create(CreateLeadFormDTO $dto): LeadForm
    {
        $leadForm = LeadForm::create($dto->toArray());

        // Attach fields if provided
        if ($fieldAttachments = $dto->getFieldAttachments()) {
            $leadForm->formFields()->sync($fieldAttachments);
        }

        // Attach member categories if provided
        if ($categoryAttachments = $dto->getMemberCategoryAttachments()) {
            $leadForm->memberCategories()->sync($categoryAttachments);
        }

        return $leadForm->load('formFields', 'memberCategories');
    }

    /**
     * Find lead form by UUID
     */
    public function findByUuid(string $uuid, array $with = [], array $withCount = []): ?LeadForm
    {
        $query = LeadForm::where('uuid', $uuid);

        if (!empty($with)) {
            $query->with($with);
        }

        if (!empty($withCount)) {
            $query->withCount($withCount);
        }

        return $query->first();
    }

    /**
     * Find lead form by UUID or fail
     */
    public function findByUuidOrFail(string $uuid, array $with = [], array $withCount = []): LeadForm
    {
        $query = LeadForm::where('uuid', $uuid);

        if (!empty($with)) {
            $query->with($with);
        }

        if (!empty($withCount)) {
            $query->withCount($withCount);
        }

        return $query->firstOrFail();
    }

    /**
     * Update lead form
     */
    public function update(string $uuid, UpdateLeadFormDTO $dto): LeadForm
    {
        $leadForm = $this->findByUuidOrFail($uuid);

        if ($dto->hasChanges()) {
            $leadForm->update($dto->toArray());
        }

        // Update field attachments
        if ($fieldAttachments = $dto->getFieldAttachments()) {
            if ($dto->shouldSyncFields()) {
                $leadForm->formFields()->sync($fieldAttachments);
            } else {
                $leadForm->formFields()->attach($fieldAttachments);
            }
        }

        // Update member category attachments
        if ($categoryAttachments = $dto->getMemberCategoryAttachments()) {
            if ($dto->shouldSyncMemberCategories()) {
                $leadForm->memberCategories()->sync($categoryAttachments);
            } else {
                $leadForm->memberCategories()->attach($categoryAttachments);
            }
        }

        return $leadForm->load('formFields', 'memberCategories');
    }

    /**
     * Delete lead form
     */
    public function delete(string $uuid): bool
    {
        $leadForm = $this->findByUuidOrFail($uuid);
        return $leadForm->delete();
    }

    /**
     * Get lead form statistics
     */
    public function getStats(string $uuid): array
    {
        $leadForm = LeadForm::where('uuid', $uuid)
            ->withCount(['visits', 'leads'])
            ->firstOrFail();

        return [
            'total_visits' => $leadForm->visits_count,
            'total_leads' => $leadForm->leads_count,
            'conversion_rate' => $leadForm->visits_count > 0
                ? round(($leadForm->leads_count / $leadForm->visits_count) * 100, 2)
                : 0,
        ];
    }

    /**
     * Find active lead form by UUID for public access
     */
    public function findActiveByUuid(string $uuid, array $with = []): ?LeadForm
    {
        $query = LeadForm::where('uuid', $uuid)
            ->where('is_active', true);

        if (!empty($with)) {
            $query->with($with);
        }

        return $query->first();
    }

    /**
     * Find active lead form by UUID or fail
     */
    public function findActiveByUuidOrFail(string $uuid, array $with = []): LeadForm
    {
        $query = LeadForm::where('uuid', $uuid)
            ->where('is_active', true);

        if (!empty($with)) {
            $query->with($with);
        }

        return $query->firstOrFail();
    }
}
