<?php

namespace App\Services\Lead;

use App\DTOs\Lead\LeadFormVisit\CreateLeadFormVisitDTO;
use App\DTOs\Lead\LeadFormVisit\UpdateLeadFormVisitDTO;
use App\Models\Lead\LeadForm;
use App\Models\Lead\LeadFormVisit;
use App\Repositories\Contracts\Lead\LeadFormVisitRepositoryInterface;
use Illuminate\Http\Request;

class LeadFormVisitService
{
    protected LeadFormVisitRepositoryInterface $repository;

    public function __construct(LeadFormVisitRepositoryInterface $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Track a form visit
     */
    public function trackVisit(LeadForm $leadForm, CreateLeadFormVisitDTO $dto): LeadFormVisit
    {
        return $this->repository->create($leadForm, $dto->toArray());
    }

    /**
     * Auto-capture and track visit from request
     */
    public function autoTrackVisit(Request $request, LeadForm $leadForm): LeadFormVisit
    {
        $visitDTO = CreateLeadFormVisitDTO::autoCapture($request, $leadForm->id);

        return $this->trackVisit($leadForm, $visitDTO);
    }

    /**
     * Find visit by UUID
     */
    public function findByUuid(string $uuid, array $with = []): ?LeadFormVisit
    {
        if (empty($with)) {
            return $this->repository->findByUuid($uuid);
        }

        return $this->repository->findByUuidOrFail($uuid, $with);
    }

    /**
     * Find visit by UUID or fail
     */
    public function findByUuidOrFail(string $uuid, array $with = []): LeadFormVisit
    {
        return $this->repository->findByUuidOrFail($uuid, $with);
    }

    /**
     * Update visit
     */
    public function update(string $uuid, UpdateLeadFormVisitDTO $dto): LeadFormVisit
    {
        $visit = $this->findByUuidOrFail($uuid);

        if ($dto->hasChanges()) {
            return $this->repository->update($visit, $dto->toArray());
        }

        return $visit;
    }

    /**
     * Delete visit
     */
    public function delete(string $uuid): bool
    {
        $visit = $this->findByUuidOrFail($uuid);
        return $this->repository->delete($visit);
    }
}
