<?php

namespace App\Services\Lead;

use App\DTOs\Lead\Lead\CreateLeadDTO;
use App\DTOs\Lead\Lead\UpdateLeadDTO;
use App\DTOs\Lead\Lead\LeadFilterDTO;
use App\Models\Lead\Lead;
use App\Enums\LeadStatus;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;

class LeadService
{
    /**
     * Get filtered and paginated leads
     */
    public function getFiltered(LeadFilterDTO $filterDTO): LengthAwarePaginator
    {
        $query = Lead::query();

        // Apply filters
        if ($filterDTO->lead_form_id) {
            $query->where('lead_form_id', $filterDTO->lead_form_id);
        }

        if ($filterDTO->lead_source) {
            $query->where('lead_source', $filterDTO->lead_source);
        }

        if ($filterDTO->lead_statuses) {
            $query->whereIn('lead_status', $filterDTO->lead_statuses);
        }

        if ($filterDTO->search) {
            $query->where(function ($q) use ($filterDTO) {
                $q->where('first_name', 'like', '%' . $filterDTO->search . '%')
                  ->orWhere('last_name', 'like', '%' . $filterDTO->search . '%')
                  ->orWhere('email', 'like', '%' . $filterDTO->search . '%')
                  ->orWhere('phone', 'like', '%' . $filterDTO->search . '%');
            });
        }

        if ($filterDTO->email) {
            $query->where('email', $filterDTO->email);
        }

        if ($filterDTO->created_by) {
            $query->where('created_by', $filterDTO->created_by);
        }

        if ($filterDTO->assigned_to_user_id) {
            $query->whereHas('users', function ($q) use ($filterDTO) {
                $q->where('user_id', $filterDTO->assigned_to_user_id);
            });
        }

        // Date filters
        if ($filterDTO->created_from) {
            $query->whereDate('created_at', '>=', $filterDTO->created_from);
        }
        if ($filterDTO->created_to) {
            $query->whereDate('created_at', '<=', $filterDTO->created_to);
        }

        // Soft deletes
        if ($filterDTO->with_trashed) {
            $query->withTrashed();
        } elseif ($filterDTO->only_trashed) {
            $query->onlyTrashed();
        }

        // Load relationships
        $relations = $filterDTO->getRelationsToLoad();
        if (!empty($relations)) {
            $query->with($relations);
        }

        // Apply sorting
        $query->orderBy($filterDTO->order_by, $filterDTO->order_direction);

        // Paginate
        return $query->paginate($filterDTO->per_page);
    }

    /**
     * Create a new lead
     */
    public function create(CreateLeadDTO $dto): Lead
    {
        $lead = Lead::create($dto->toArray());
        $metaFields = $dto->getMeta();

        // Assign users if provided
        if ($assignedUsers = $dto->getAssignedUsers()) {
            $userStatuses = $dto->getUserStatuses() ?? [];
            $pivotData = [];

            foreach ($assignedUsers as $userId) {
                $pivotData[$userId] = [
                    'lead_status' => $userStatuses[$userId] ?? null
                ];
            }

            $lead->users()->attach($pivotData);
        }

        // Add metadata if provided
        if (!empty($metaFields)) {
            foreach ($metaFields as $key => $value) {
                $lead->setMeta($key, $value);
            }
        }
        $lead->save();
        
        return $lead->load('leadForm', 'users', 'author');
    }

    /**
     * Find lead by UUID
     */
    public function findByUuid(string $uuid, array $with = []): ?Lead
    {
        $query = Lead::where('uuid', $uuid);

        if (!empty($with)) {
            $query->with($with);
        }

        return $query->first();
    }

    /**
     * Find lead by UUID or fail
     */
    public function findByUuidOrFail(string $uuid, array $with = []): Lead
    {
        $query = Lead::where('uuid', $uuid);

        if (!empty($with)) {
            $query->with($with);
        }

        return $query->firstOrFail();
    }

    /**
     * Update lead
     */
    public function update(string $uuid, UpdateLeadDTO $dto): Lead
    {
        $lead = $this->findByUuidOrFail($uuid);

        if ($dto->hasChanges()) {
            $lead->update($dto->toArray());
        }

        // Update user assignments if provided
        if ($assignedUsers = $dto->getAssignedUsers()) {
            if ($dto->shouldSyncUsers()) {
                $lead->users()->sync($assignedUsers);
            } else {
                $lead->users()->attach($assignedUsers);
            }
        }

        // Update metadata if provided
        if ($meta = $dto->getMeta()) {
            if ($dto->shouldReplaceMeta()) {
                $lead->setMeta($meta);
            } else {
                foreach ($meta as $key => $value) {
                    $lead->setMeta($key, $value);
                }
            }
        }

        return $lead->load('leadForm', 'users', 'author');
    }

    /**
     * Delete lead
     */
    public function delete(string $uuid): bool
    {
        $lead = $this->findByUuidOrFail($uuid);
        return $lead->delete();
    }

    /**
     * Assign lead to users
     */
    public function assignToUsers(string $uuid, array $assignedUserIds, array $userStatuses = [], bool $sync = true): Lead
    {
        $lead = $this->findByUuidOrFail($uuid);

        // Prepare pivot data with statuses
        $pivotData = [];
        foreach ($assignedUserIds as $userId) {
            $pivotData[$userId] = [
                'lead_status' => $userStatuses[$userId] ?? LeadStatus::Pending->value
            ];
        }

        if ($sync) {
            $lead->users()->sync($pivotData);
        } else {
            $lead->users()->attach($pivotData);
        }

        return $lead->load('users');
    }

    /**
     * Update lead status
     */
    public function updateStatus(string $uuid, string $leadStatus, ?int $userId = null): Lead
    {
        $lead = $this->findByUuidOrFail($uuid);

        if ($userId) {
            // Update status for specific user assignment
            $lead->users()->updateExistingPivot($userId, [
                'lead_status' => $leadStatus
            ]);
        } else {
            // Update global lead status
            $lead->update(['lead_status' => LeadStatus::from($leadStatus)]);
        }

        return $lead->load('users');
    }
}
