<?php

declare(strict_types=1);

namespace App\Services\Location;

use App\Models\Location\Area;
use App\Models\Location\City;
use App\Models\User\UserLocation;
use App\Repositories\Location\AreaRepository;
use Illuminate\Support\Collection;

class AreaService
{
    protected AreaRepository $areaRepository;

    public function __construct(AreaRepository $areaRepository)
    {
        $this->areaRepository = $areaRepository;
    }

    /**
     * Create or find areas by names and city ID
     */
    public function createOrFindAreasByNames(array $areaNames, int $cityId): Collection
    {
        $areas = collect();
        
        foreach ($areaNames as $areaName) {
            $areaName = trim($areaName);
            if (empty($areaName)) {
                continue;
            }

            $area = $this->areaRepository->findOrCreateByNameAndCity($areaName, $cityId);
            $areas->push($area);
        }

        return $areas;
    }

    /**
     * Sync areas to a user location
     */
    public function syncAreasToLocation(UserLocation $location, array $areaNames): void
    {
        if (empty($areaNames) || !$location->city_id) {
            $location->areas()->detach();
            return;
        }

        $areas = $this->createOrFindAreasByNames($areaNames, $location->city_id);
        $location->areas()->sync($areas->pluck('id')->toArray());
    }

    /**
     * Get areas by city ID
     */
    public function getAreasByCity(int $cityId): Collection
    {
        return $this->areaRepository->getByCity($cityId);
    }

    /**
     * Search areas by name and city
     */
    public function searchAreas(string $search, ?int $cityId = null): Collection
    {
        return $this->areaRepository->search($search, $cityId);
    }
}