<?php

declare(strict_types=1);

namespace App\Services\Payment;

use App\Models\User\User;
use App\Models\Membership\Purchase;
use App\Models\Membership\Payment;
use App\Enums\PaymentStatus;
use App\Notifications\CreditPurchaseSuccessful;
use App\Notifications\CreditPurchaseFailed;
use Illuminate\Support\Facades\Notification;
use Srmklive\PayPal\Services\PayPal as PayPalClient;
use Exception;

class PayPalPaymentService
{
    /**
     * Create PayPal order for credit purchase
     */
    public function createOrder(Payment $payment): array
    {
        try {
            // Initialize PayPal client
            $paypalClient = $this->initializePayPal();

            $user = $payment->user;
            $amount = (float) $payment->amount;
            $currency = strtoupper($payment->currency);
            $purchase = $payment->purchase;

            // Format amount to 2 decimal places
            $formattedAmount = number_format($amount, 2, '.', '');

            // Create order data for PayPal API
            $orderData = [
                'intent' => 'CAPTURE',
                'purchase_units' => [
                    [
                        'reference_id' => 'credit_purchase_' . $purchase->id,
                        'description' => 'Credit Purchase - ' . $purchase->quantity . ' credits',
                        'custom_id' => $payment->id,
                        'amount' => [
                            'currency_code' => $currency,
                            'value' => $formattedAmount,
                            'breakdown' => [
                                'item_total' => [
                                    'currency_code' => $currency,
                                    'value' => $formattedAmount,
                                ],
                            ],
                        ],
                        'items' => [
                            [
                                'name' => 'Credit Package',
                                'description' => $purchase->quantity . ' credits',
                                'quantity' => '1',
                                'unit_amount' => [
                                    'currency_code' => $currency,
                                    'value' => $formattedAmount,
                                ],
                            ],
                        ],
                    ],
                ],
                'application_context' => [
                    'brand_name' => config('app.name'),
                    'return_url' => route('dashboard.credits.paypal.success'),
                    'cancel_url' => route('dashboard.credits.paypal.cancel'),
                    'user_action' => 'PAY_NOW',
                ],
            ];

            // Create PayPal order
            $response = $paypalClient->createOrder($orderData);

            if (!isset($response['id'])) {
                throw new Exception('PayPal order creation failed: ' . json_encode($response));
            }

            // Get approval URL from links
            $approvalUrl = '';
            foreach ($response['links'] as $link) {
                if ($link['rel'] === 'approve') {
                    $approvalUrl = $link['href'];
                    break;
                }
            }

            // Update payment with transaction ID
            $payment->update(['transaction_id' => $response['id']]);

            // Store order ID in session for retrieval after redirect
            session(['paypal_order_id' => $response['id']]);

            return [
                'success' => true,
                'requires_checkout' => true,
                'checkout_url' => $approvalUrl,
                'order_id' => $response['id'],
                'message' => 'PayPal order created successfully',
            ];
        } catch (Exception $e) {
            // Update payment status to failed
            $payment->update([
                'status' => PaymentStatus::Failed->value,
                'service_response' => json_encode([
                    'error' => $e->getMessage(),
                    'timestamp' => now()->toISOString(),
                ]),
            ]);

            // Send failure notification
            $this->sendFailureNotification($payment, $e->getMessage());

            return [
                'success' => false,
                'message' => 'PayPal order creation failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Capture a PayPal payment to complete the order
     */
    public function capturePayment(string $orderId): array
    {
        try {
            // Initialize PayPal client
            $paypalClient = $this->initializePayPal();

            // Find the payment record by PayPal order ID
            $payment = Payment::where('transaction_id', $orderId)->first();

            if (!$payment) {
                throw new Exception('Payment record not found for order: ' . $orderId);
            }

            // Check if payment is already processed
            if ($payment->status === PaymentStatus::Paid->value) {
                return [
                    'success' => true,
                    'payment' => $payment,
                    'message' => 'Payment already processed',
                ];
            }

            // Capture the payment
            $response = $paypalClient->capturePaymentOrder($orderId);

            // Check if capture was successful
            if ($response['status'] === 'COMPLETED') {
                // Update payment status to paid
                $payment->update([
                    'status' => PaymentStatus::Paid->value,
                    'service_response' => json_encode([
                        'paypal_capture' => $response,
                        'captured_at' => now()->toISOString(),
                    ]),
                ]);

                // Send success notification
                $this->sendSuccessNotification($payment);

                return [
                    'success' => true,
                    'payment' => $payment,
                    'message' => 'Payment captured successfully',
                ];
            } else {
                throw new Exception('PayPal payment capture not completed: ' . $response['status']);
            }
        } catch (Exception $e) {
            if (isset($payment)) {
                $payment->update([
                    'status' => PaymentStatus::Failed->value,
                    'service_response' => json_encode([
                        'error' => $e->getMessage(),
                        'timestamp' => now()->toISOString(),
                    ]),
                ]);

                $this->sendFailureNotification($payment, $e->getMessage());
            }

            return [
                'success' => false,
                'message' => 'Payment capture failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Handle webhook notification for payment completion
     */
    public function handleWebhookNotification(array $eventData): bool
    {
        try {
            if ($eventData['event_type'] !== 'CHECKOUT.ORDER.APPROVED') {
                return false;
            }

            $orderId = $eventData['resource']['id'] ?? null;
            if (!$orderId) {
                return false;
            }

            // Capture the payment
            $captureResult = $this->capturePayment($orderId);

            return $captureResult['success'];
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Initialize PayPal client with configuration
     */
    private function initializePayPal(): PayPalClient
    {
        $provider = new PayPalClient;
        
        // Configure PayPal SDK with credentials
        $config = [
            'mode' => config('paypal.mode', 'sandbox'),
            'sandbox' => [
                'client_id' => config('paypal.sandbox.client_id'),
                'client_secret' => config('paypal.sandbox.client_secret'),
                'app_id' => config('paypal.sandbox.app_id', ''),
            ],
            'live' => [
                'client_id' => config('paypal.live.client_id'),
                'client_secret' => config('paypal.live.client_secret'),
                'app_id' => config('paypal.live.app_id', ''),
            ],
            'payment_action' => 'Sale',
            'currency' => config('paypal.currency', 'USD'),
            'notify_url' => config('paypal.notify_url'),
            'locale' => config('paypal.locale', 'en_US'),
            'validate_ssl' => config('paypal.validate_ssl', true),
        ];
        
        $provider->setApiCredentials($config);
        $provider->getAccessToken();
        
        return $provider;
    }

    /**
     * Send success notification to user
     */
    private function sendSuccessNotification(Payment $payment): void
    {
        $user = $payment->user;
        $purchase = $payment->purchase;

        $user->notify(new CreditPurchaseSuccessful(
            $purchase,
            $payment,
            $purchase->quantity
        ));
    }

    /**
     * Send failure notification to user
     */
    private function sendFailureNotification(Payment $payment, string $errorMessage): void
    {
        $user = $payment->user;
        $purchase = $payment->purchase;

        $user->notify(new CreditPurchaseFailed(
            $purchase,
            $payment,
            $errorMessage
        ));
    }
}
