<?php

declare(strict_types=1);

namespace App\Services\Payment;

use App\Models\User\User;
use App\Models\Membership\Purchase;
use App\Models\Membership\Payment;
use App\Enums\PaymentStatus;
use App\Notifications\CreditPurchaseSuccessful;
use App\Notifications\CreditPurchaseFailed;
use Illuminate\Support\Facades\Notification;
use Laravel\Cashier\Cashier;
use Exception;

class StripePaymentService
{
    /**
     * Process a payment using Laravel Cashier simple charge
     */
    public function processPayment(Payment $payment, array $paymentData): array
    {
        try {
            // Ensure Stripe API key is set
            $this->ensureStripeConfigured();

            $user = $payment->user;
            $amount = (float) $payment->amount;
            $currency = strtolower($payment->currency);

            // Convert amount to cents for Stripe
            $amountInCents = $this->convertToCents($amount, $currency);

            // Create a simple payment using Laravel Cashier's pay method
            $cashierPayment = $user->pay((int) $amountInCents, [
                'currency' => $currency,
                'metadata' => [
                    'payment_id' => $payment->id,
                    'user_id' => $user->id,
                    'purchase_id' => $payment->purchase_id,
                    'type' => 'credit_purchase',
                ],
            ]);

            // Update payment with payment intent ID
            $payment->update([
                'transaction_id' => $cashierPayment->id,
                'status' => PaymentStatus::Pending->value,
                'service_response' => json_encode([
                    'cashier_payment' => $cashierPayment->asStripePaymentIntent()->toArray(),
                    'client_secret' => $cashierPayment->client_secret,
                ]),
            ]);

            // Return client secret for frontend confirmation
            return [
                'success' => true,
                'requires_action' => true,
                'client_secret' => $cashierPayment->client_secret,
                'payment_id' => $cashierPayment->id,
                'message' => 'Payment created successfully using Laravel Cashier',
            ];
        } catch (Exception $e) {
            // Update payment status to failed
            $payment->update([
                'status' => PaymentStatus::Failed->value,
                'service_response' => json_encode([
                    'error' => $e->getMessage(),
                    'timestamp' => now()->toISOString(),
                ]),
            ]);

            // Send failure notification
            $this->sendFailureNotification($payment, $e->getMessage());

            return [
                'success' => false,
                'message' => 'Payment processing failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Process direct payment using Laravel Cashier's charge method
     */
    public function processDirectPayment(Payment $payment, string $paymentMethodId, int $amountInCents, string $currency): array
    {
        try {
            // Ensure Stripe API key is set
            $this->ensureStripeConfigured();

            $user = $payment->user;

            // Process the payment using Laravel Cashier's charge method
            $paymentResult = $user->charge($amountInCents, $paymentMethodId, [
                'currency' => $currency,
                'metadata' => [
                    'payment_id' => $payment->id,
                    'user_id' => $user->id,
                    'purchase_id' => $payment->purchase_id,
                    'type' => 'credit_purchase',
                ],
            ]);

            // Update payment with success result
            $payment->update([
                'transaction_id' => $paymentResult->id,
                'status' => PaymentStatus::Paid->value,
                'service_response' => json_encode([
                    'payment_intent' => $paymentResult->toArray(),
                    'charged_at' => now()->toISOString(),
                ]),
            ]);

            // Send success notification
            $this->sendSuccessNotification($payment);

            return [
                'success' => true,
                'payment_intent' => $paymentResult,
                'message' => 'Payment processed successfully',
            ];
        } catch (Exception $e) {
            $payment->update([
                'status' => PaymentStatus::Failed->value,
                'service_response' => json_encode([
                    'error' => $e->getMessage(),
                    'timestamp' => now()->toISOString(),
                ]),
            ]);

            $this->sendFailureNotification($payment, $e->getMessage());

            return [
                'success' => false,
                'message' => 'Payment processing failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Confirm a payment using Laravel Cashier
     */
    public function confirmPaymentIntent(string $paymentIntentId): array
    {
        try {
            // Ensure Stripe API key is set
            $this->ensureStripeConfigured();

            // Find the payment record
            $payment = Payment::where('transaction_id', $paymentIntentId)->first();

            if (!$payment) {
                throw new Exception('Payment record not found');
            }

            // Check if payment is already processed
            if ($payment->status === PaymentStatus::Paid->value) {
                return [
                    'success' => true,
                    'payment' => $payment,
                    'message' => 'Payment already processed',
                ];
            }

            // Retrieve payment intent directly from Stripe using Laravel Cashier
            $stripePaymentIntent = Cashier::stripe()->paymentIntents->retrieve($paymentIntentId);

            if (!$stripePaymentIntent) {
                throw new Exception('Payment not found in Stripe');
            }

            // Check if payment intent was successful
            if ($stripePaymentIntent->status === 'succeeded') {
                // Update payment status to paid
                $payment->update([
                    'status' => PaymentStatus::Paid->value,
                    'service_response' => json_encode([
                        'payment_intent' => $stripePaymentIntent->toArray(),
                        'confirmed_at' => now()->toISOString(),
                    ]),
                ]);

                // Send success notification
                $this->sendSuccessNotification($payment);

                return [
                    'success' => true,
                    'payment' => $payment,
                    'message' => 'Payment confirmed successfully',
                ];
            } else {
                throw new Exception('Payment not successful: ' . $stripePaymentIntent->status);
            }
        } catch (Exception $e) {
            if (isset($payment)) {
                $payment->update([
                    'status' => PaymentStatus::Failed->value,
                    'service_response' => json_encode([
                        'error' => $e->getMessage(),
                        'timestamp' => now()->toISOString(),
                    ]),
                ]);

                $this->sendFailureNotification($payment, $e->getMessage());
            }

            return [
                'success' => false,
                'message' => 'Payment confirmation failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Create a checkout session for single charge (no JavaScript needed)
     */
    public function createCheckoutSession(Payment $payment): array
    {
        try {
            // Ensure Stripe API key is set
            $this->ensureStripeConfigured();

            $user = $payment->user;
            $amount = (float) $payment->amount;
            $currency = strtolower($payment->currency);

            // Convert amount to cents for Stripe
            $amountInCents = $this->convertToCents($amount, $currency);

            // Create checkout session using Laravel Cashier
            $checkout = $user->checkout([
                [
                    'price_data' => [
                        'currency' => $currency,
                        'product_data' => [
                            'name' => 'Credit Purchase - ' . $payment->purchase->quantity . ' credits',
                            'description' => 'Purchase ' . $payment->purchase->quantity . ' credits for your account',
                        ],
                        'unit_amount' => (int) $amountInCents,
                    ],
                    'quantity' => 1,
                ],
            ], [
                'success_url' => route('dashboard.credits.stripe.success') . '?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => route('dashboard.credits.purchase') . '?cancelled=1',
                'metadata' => [
                    'payment_id' => $payment->id,
                    'user_id' => $user->id,
                    'purchase_id' => $payment->purchase_id,
                    'type' => 'credit_purchase',
                ],
                'payment_intent_data' => [
                    'metadata' => [
                        'payment_id' => $payment->id,
                        'user_id' => $user->id,
                        'purchase_id' => $payment->purchase_id,
                        'type' => 'credit_purchase',
                    ],
                ],
            ]);

            // Update payment with checkout session ID
            $payment->update([
                'transaction_id' => $checkout->id,
                'status' => PaymentStatus::Pending->value,
                'service_response' => json_encode([
                    'checkout_session' => $checkout->toArray(),
                    'checkout_url' => $checkout->url,
                ]),
            ]);

            return [
                'success' => true,
                'checkout_url' => $checkout->url,
                'checkout_session_id' => $checkout->id,
                'message' => 'Checkout session created successfully',
            ];
        } catch (Exception $e) {
            $payment->update([
                'status' => PaymentStatus::Failed->value,
                'service_response' => json_encode([
                    'error' => $e->getMessage(),
                    'timestamp' => now()->toISOString(),
                ]),
            ]);

            $this->sendFailureNotification($payment, $e->getMessage());

            return [
                'success' => false,
                'message' => 'Checkout session creation failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Create a simple charge using Laravel Cashier (alternative approach)
     */
    public function createSimpleCharge(Payment $payment, string $paymentMethodId): array
    {
        try {
            // Ensure Stripe API key is set
            $this->ensureStripeConfigured();

            $user = $payment->user;
            $amount = (float) $payment->amount;
            $currency = strtolower($payment->currency);
            $amountInCents = $this->convertToCents($amount, $currency);

            // Use Laravel Cashier's simple charge method
            $charge = $user->charge($amountInCents, $paymentMethodId, [
                'currency' => $currency,
                'metadata' => [
                    'payment_id' => $payment->id,
                    'user_id' => $user->id,
                    'purchase_id' => $payment->purchase_id,
                    'type' => 'credit_purchase',
                ],
            ]);

            // Update payment with charge result
            $payment->update([
                'transaction_id' => $charge->id,
                'status' => PaymentStatus::Paid->value,
                'service_response' => json_encode([
                    'charge' => $charge->toArray(),
                    'charged_at' => now()->toISOString(),
                ]),
            ]);

            // Send success notification
            $this->sendSuccessNotification($payment);

            return [
                'success' => true,
                'charge' => $charge,
                'payment' => $payment,
                'message' => 'Payment charged successfully',
            ];
        } catch (Exception $e) {
            $payment->update([
                'status' => PaymentStatus::Failed->value,
                'service_response' => json_encode([
                    'error' => $e->getMessage(),
                    'timestamp' => now()->toISOString(),
                ]),
            ]);

            $this->sendFailureNotification($payment, $e->getMessage());

            return [
                'success' => false,
                'message' => 'Payment failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Handle successful payment webhook
     */
    public function handleSuccessfulPayment(string $paymentIntentId): bool
    {
        try {
            // Find payment by transaction ID
            $payment = Payment::where('transaction_id', $paymentIntentId)->first();

            if (!$payment) {
                return false;
            }

            // Update payment status
            $payment->update([
                'status' => PaymentStatus::Paid->value,
                'service_response' => json_encode([
                    'webhook_processed' => true,
                    'processed_at' => now()->toISOString(),
                ]),
            ]);

            // Send success notification
            $this->sendSuccessNotification($payment);

            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Retrieve and validate a checkout session
     */
    public function retrieveCheckoutSession(string $sessionId): array
    {
        try {
            // Ensure Stripe API key is set
            $this->ensureStripeConfigured();

            // Retrieve the session from Stripe
            $session = Cashier::stripe()->checkout->sessions->retrieve($sessionId);

            if (!$session) {
                throw new Exception('Checkout session not found: ' . $sessionId);
            }

            // Check session status
            $status = $session->status;
            $paymentStatus = $session->payment_status;

            if ($paymentStatus === 'paid') {
                return [
                    'success' => true,
                    'session' => $session,
                    'status' => $status,
                    'payment_status' => $paymentStatus,
                    'message' => 'Checkout session payment completed successfully',
                ];
            } else {
                return [
                    'success' => false,
                    'session' => $session,
                    'status' => $status,
                    'payment_status' => $paymentStatus,
                    'message' => 'Checkout session payment not completed: ' . $paymentStatus,
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Error retrieving checkout session: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Handle failed payment webhook
     */
    public function handleFailedPayment(string $paymentIntentId, string $failureReason): bool
    {
        try {
            // Find payment by transaction ID
            $payment = Payment::where('transaction_id', $paymentIntentId)->first();

            if (!$payment) {
                return false;
            }

            // Update payment status
            $payment->update([
                'status' => PaymentStatus::Failed->value,
                'service_response' => json_encode([
                    'failure_reason' => $failureReason,
                    'webhook_processed' => true,
                    'processed_at' => now()->toISOString(),
                ]),
            ]);

            // Send failure notification
            $this->sendFailureNotification($payment, $failureReason);

            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Create a setup intent for saving payment methods
     */
    public function createSetupIntent(User $user): array
    {
        try {
            // Ensure Stripe API key is set
            $this->ensureStripeConfigured();

            $setupIntent = $user->createSetupIntent();

            return [
                'success' => true,
                'client_secret' => $setupIntent->client_secret,
                'setup_intent_id' => $setupIntent->id,
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Failed to create setup intent: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get user's saved payment methods
     */
    public function getPaymentMethods(User $user): array
    {
        try {
            $paymentMethods = $user->paymentMethods();

            return [
                'success' => true,
                'payment_methods' => $paymentMethods->map(function ($method) {
                    return [
                        'id' => $method->id,
                        'type' => $method->type,
                        'card' => $method->card ?? null,
                        'created' => $method->created,
                    ];
                })->toArray(),
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Failed to get payment methods: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Convert amount to cents for Stripe
     */
    private function convertToCents(float $amount, string $currency): int
    {
        // Zero decimal currencies (like JPY)
        $zeroDecimalCurrencies = ['jpy', 'krw', 'vnd', 'cld', 'bif', 'djf', 'gnf', 'kmf', 'mga', 'pyg', 'rwf', 'ugx', 'xaf', 'xof', 'xpf'];

        if (in_array(strtolower($currency), $zeroDecimalCurrencies)) {
            return (int) round($amount);
        }

        // Regular currencies (multiply by 100)
        return (int) round($amount * 100);
    }

    /**
     * Send success notification
     */
    private function sendSuccessNotification(Payment $payment): void
    {
        try {
            $user = $payment->user;
            $purchase = $payment->purchase;

            $notification = new CreditPurchaseSuccessful(
                $purchase,
                $payment,
                $purchase->quantity
            );

            $user->notify($notification);
        } catch (Exception $e) {
            // Silently fail
        }
    }

    /**
     * Send failure notification
     */
    private function sendFailureNotification(Payment $payment, string $errorMessage): void
    {
        try {
            $user = $payment->user;
            $purchase = $payment->purchase;

            $notification = new CreditPurchaseFailed(
                $purchase,
                $payment,
                $errorMessage
            );

            $user->notify($notification);
        } catch (Exception $e) {
            // Silently fail
        }
    }

    /**
     * Ensure Stripe is properly configured with API keys
     */
    private function ensureStripeConfigured(): void
    {
        try {
            // Get keys from settings helper functions
            $publishableKey = get_stripe_publishable_key();
            $secretKey = get_stripe_secret_key();

            if ($publishableKey && $secretKey) {
                // Set both Laravel config and global Stripe API key
                config(['cashier.key' => $publishableKey]);
                config(['cashier.secret' => $secretKey]);
                \Stripe\Stripe::setApiKey($secretKey);
            } else {
                throw new Exception('Stripe API keys not configured');
            }
        } catch (Exception $e) {
            throw $e;
        }
    }
}
