<?php

declare(strict_types=1);

namespace App\Services\Purchase;

use App\Models\User\User;
use App\Models\Membership\Purchase;
use App\Models\User\UserCreditHistory;
use App\Enums\PurchaseItemType;
use App\Repositories\Contracts\Purchase\PurchaseRepositoryInterface;
use Illuminate\Database\Eloquent\Model;

class PurchaseService
{
    public function __construct(
        private PurchaseRepositoryInterface $purchaseRepository
    ) {}

    /**
     * Create a credit purchase record
     */
    public function createCreditPurchase(
        User $user,
        int $quantity,
        float $basePrice,
        float $amount,
        float $discountAmount,
        string $currency
    ): Purchase {
        return $this->purchaseRepository->createCreditPurchase(
            $user,
            $quantity,
            $basePrice,
            $amount,
            $discountAmount,
            $currency
        );
    }

    /**
     * Update purchase purchaseable_id
     */
    public function updatePurchaseableId(Purchase $purchase, int $purchaseableId): bool
    {
        return $this->purchaseRepository->updatePurchaseableId($purchase, $purchaseableId);
    }

    // Merged methods from MemberPurchaseService
    /**
     * Get all purchases with optional filters
     */
    public function getAll(array $args = [])
    {
        if (!empty($args['memberPlan'])) {
            $args['with_plan'] = $args['memberPlan'] ?? null;
        }

        return $this->purchaseRepository->getAll($args);
    }

    /**
     * Find purchase by ID
     */
    public function findById(int $id): ?Model
    {
        return $this->purchaseRepository->findById($id);
    }

    /**
     * Find purchase by UUID
     */
    public function findByUuid(string $uuid): ?Model
    {
        return $this->purchaseRepository->findByUuid($uuid);
    }

    /**
     * Create a new purchase record
     */
    public function create(array $data): Model
    {
        return $this->purchaseRepository->create($data);
    }

    /**
     * Update an existing purchase record
     */
    public function update(Model $model, array $data): bool
    {
        return $this->purchaseRepository->update($model, $data);
    }

    /**
     * Delete a purchase record
     */
    public function delete(Model $model): bool
    {
        return $this->purchaseRepository->delete($model);
    }

    /**
     * Delete purchase by ID
     */
    public function deleteById(int $id): bool
    {
        return $this->purchaseRepository->deleteById($id);
    }

    /**
     * Delete purchase by UUID
     */
    public function deleteByUuid(string $uuid): bool
    {
        return $this->purchaseRepository->deleteByUuid($uuid);
    }
}
