<?php

declare(strict_types=1);

namespace App\Services\Review;

use App\Repositories\Contracts\Review\ReviewRepositoryInterface;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

class ReviewService
{
    public function __construct(
        protected ReviewRepositoryInterface $reviewRepository
    ) {}

    public function getAll(array $args = []): Builder
    {
        return $this->reviewRepository->getAll($args);
    }

    public function find(int $id): ?Model
    {
        return $this->reviewRepository->find($id);
    }

    public function findByUuid(string $uuid): ?Model
    {
        return $this->reviewRepository->findByUuid($uuid);
    }

    public function create(array $data): Model
    {
        return $this->reviewRepository->create($data);
    }

    public function update(int $id, array $data): bool
    {
        return $this->reviewRepository->update($id, $data);
    }

    public function updateByUuid(string $uuid, array $data): bool
    {
        return $this->reviewRepository->updateByUuid($uuid, $data);
    }

    public function delete(int $id): bool
    {
        return $this->reviewRepository->delete($id);
    }

    /**
     * @param string $type "App\Models\User\User" or "App\Models\Job\Job"
     * @param int $id
     * @param array $args
     * @return Builder
     */
    public function getReviewsByReviewable(string $type, int $id, array $args = []): Builder
    {
        return $this->reviewRepository->getReviewsByReviewable($type, $id, $args);
    }

    /**
     * Get the average rating for a reviewable item.
     */
    public function getAverageRating(string $type, int $id): float
    {
        $reviews = $this->getReviewsByReviewable($type, $id, ['status' => true])->get();

        if ($reviews->isEmpty()) {
            return 0;
        }

        return round($reviews->avg('review_rating'), 1);
    }

    /**
     * Get the rating distribution for a reviewable item.
     */
    public function getRatingDistribution(string $type, int $id): array
    {
        $reviews = $this->getReviewsByReviewable($type, $id, ['status' => true])->get();

        $distribution = [
            '1' => 0,
            '2' => 0,
            '3' => 0,
            '4' => 0,
            '5' => 0,
        ];

        foreach ($reviews as $review) {
            $rating = (string) $review->review_rating;
            if (isset($distribution[$rating])) {
                $distribution[$rating]++;
            }
        }

        return $distribution;
    }
}
