<?php

namespace App\Services\Settings;

use App\Models\Setting;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class SettingsManager
{
    protected const CACHE_KEY = 'app_settings';

    /** @return Collection<string,mixed> */
    public function all(): Collection
    {
        return Cache::rememberForever(self::CACHE_KEY, function () {
            $settings = Setting::all();
            $result = collect();

            foreach ($settings as $setting) {
                $fullKey = $this->buildFullKey($setting->group, $setting->key);
                $result->put($fullKey, $this->cast($setting->value));
            }

            return $result;
        });
    }

    public function get(string $key, $default = null)
    {
        return $this->all()->get($key, $default);
    }

    public function getByGroup(string $group, string $key, $default = null)
    {
        // Check directly in database for this specific group/key combination
        $setting = Setting::where('group', $group)->where('key', $key)->first();

        if ($setting) {
            return $this->cast($setting->value);
        }

        return $default;
    }

    /**
     * Set a setting value with optional metadata and file cleanup
     */
    public function set(string $group, string $key, $value, array $meta = []): void
    {
        // Get existing setting to check for file cleanup
        $existingSetting = Setting::where('group', $group)->where('key', $key)->first();

        // If this is a file field and there's an existing value, clean up the old file
        if ($existingSetting && isset($meta['type']) && $this->isFileType($meta['type'])) {
            $this->cleanupOldFile($existingSetting->value, $value);
        }

        Setting::updateOrCreate(
            ['group' => $group, 'key' => $key],
            array_merge(['value' => $value], $meta)
        );

        Cache::forget(self::CACHE_KEY);
    }

    /**
     * Delete a setting and cleanup associated files
     */
    public function delete(string $group, string $key): bool
    {
        $setting = Setting::where('group', $group)->where('key', $key)->first();

        if ($setting) {
            // Check if this is a file setting and cleanup the file
            if ($this->isFileType($setting->type)) {
                $this->cleanupFile($setting->value);
            }

            $result = $setting->delete();
            Cache::forget(self::CACHE_KEY);

            return $result;
        }

        return false;
    }

    /**
     * Check if a type represents a file upload
     */
    protected function isFileType(?string $type): bool
    {
        return in_array($type, ['file', 'image', 'document']);
    }

    /**
     * Clean up old file when value is being updated
     */
    protected function cleanupOldFile(?string $oldValue, $newValue): void
    {
        // Only cleanup if the value is actually changing and old value exists
        if ($oldValue && $oldValue !== $newValue && !empty($oldValue)) {
            $this->cleanupFile($oldValue);
        }
    }

    /**
     * Delete a file from storage
     */
    protected function cleanupFile(?string $filePath): void
    {
        if (!$filePath || empty($filePath)) {
            return;
        }

        try {
            // Handle both full paths and relative paths
            $relativePath = $filePath;

            // If it's a full URL, extract the relative path
            if (str_contains($filePath, '/storage/')) {
                $relativePath = str_replace('/storage/', '', parse_url($filePath, PHP_URL_PATH));
            }

            // Remove leading slash if present
            $relativePath = ltrim($relativePath, '/');

            // Use the public disk (same as Filament FileUpload default)
            if (Storage::disk('public')->exists($relativePath)) {
                Storage::disk('public')->delete($relativePath);
                Log::info("Cleaned up settings file: {$relativePath}");
            }
        } catch (\Exception $e) {
            Log::error("Failed to cleanup settings file: {$filePath}", [
                'error' => $e->getMessage()
            ]);
        }
    }

    public function setByFullKey(string $fullKey, $value, array $meta = []): void
    {
        [$group, $key] = $this->parseFullKey($fullKey);
        $this->set($group, $key, $value, $meta);
    }

    public function cast($value)
    {
        // Handle JSON strings
        if (is_string($value) && (str_starts_with($value, '{') || str_starts_with($value, '['))) {
            $decoded = json_decode($value, true);
            return json_last_error() === JSON_ERROR_NONE ? $decoded : $value;
        }

        // Handle boolean strings
        if (is_string($value)) {
            if ($value === 'true') return true;
            if ($value === 'false') return false;
        }

        return $value;
    }

    protected function buildFullKey(string $group, string $key): string
    {
        return "{$group}.{$key}";
    }

    protected function parseFullKey(string $fullKey): array
    {
        $parts = explode('.', $fullKey, 2);
        if (count($parts) !== 2) {
            throw new \InvalidArgumentException("Invalid setting key format: {$fullKey}");
        }
        return $parts;
    }
}
