<?php

declare(strict_types=1);

namespace App\Services\User;

use App\Models\User\BusinessHour;
use App\Models\User\User;
use App\Enums\DayOfWeek;
use Illuminate\Support\Collection;
use Carbon\Carbon;

class BusinessHourService
{
    public function getBusinessHours(User $user): Collection
    {
        return $user->businessHours()->orderByRaw("
            CASE day_of_week
                WHEN 'monday' THEN 1
                WHEN 'tuesday' THEN 2
                WHEN 'wednesday' THEN 3
                WHEN 'thursday' THEN 4
                WHEN 'friday' THEN 5
                WHEN 'saturday' THEN 6
                WHEN 'sunday' THEN 7
            END
        ")->get();
    }

    public function createOrUpdateBusinessHours(User $user, array $businessHoursData): Collection
    {
        // Delete existing business hours
        $user->businessHours()->delete();

        $businessHours = collect();

        foreach ($businessHoursData as $dayData) {
            if (isset($dayData['day_of_week'])) {
                $businessHour = $user->businessHours()->create([
                    'day_of_week' => $dayData['day_of_week'],
                    'is_open' => $dayData['is_open'] ?? true,
                    'hours_label' => $dayData['hours_label'] ?? null,
                    'open_time' => $dayData['open_time'] ?? null,
                    'close_time' => $dayData['close_time'] ?? null,
                ]);

                $businessHours->push($businessHour);
            }
        }

        return $businessHours;
    }

    public function isCurrentlyOpen(User $user): bool
    {
        $now = Carbon::now();
        $currentDay = strtolower($now->format('l'));
        $currentTime = $now->format('H:i');

        $todayHours = $user->businessHours()
            ->where('day_of_week', $currentDay)
            ->where('is_open', true)
            ->first();

        if (!$todayHours) {
            return false;
        }

        return $todayHours->open_time <= $currentTime && $todayHours->close_time >= $currentTime;
    }

    public function getTodaysHours(User $user): ?BusinessHour
    {
        $currentDay = strtolower(Carbon::now()->format('l'));
        
        return $user->businessHours()
            ->where('day_of_week', $currentDay)
            ->first();
    }

    public function getFormattedBusinessHours(User $user): array
    {
        $businessHours = $this->getBusinessHours($user);
        $formatted = [];

        foreach ($businessHours as $hour) {
            $formatted[] = [
                'day' => $hour->day_of_week->getLabel(),
                'is_open' => $hour->is_open,
                'hours' => $hour->formatted_hours,
                'hours_label' => $hour->hours_label,
                'open_time' => $hour->open_time,
                'close_time' => $hour->close_time,
            ];
        }

        return $formatted;
    }

    public function createDefaultBusinessHours(User $user): Collection
    {
        $defaultHours = [
            ['day_of_week' => DayOfWeek::MONDAY->value, 'is_open' => true, 'hours_label' => 'Regular Hours', 'open_time' => '09:00', 'close_time' => '17:00'],
            ['day_of_week' => DayOfWeek::TUESDAY->value, 'is_open' => true, 'hours_label' => 'Regular Hours', 'open_time' => '09:00', 'close_time' => '17:00'],
            ['day_of_week' => DayOfWeek::WEDNESDAY->value, 'is_open' => true, 'hours_label' => 'Regular Hours', 'open_time' => '09:00', 'close_time' => '17:00'],
            ['day_of_week' => DayOfWeek::THURSDAY->value, 'is_open' => true, 'hours_label' => 'Regular Hours', 'open_time' => '09:00', 'close_time' => '17:00'],
            ['day_of_week' => DayOfWeek::FRIDAY->value, 'is_open' => true, 'hours_label' => 'Regular Hours', 'open_time' => '09:00', 'close_time' => '17:00'],
            ['day_of_week' => DayOfWeek::SATURDAY->value, 'is_open' => false, 'hours_label' => null, 'open_time' => null, 'close_time' => null],
            ['day_of_week' => DayOfWeek::SUNDAY->value, 'is_open' => false, 'hours_label' => null, 'open_time' => null, 'close_time' => null],
        ];

        return $this->createOrUpdateBusinessHours($user, $defaultHours);
    }

    public function convertLegacyBusinessHours(User $user, array $legacyHours): Collection
    {
        // Convert old business_hours array format to new table structure
        $businessHoursData = [];

        if (!empty($legacyHours)) {
            // Initialize all days as closed first
            foreach (DayOfWeek::getAllDays() as $day) {
                $businessHoursData[] = [
                    'day_of_week' => $day->value,
                    'is_open' => false,
                    'hours_label' => null,
                    'open_time' => null,
                    'close_time' => null,
                ];
            }

            // Process legacy hours and map to specific days
            foreach ($legacyHours as $hour) {
                if (isset($hour['days'], $hour['start_time'], $hour['end_time'])) {
                    $isOpen = $hour['is_open'] ?? true;
                    $days = strtolower($hour['days']);
                    
                    $hoursLabel = $hour['label'] ?? 'Regular Hours';
                    
                    // Map common day patterns to individual days
                    if (str_contains($days, 'monday to friday') || str_contains($days, 'weekdays')) {
                        $weekdays = DayOfWeek::getWeekdays();
                        foreach ($weekdays as $day) {
                            $this->updateDayData($businessHoursData, $day->value, $isOpen, $hoursLabel, $hour['start_time'], $hour['end_time']);
                        }
                    } elseif (str_contains($days, 'weekend')) {
                        $weekends = DayOfWeek::getWeekends();
                        foreach ($weekends as $day) {
                            $this->updateDayData($businessHoursData, $day->value, $isOpen, $hoursLabel, $hour['start_time'], $hour['end_time']);
                        }
                    } else {
                        // Try to match individual days
                        foreach (DayOfWeek::getAllDays() as $day) {
                            if (str_contains($days, strtolower($day->getLabel()))) {
                                $this->updateDayData($businessHoursData, $day->value, $isOpen, $hoursLabel, $hour['start_time'], $hour['end_time']);
                            }
                        }
                    }
                }
            }
        } else {
            // No legacy data, create default hours
            return $this->createDefaultBusinessHours($user);
        }

        return $this->createOrUpdateBusinessHours($user, $businessHoursData);
    }

    private function updateDayData(array &$businessHoursData, string $dayValue, bool $isOpen, ?string $hoursLabel, ?string $openTime, ?string $closeTime): void
    {
        foreach ($businessHoursData as &$dayData) {
            if ($dayData['day_of_week'] === $dayValue) {
                $dayData['is_open'] = $isOpen;
                $dayData['hours_label'] = $isOpen ? $hoursLabel : null;
                $dayData['open_time'] = $isOpen ? $openTime : null;
                $dayData['close_time'] = $isOpen ? $closeTime : null;
                break;
            }
        }
    }
}