<?php

declare(strict_types=1);

namespace App\Services\User;

use App\Repositories\Contracts\User\UserFavoriteRepositoryInterface;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;

class UserFavoriteService
{
    public function __construct(
        protected UserFavoriteRepositoryInterface $userFavoriteRepository
    ) {}

    public function getUserFavorites(int $userId, string $modelType = null): array
    {
        $favorites = $this->userFavoriteRepository->getUserFavorites($userId, $modelType)->get();
        
        return $favorites->map(function ($favorite) {
            return [
                'id' => $favorite->id,
                'model_type' => $favorite->model_type,
                'model_id' => $favorite->model_id,
                'model' => $favorite->model,
                'created_at' => $favorite->created_at,
            ];
        })->toArray();
    }

    public function getUserFavoritesPaginated(int $userId, string $modelType = null, int $perPage = 15): LengthAwarePaginator
    {
        return $this->userFavoriteRepository->getUserFavoritesPaginated($userId, $modelType, $perPage);
    }

    public function addFavorite(int $userId, string $modelType, int $modelId): Model
    {
        return $this->userFavoriteRepository->create([
            'user_id' => $userId,
            'model_type' => $modelType,
            'model_id' => $modelId,
        ]);
    }

    public function removeFavorite(int $userId, string $modelType, int $modelId): bool
    {
        return $this->userFavoriteRepository->removeFavorite($userId, $modelType, $modelId);
    }

    public function isFavorite(int $userId, string $modelType, int $modelId): bool
    {
        return $this->userFavoriteRepository->isFavorite($userId, $modelType, $modelId);
    }

    public function getFavoritesByType(int $userId): array
    {
        $modelTypes = [
            'App\\Models\\Blog\\Post' => 'post',
            'App\\Models\\Classified\\Classified' => 'classified',
            'App\\Models\\Job\\Job' => 'job',
            'App\\Models\\Equipment\\Equipment' => 'equipment',
            'App\\Models\\User\\User' => 'users',
            'App\\Models\\Project\\Project' => 'project',
            'App\\Models\\Product\\Product' => 'product',
        ];

        $favoritesByType = [];
        
        foreach ($modelTypes as $modelClass => $type) {
            $count = $this->userFavoriteRepository->getUserFavorites($userId, $modelClass)->count();
            $favoritesByType[$type] = [
                'count' => $count,
                'model_type' => $modelClass,
                'label' => ucfirst($type),
            ];
        }

        return $favoritesByType;
    }
}