<?php

declare(strict_types=1);

namespace App\Services\User;

use App\Models\User\UserLocation;
use App\Repositories\Contracts\User\UserLocationRepositoryInterface;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class UserLocationService
{
    public function __construct(
        private UserLocationRepositoryInterface $userLocationRepository
    ) {}

    public function getAllLocations(): Collection
    {
        return $this->userLocationRepository->all();
    }

    public function getLocationByUuid(string $uuid): ?UserLocation
    {
        return $this->userLocationRepository->findByUuid($uuid);
    }

    public function createLocation(array $data): UserLocation
    {
        try {
            DB::beginTransaction();

            if (isset($data['primary_location']) && $data['primary_location']) {
                $this->removePrimaryLocationFlag($data['user_id']);
            }

            $location = $this->userLocationRepository->create($data);

            DB::commit();
            return $location;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function updateLocation(UserLocation $location, array $data): UserLocation
    {
        try {
            DB::beginTransaction();

            if (isset($data['primary_location']) && $data['primary_location']) {
                $this->removePrimaryLocationFlag($location->user_id, $location->id);
            }

            $this->userLocationRepository->update($location, $data);
            $location->refresh();

            DB::commit();
            return $location;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function deleteLocation(UserLocation $location): bool
    {
        try {
            DB::beginTransaction();

            if ($location->primary_location) {
                // Get the latest active location excluding the one being deleted
                $latestLocation = $this->userLocationRepository
                    ->getActiveLocations($location->user_id)
                    ->where('id', '!=', $location->id)
                    ->first();

                if ($latestLocation) {
                    $this->userLocationRepository->update($latestLocation, ['primary_location' => true]);
                } else {
                    throw new ModelNotFoundException('Cannot delete the only location');
                }
            }

            $result = $this->userLocationRepository->delete($location);

            DB::commit();
            return $result;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function getPrimaryLocation(int|string $userId): ?UserLocation
    {
        return $this->userLocationRepository->getPrimaryLocation($userId);
    }

    public function getActiveLocations(int|string $userId): Collection
    {
        return $this->userLocationRepository->getActiveLocations($userId);
    }

    private function removePrimaryLocationFlag(int|string $userId, ?int $excludeId = null): void
    {
        $primaryLocation = $this->userLocationRepository->getPrimaryLocation($userId);
        
        if ($primaryLocation && (!$excludeId || $primaryLocation->id !== $excludeId)) {
            $this->userLocationRepository->update($primaryLocation, ['primary_location' => false]);
        }
    }

    public function removeOtherPrimaryLocations(int|string $userId, ?int $excludeLocationId = null): void
    {
        $query = UserLocation::where('user_id', $userId)
            ->where('primary_location', true);
            
        if ($excludeLocationId) {
            $query->where('id', '!=', $excludeLocationId);
        }
        
        $query->update(['primary_location' => false]);
    }
} 