<?php

declare(strict_types=1);

namespace App\Services\User;

use App\Repositories\Contracts\User\UserRepositoryInterface;
use App\Repositories\Contracts\User\MemberCategoryRepositoryInterface;
use App\Repositories\Contracts\Location\CityRepositoryInterface;
use App\Repositories\Contracts\Location\StateRepositoryInterface;
use App\Repositories\Contracts\Location\CountryRepositoryInterface;
use App\Enums\MemberRatingType;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Hash;
use Illuminate\Pagination\LengthAwarePaginator;
use App\Enums\MembershipType;
use App\Services\User\MemberPlanService;
use App\Repositories\Contracts\User\MemberPlanRepositoryInterface;
use App\Models\User\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class UserService
{
    public function __construct(
        protected UserRepositoryInterface $userRepository,
        protected MemberCategoryRepositoryInterface $categoryRepository ,
        protected CityRepositoryInterface $cityRepository ,
        protected StateRepositoryInterface $stateRepository ,
        protected CountryRepositoryInterface $countryRepository ,
        protected MemberPlanService $memberPlanService ,
        protected MemberPlanRepositoryInterface $memberPlanRepository
    ) {}

    public function getAll(array $args = []): Builder
    {
        // Convert category slugs to IDs
        if (!empty($args['categories'])) {
            if (is_string($args['categories'])) {
                $args['categories'] = [$args['categories']];
            }

            $categoryIds = [];
            foreach ($args['categories'] as $categorySlug) {
                if (!is_numeric($categorySlug)) {
                    $category = $this->categoryRepository->getCategoryBySlug($categorySlug);
                    if ($category) {
                        $categoryIds[] = $category->id;
                    }
                } else {
                    $categoryIds[] = (int) $categorySlug;
                }
            }
            $args['categories'] = $categoryIds;
        }

        // Convert state slug to ID
        if (!empty($args['state']) && !is_numeric($args['state'])) {
            $state = $this->stateRepository->findBySlug($args['state']);
            if ($state) {
                $args['state'] = $state->id;
            }
        }

        // Convert country slug to ID
        if (!empty($args['country']) && !is_numeric($args['country'])) {
            $country = $this->countryRepository->findBySlug($args['country']);
            if ($country) {
                $args['country'] = $country->id;
            }
        }

        // Convert city slug to ID
        if (!empty($args['city']) && !is_numeric($args['city'])) {
            
            if (is_numeric($args['country'])) {
                
                $city = $this->cityRepository->findBySlug($args['city']);
                if ($city) {
                    $args['city'] = $city->id;
                } else {
                    $args['city'] = null;
                }
            } else {
                $args['city'] = null;
            }
        }
        return $this->userRepository->getAll($args);
    }

    public function find(int $id): ?Model
    {
        return $this->userRepository->find($id);
    }

    public function findBySlug(string $slug): ?Model
    {
        return $this->userRepository->findBySlug($slug);
    }

    public function findByEmail(string $email): ?Model
    {
        return $this->userRepository->findByEmail($email);
    }

    public function create(array $data): Model
    {
        if (isset($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        }

        return $this->userRepository->create($data);
    }

    public function update(int $id, array $data): bool
    {
        if (isset($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        }

        return $this->userRepository->update($id, $data);
    }

    public function delete(int $id): bool
    {
        return $this->userRepository->delete($id);
    }

    public function getMemberList(array $args = [])
    {
        // DB::enableQueryLog()`;
        $args = $this->filterSearchParams($args);
        $userCollection = $this->getAll($args)->active()->paginate($args['limit'] ?? 15);
        // Log::info(DB::getQueryLog());
        // DB::disableQueryLog();
        return $this->sortUsersByMembershipPriority($userCollection);
    }

    public function filterSearchParams(array $args = []): array
    {
        // Check if we have services, subcategories, or categories and prioritize accordingly
        if (!empty($args['service'])) {
            // If services exist, treat them as categories
            $args['categories'] = $args['service'];
        } elseif (!empty($args['subcategory'])) {
            // If no services but subcategories exist, treat subcategories as categories
            $args['categories'] = $args['subcategory'];
        } elseif (!empty($args['category'])) {
            // If neither services nor subcategories exist, use categories as categories
            $args['categories'] = $args['category'];
        }

        if (!empty($args['rating']) && $args['rating'] == 'all') {
            $args['rating'] = null;
        }

        if (!empty($args['country']) && $args['country'] == 'all') {
            $args['country'] = null;
        }

        return $args;
    }

    /**
     * Sort users by membership plan priority on paginated collection
     * Elite = 1 (highest priority)
     * Pro = 2 (medium priority) 
     * Free = 3 (lowest priority)
     *
     * @param LengthAwarePaginator $users The paginated users collection
     * @return LengthAwarePaginator The paginated collection with membership sorting applied
     */
    public function sortUsersByMembershipPriority(LengthAwarePaginator $users): LengthAwarePaginator
    {
        // Sort the collection
        $sortedUsers = $users->getCollection()->sortByDesc(function ($user) {
            $membershipType = $user->membershipType->value;
            if (empty($membershipType)) {
                $membershipType = MembershipType::Free->value;
            }

            // Sort by membership type priority
            if (strtolower($membershipType) === MembershipType::Featured->value) {
                return 3; // Give it the highest priority (elite members come first)
            }

            // Then sort by pro membership
            if (strtolower($membershipType) === MembershipType::Pro->value) {
                return 2; // Pro members come second
            }

            // Free members come third
            if (strtolower($membershipType) === MembershipType::Free->value) {
                return 1; // Free members come third
            }

            // Users without membership plan come last
            return 0;
        });

        // Re-create the paginator with the sorted collection
        return new LengthAwarePaginator(
            $sortedUsers,
            $users->total(),
            $users->perPage(),
            $users->currentPage(),
            ['path' => \Illuminate\Pagination\Paginator::resolveCurrentPath()]
        );
    }

    /**
     * Get members list sorted by membership plan priority
     *
     * @param array $args Search and filter parameters
     * @param int $perPage Number of items per page
     * @return LengthAwarePaginator Paginated collection with membership priority sorting
     */
    public function getMemberListByMembershipPriority(array $args = [], int $perPage = 15): LengthAwarePaginator
    {
        $args = $this->filterSearchParams($args);
        $users = $this->getAll($args)->active()->paginate($perPage);

        return $this->sortUsersByMembershipPriority($users);
    }

    /**
     * Generate comprehensive review statistics for a user
     *
     * @param int $userId The ID of the user to get statistics for
     * @return array Review statistics including total count, average ratings, and distribution
     */
    public function getUserReviewsStats(int $userId): array
    {
        $user = $this->userRepository->find($userId);

        if (!$user) {
            return [
                'total_reviews' => 0,
                'average_rating' => 0,
                'rating_distribution' => [],
                'rating_types' => [],
            ];
        }

        // Load reviews with their ratings
        $reviews = $user->reviews()->with('ratings')->where('status', true)->get();

        // If no reviews, return empty stats
        if ($reviews->isEmpty()) {
            return [
                'total_reviews' => 0,
                'average_rating' => 0,
                'rating_distribution' => [
                    5 => ['count' => 0, 'percentage' => 0],
                    4 => ['count' => 0, 'percentage' => 0],
                    3 => ['count' => 0, 'percentage' => 0],
                    2 => ['count' => 0, 'percentage' => 0],
                    1 => ['count' => 0, 'percentage' => 0],
                ],
                'rating_types' => $this->getEmptyRatingTypes(),
            ];
        }

        // Calculate total reviews count
        $totalReviews = $reviews->count();

        // Calculate average overall rating
        $averageRating = round($reviews->avg('review_rating'), 1);

        // Calculate rating distribution (5★, 4★, 3★, etc.)
        $ratingDistribution = [
            5 => ['count' => 0, 'percentage' => 0],
            4 => ['count' => 0, 'percentage' => 0],
            3 => ['count' => 0, 'percentage' => 0],
            2 => ['count' => 0, 'percentage' => 0],
            1 => ['count' => 0, 'percentage' => 0],
        ];

        foreach ($reviews as $review) {
            $rating = $review->review_rating;
            if (isset($ratingDistribution[$rating])) {
                $ratingDistribution[$rating]['count']++;
            }
        }

        // Calculate percentages
        foreach ($ratingDistribution as $rating => &$data) {
            $data['percentage'] = $totalReviews > 0 ? round(($data['count'] / $totalReviews) * 100, 1) : 0;
        }

        // Calculate rating type statistics
        $ratingTypeStats = $this->calculateRatingTypeStats($reviews);

        return [
            'total_reviews' => $totalReviews,
            'average_rating' => $averageRating,
            'rating_distribution' => $ratingDistribution,
            'rating_types' => $ratingTypeStats,
        ];
    }

    /**
     * Calculate statistics for each rating type
     *
     * @param \Illuminate\Support\Collection $reviews Collection of reviews
     * @return array Statistics for each rating type
     */
    private function calculateRatingTypeStats($reviews): array
    {
        $ratingTypes = $this->getEmptyRatingTypes();

        // Process all ratings across all reviews
        foreach ($reviews as $review) {
            foreach ($review->ratings as $rating) {
                $type = $rating->rating_type->value;

                // Increment count and sum for the specific rating type
                $ratingTypes[$type]['count']++;
                $ratingTypes[$type]['sum'] += $rating->rating;
            }
        }

        // Calculate averages and percentages
        foreach ($ratingTypes as $type => &$stats) {
            if ($stats['count'] > 0) {
                $stats['average'] = round($stats['sum'] / $stats['count'], 1);
                $stats['percentage'] = round(($stats['average'] / 5) * 100);
            }
        }

        return $ratingTypes;
    }

    /**
     * Get empty rating type statistics structure
     *
     * @return array Empty rating type statistics
     */
    private function getEmptyRatingTypes(): array
    {
        $ratingTypes = [];

        foreach (MemberRatingType::cases() as $type) {
            $ratingTypes[$type->value] = [
                'label' => $type->getLabel(),
                'count' => 0,
                'sum' => 0,
                'average' => 0,
                'percentage' => 0,
            ];
        }

        return $ratingTypes;
    }

    public function getSponsoredListings($limit = 10)
    {
        $args = [
            'role' => ESP_ROLE_MEMBER,
            'limit' => $limit,
            'membershipType' => MembershipType::Featured->value,
        ];
        return $this->userRepository->getAll($args)->limit($limit)->get();
    }

    public function getFeaturedListings($limit = 10)
    {
        $args = [
            'role' => ESP_ROLE_MEMBER,
            'limit' => $limit,
            'membershipType' => MembershipType::Featured->value,
        ];
        return $this->userRepository->getAll($args)->inRandomOrder()->limit($limit)->get();
    }

    /**
     * Get count of users grouped by rating ranges
     * 1 star: 0-1.99
     * 2 stars: 2-2.99
     * 3 stars: 3-3.99
     * 4 stars: 4-4.99
     * 5 stars: 5
     *
     * @return array
     */
    public function getUserCountsByRating(): array
    {
        $ratingCounts = [
            1 => 0,
            2 => 0,
            3 => 0,
            4 => 0,
            5 => 0
        ];

        // Get users with member role and active status
        $users = User::query()
            ->whereHas('roles', function ($q) {
                $q->where('name', ESP_ROLE_MEMBER);
            })
            ->whereHas('profile', function ($q) {
                $q->where('profile_rating', '>', 0);
            })
            ->active()
            ->join('user_profiles', 'users.id', '=', 'user_profiles.user_id')
            ->select('user_profiles.profile_rating')
            ->get();

        foreach ($users as $user) {
            $rating = $user->profile_rating;

            if ($rating >= 0 && $rating < 2) {
                $ratingCounts[1]++;
            } elseif ($rating >= 2 && $rating < 3) {
                $ratingCounts[2]++;
            } elseif ($rating >= 3 && $rating < 4) {
                $ratingCounts[3]++;
            } elseif ($rating >= 4 && $rating < 5) {
                $ratingCounts[4]++;
            } elseif ($rating == 5) {
                $ratingCounts[5]++;
            }
        }

        return $ratingCounts;
    }
}
