<?php

namespace App\Traits;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Http\Resources\Json\ResourceCollection;

trait ApiResponseTrait
{
    /**
     * Send a success response
     */
    protected function successResponse(
        $data = null,
        string $message = null,
        int $statusCode = 200
    ): JsonResponse {
        $response = [
            'success' => true,
            'message' => $message,
            'data' => $data,
            'meta' => null,
        ];

        // Extract meta data from ResourceCollection
        if ($data instanceof ResourceCollection) {
            $response['meta'] = $this->extractMetaData($data);
        }

        return response()->json(array_filter($response, fn($value) => $value !== null), $statusCode);
    }

    /**
     * Send a created response (201)
     */
    protected function createdResponse(
        $data = null,
        string $message = 'Resource created successfully'
    ): JsonResponse {
        return $this->successResponse($data, $message, 201);
    }

    /**
     * Send a no content response (204)
     */
    protected function noContentResponse(
        string $message = 'Resource deleted successfully'
    ): JsonResponse {
        return response()->json([
            'success' => true,
            'message' => $message,
            'data' => null,
            'meta' => null,
        ], 204);
    }

    /**
     * Send an error response
     */
    protected function errorResponse(
        string $message,
        int $statusCode = 400,
        $errors = null,
        string $errorCode = null
    ): JsonResponse {
        $response = [
            'success' => false,
            'message' => $message,
            'errors' => $errors,
            'error_code' => $errorCode ?? $this->getErrorCodeFromStatus($statusCode),
            'meta' => null,
        ];

        return response()->json(array_filter($response, fn($value) => $value !== null), $statusCode);
    }

    /**
     * Send a validation error response (422)
     */
    protected function validationErrorResponse(
        $errors,
        string $message = 'Validation failed'
    ): JsonResponse {
        return $this->errorResponse($message, 422, $errors, 'VALIDATION_ERROR');
    }

    /**
     * Send a not found response (404)
     */
    protected function notFoundResponse(
        string $message = 'Resource not found'
    ): JsonResponse {
        return $this->errorResponse($message, 404, null, 'RESOURCE_NOT_FOUND');
    }

    /**
     * Send an unauthorized response (401)
     */
    protected function unauthorizedResponse(
        string $message = 'Unauthorized'
    ): JsonResponse {
        return $this->errorResponse($message, 401, null, 'UNAUTHORIZED');
    }

    /**
     * Send a forbidden response (403)
     */
    protected function forbiddenResponse(
        string $message = 'Forbidden'
    ): JsonResponse {
        return $this->errorResponse($message, 403, null, 'FORBIDDEN');
    }

    /**
     * Send a server error response (500)
     */
    protected function serverErrorResponse(
        string $message = 'Internal server error'
    ): JsonResponse {
        return $this->errorResponse($message, 500, null, 'SERVER_ERROR');
    }

    /**
     * Send a bad request response (400)
     */
    protected function badRequestResponse(
        string $message = 'Bad request',
        $errors = null
    ): JsonResponse {
        return $this->errorResponse($message, 400, $errors, 'BAD_REQUEST');
    }

    /**
     * Extract meta data from paginated resource
     */
    private function extractMetaData($data): ?array
    {
        if (method_exists($data, 'resource') && $data->resource instanceof \Illuminate\Pagination\LengthAwarePaginator) {
            $paginator = $data->resource;
            return [
                'current_page' => $paginator->currentPage(),
                'per_page' => $paginator->perPage(),
                'total' => $paginator->total(),
                'last_page' => $paginator->lastPage(),
                'from' => $paginator->firstItem(),
                'to' => $paginator->lastItem(),
            ];
        }

        return null;
    }

    /**
     * Get error code from HTTP status code
     */
    private function getErrorCodeFromStatus(int $statusCode): string
    {
        return match ($statusCode) {
            400 => 'BAD_REQUEST',
            401 => 'UNAUTHORIZED',
            403 => 'FORBIDDEN',
            404 => 'RESOURCE_NOT_FOUND',
            422 => 'VALIDATION_ERROR',
            500 => 'SERVER_ERROR',
            default => 'ERROR',
        };
    }
}
