<?php

namespace App\View\Components\Frontend\Job\Partials;

use Closure;
use Illuminate\Contracts\View\View;
use Illuminate\View\Component;
use App\Models\Job\Job;
use App\Services\Job\JobService;
use App\Services\Location\CityService;
use App\Services\Location\CountryService;
use App\Services\Location\StateService;
use App\Services\Job\JobCareerLevelService;
use App\Services\Job\JobTypeService;
use App\Services\Job\JobRoleService;
use App\Enums\JobShift;
use App\Enums\JobWorkMode;
use Illuminate\Support\Collection;

class SearchSidebar extends Component
{

    protected $countryService;
    protected $stateService;
    protected $cityService;
    protected $jobCareerLevelService;
    protected $jobTypeService;
    protected $jobRoleService;

    public Collection $countryList;
    public Collection $stateList;
    public Collection $cityList;
    public Collection $jobCareerLevelList;
    public Collection $jobTypeList;
    public Collection $jobRoleList;
    public $jobShiftList;
    public $workModeList;

    public $search;
    public $country = '';
    public $state = '';
    public $city = '';
    public $career_level = [];
    public $job_type = [];
    public $shift = [];
    public $min_salary;
    public $max_salary;
    public $date_posted;
    public $job_role = [];
    public $openJob = false;
    public $work_mode = [];
    public $open_jobs_only = true;

    function __construct(CountryService $countryService, StateService $stateService, CityService $cityService, JobCareerLevelService $jobCareerLevelService, JobTypeService $jobTypeService, JobRoleService $jobRoleService)
    {
        $this->countryService = $countryService;
        $this->stateService   = $stateService;
        $this->cityService    = $cityService;
        $this->jobCareerLevelService = $jobCareerLevelService;
        $this->jobTypeService = $jobTypeService;
        $this->jobRoleService = $jobRoleService;

        $this->getCountryList();

        // Load states if country is selected
        $this->getStatesList();

        // Load cities if states are selected
        $this->getCitiesList();

        $this->getJobCareerLevelList();

        $this->getJobTypeList();

        $this->getJobRoleList();

        $this->jobShiftList = JobShift::cases();

        $this->workModeList = JobWorkMode::cases();

        // Initialize date_posted if empty
        if (empty($this->date_posted)) {
            $this->date_posted = ''; // Start with no dates selected
        }
    }

    public function getCountryList()
    {
        $this->countryList = $this->countryService->getAll()->get();
    }

    /**
     * Get all states for selected country
     *
     * @param string|null $countrySlug Country slug to get states for
     * @return Collection Collection of states
     */
    private function fetchStatesForCountry($countrySlug): Collection
    {
        $allStates = collect();

        // If country is provided
        if (! empty($countrySlug) && is_string($countrySlug)) {
            $country = $this->countryService->getCountryBySlug($countrySlug);

            if ($country) {
                // Get states for this country
                $states = $this->stateService->getAll(['country_id' => $country->id])->get();

                // Ensure states is a collection
                $allStates = $states instanceof Collection ? $states : collect($states);
            }
        }

        // Return states
        return $allStates->unique('id')->values();
    }

    /**
     * Get all cities for selected state
     *
     * @param string|null $stateSlug State slug to get cities for
     * @return Collection Collection of cities
     */
    private function fetchCitiesForState($stateSlug): Collection
    {
        $allCities = collect();

        // If state is provided
        if (! empty($stateSlug) && is_string($stateSlug)) {
            $state = $this->stateService->getStateBySlug($stateSlug);

            if ($state) {
                // Get cities for this state
                $cities = $this->cityService->getAll(['state_id' => $state->id])->get();

                // Ensure cities is a collection
                $allCities = $cities instanceof Collection ? $cities : collect($cities);
            }
        }

        // Return cities
        return $allCities->unique('id')->values();
    }

    public function getStatesList()
    {
        $stateList = collect();
        // If no country is selected, clear states and cities
        if (!empty($this->country)) {
            // Get all states for selected country
            $stateList = $this->fetchStatesForCountry($this->country);
        }
        $this->stateList = $stateList;
    }

    public function getCitiesList()
    {
        $cityList = collect();

        if (!empty($this->state)) {
            // Get all cities for selected state
            $cityList = $this->fetchCitiesForState($this->state);
        }
        $this->cityList = $cityList;
    }

    public function getJobCareerLevelList()
    {
        $this->jobCareerLevelList = $this->jobCareerLevelService->getAll()->get();
    }

    public function getJobTypeList()
    {
        $this->jobTypeList = $this->jobTypeService->getAll()->get();
    }

    public function getJobRoleList()
    {
        $this->jobRoleList = $this->jobRoleService->getAll()->onlyParent()->get();
    }

    public function render(): View|Closure|string
    {
        return view('components.frontend.job.partials.search-sidebar');
    }
}
