<?php

use Illuminate\Foundation\Application;
use Illuminate\Foundation\Configuration\Exceptions;
use Illuminate\Foundation\Configuration\Middleware;
use Illuminate\Support\Facades\Route;

return Application::configure(basePath: dirname(__DIR__))
    ->withRouting(
        web: null,
        api: __DIR__ . '/../routes/api.php',
        commands: __DIR__ . '/../routes/console.php',
        health: '/up',
        then: function () {
            // Load dashboard routes first
            Route::middleware('web')
                ->group(base_path('routes/dashboard.php'));

            // Load web routes after
            Route::middleware('web')
                ->group(base_path('routes/web.php'));
        }
    )
    ->withMiddleware(function (Middleware $middleware) {
        $middleware->validateCsrfTokens(except: [
            'stripe/*',
            'webhook/cashier',
        ]);

        // API middleware configuration
        $middleware->api(prepend: [
            \Illuminate\Http\Middleware\HandleCors::class,
        ]);
    })
    ->withExceptions(function (Exceptions $exceptions) {
        // Ensure API requests return JSON responses
        $exceptions->shouldRenderJsonWhen(function ($request) {
            if ($request->is('api/*')) {
                return true;
            }

            return $request->expectsJson();
        });

        // Custom response for unauthenticated API requests
        $exceptions->respond(function ($response, $_, $request) {
            if ($request->is('api/*') && $response->getStatusCode() === 401) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthenticated. Please provide a valid API token.',
                    'errors' => null,
                    'error_code' => 'UNAUTHORIZED',
                    'meta' => null,
                ], 401);
            }

            if ($request->is('api/*') && $response->getStatusCode() === 403) {
                return response()->json([
                    'success' => false,
                    'message' => 'This action is unauthorized.',
                    'errors' => null,
                    'error_code' => 'FORBIDDEN',
                    'meta' => null,
                ], 403);
            }

            if ($request->is('api/*') && $response->getStatusCode() === 404) {
                return response()->json([
                    'success' => false,
                    'message' => 'The requested resource was not found.',
                    'errors' => null,
                    'error_code' => 'RESOURCE_NOT_FOUND',
                    'meta' => null,
                ], 404);
            }

            if ($request->is('api/*') && $response->getStatusCode() === 500) {
                return response()->json([
                    'success' => false,
                    'message' => 'An internal server error occurred.',
                    'errors' => null,
                    'error_code' => 'SERVER_ERROR',
                    'meta' => null,
                ], 500);
            }

            return $response;
        });
    })->create();
