<?php

/*
|--------------------------------------------------------------------------
| Settings Configuration
|--------------------------------------------------------------------------
|
| This file defines the structure of the dynamic settings system.
|
| Structure:
|  - groups: top-level tabs in the settings interface
|  - sections: sub-groups within each tab
|  - fields: individual setting rows
|
| For file upload fields with type 'file', you can add 'image_options' array
| with the following available options:
|
| Image Options (all optional):
| - 'avatar' => bool               // Display as circular avatar
| - 'image_editor' => bool         // Enable image editing capabilities
| - 'resize_mode' => string        // 'contain', 'cover', 'stretch', 'crop'
| - 'resize_target_width' => int   // Target width in pixels
| - 'resize_target_height' => int  // Target height in pixels
| - 'resize_upscale' => bool       // Allow upscaling smaller images
| - 'crop_aspect_ratio' => string  // e.g., '16:9', '1:1', '3:1'
| - 'circle_cropper' => bool       // Enable circular cropping tool
| - 'accepted_types' => array      // MIME types ['image/jpeg', 'image/png']
| - 'previewable' => bool          // Show file preview
| - 'downloadable' => bool         // Add download button
| - 'openable' => bool             // Add "open in new tab" button
| - 'deletable' => bool            // Show delete button
| - 'viewport_width' => int        // Editor viewport width
| - 'viewport_height' => int       // Editor viewport height
| - 'editor_mode' => int           // 1=crop only, 2=crop+resize
| - 'editor_empty_fill_color' => string // Background color for transparent areas
|
*/

return [

    /*
    |--------------------------------------------------------------------------
    | Settings Definition
    |--------------------------------------------------------------------------
    | Structure:
    |  - groups: top-level tabs with unique IDs
    |  - sections: sub-groups within each tab
    |  - fields: individual setting rows
    |
    | Options types:
    |  - array: static array of options
    |  - enum: PHP enum class
    |  - callback: callback function name
    |  - model: Eloquent model class
    |
    | Radio field attributes:
    |  - inline: true/false (display radio buttons inline)
    |  - descriptions: array of descriptions for each option
    */

    'groups'    => [

        'general'    => [
            'id'       => 'general',
            'label'    => 'General',
            'icon'     => 'heroicon-o-cog',
            'sections' => [
                'general'      => [
                    'label'       => 'General',
                    'description' => 'Basic app settings',
                    'fields'      => [
                        [
                            'key'         => 'app_name',
                            'type'        => 'text',
                            'label'       => 'App Name',
                            'placeholder' => 'My Application',
                            'default'     => config('app.name'),
                            'rules'       => 'required|string|max:255',
                        ],
                        [
                            'key'     => 'app_tagline',
                            'type'    => 'textarea',
                            'label'   => 'Tagline',
                            'default' => 'Welcome to our app',
                            'rules'   => 'nullable|string|max:500',
                        ],
                        [
                            'key'     => 'admin_email',
                            'type'    => 'text',
                            'label'   => 'Admin Email',
                            'default' => 'info@espmarketplace.com',
                            'rules'   => 'required|email',
                        ],
                        [
                            'key'          => 'maintenance_mode',
                            'type'         => 'radio',
                            'label'        => 'Maintenance Mode',
                            'options_type' => 'array',
                            'options'      => ['enabled' => 'Enabled', 'disabled' => 'Disabled'],
                            'descriptions' => [
                                'enabled'  => 'Site will be down for maintenance',
                                'disabled' => 'Site will be accessible to all users',
                            ],
                            'inline'       => true,
                            'default'      => 'disabled',
                            'rules'        => 'required|string',
                        ],
                    ],
                ],
                'logos'        => [
                    'label'       => 'Logos & Branding',
                    'description' => 'Application logos and favicon',
                    'fields'      => [
                        [
                            'key'           => 'ligth_logo',
                            'type'          => 'file',
                            'label'         => 'Light Logo',
                            'rules'         => 'nullable|image|max:5120', // Increased to 5MB for better quality
                            'image_options' => [
                                'image_editor'        => true,
                                'resize_mode'         => 'contain',
                                'resize_target_width' => 180,
                                'resize_upscale'      => true,
                                'accepted_types'      => ['image/png', 'image/webp', 'image/jpeg', 'image/svg+xml'], // Lossless formats for quality
                                'previewable'         => true,
                                'downloadable'        => true,
                                'openable'            => true,
                                'deletable'           => true,
                            ],
                        ],
                        [
                            'key'           => 'dark_logo',
                            'type'          => 'file',
                            'label'         => 'Dark Logo',
                            'rules'         => 'nullable|image|max:5120', // Increased to 5MB for better quality
                            'image_options' => [
                                'image_editor'   => true,
                                'resize_mode'    => 'contain',
                                'resize_upscale' => true,
                                'accepted_types' => ['image/png', 'image/webp', 'image/jpeg', 'image/svg+xml'], // Lossless formats for quality
                                'previewable'    => true,
                                'downloadable'   => true,
                                'openable'       => true,
                                'deletable'      => true,
                            ],
                        ],
                        [
                            'key'           => 'app_favicon',
                            'type'          => 'file',
                            'label'         => 'Favicon',
                            'rules'         => 'nullable|image|max:512|dimensions:width=16,height=16',
                            'image_options' => [
                                'avatar'               => true,
                                'image_editor'         => true,
                                'resize_mode'          => 'cover',
                                'resize_target_width'  => 32,
                                'resize_target_height' => 32,
                                'resize_upscale'       => false,
                                'crop_aspect_ratio'    => '1:1',
                                'circle_cropper'       => true,
                                'accepted_types'       => ['image/png', 'image/x-icon', 'image/vnd.microsoft.icon'],
                                'previewable'          => true,
                                'downloadable'         => true,
                                'openable'             => true,
                                'viewport_width'       => 64,
                                'viewport_height'      => 64,
                                'deletable'            => true,
                            ],
                        ],
                    ],
                ],
                'localization' => [
                    'label'       => 'Localization',
                    'description' => 'Date, time, currency',
                    'fields'      => [
                        [
                            'key'          => 'app_timezone',
                            'type'         => 'select',
                            'label'        => 'Timezone',
                            'options_type' => 'callback',
                            'options'      => 'getTimezoneOptions',
                            'default'      => config('app.timezone'),
                            'rules'        => 'required|string',
                        ],
                        [
                            'key'          => 'app_currency',
                            'type'         => 'select',
                            'label'        => 'Currency',
                            'options_type' => 'array',
                            'options'      => ['USD' => 'USD', 'EUR' => 'EUR', 'PKR' => 'PKR'],
                            'default'      => 'USD',
                            'rules'        => 'required|string',
                        ],
                        [
                            'key'                 => 'date_format',
                            'type'                => 'radio',
                            'label'               => 'Date Format',
                            'options_type'        => 'enum',
                            'options'             => 'App\Enums\DateFormat',
                            'option_descriptions' => false,
                            'inline'              => true,
                            'default'             => 'Y-m-d',
                            'rules'               => 'required|string',
                        ],
                    ],
                ],
            ],
        ],

        'email'      => [
            'id'       => 'email',
            'label'    => 'Email',
            'icon'     => 'heroicon-o-envelope',
            'sections' => [
                'smtp'   => [
                    'label'       => 'SMTP Configuration',
                    'description' => 'Email server settings',
                    'fields'      => [
                        [
                            'key'     => 'smtp_host',
                            'type'    => 'text',
                            'label'   => 'SMTP Host',
                            'default' => env('MAIL_HOST', 'smtp.mailgun.org'),
                            'rules'   => 'required|string',
                        ],
                        [
                            'key'     => 'smtp_port',
                            'type'    => 'text',
                            'label'   => 'SMTP Port',
                            'default' => env('MAIL_PORT', '587'),
                            'rules'   => 'required|numeric',
                        ],
                        [
                            'key'     => 'smtp_username',
                            'type'    => 'text',
                            'label'   => 'SMTP Username',
                            'default' => env('MAIL_USERNAME'),
                            'rules'   => 'nullable|string',
                        ],
                        [
                            'key'     => 'smtp_password',
                            'type'    => 'password',
                            'label'   => 'SMTP Password',
                            'default' => env('MAIL_PASSWORD'),
                            'rules'   => 'nullable|string',
                        ],
                        [
                            'key'          => 'smtp_encryption',
                            'type'         => 'select',
                            'label'        => 'Encryption',
                            'options_type' => 'array',
                            'options'      => ['tls' => 'TLS', 'ssl' => 'SSL', 'none' => 'None'],
                            'default'      => env('MAIL_ENCRYPTION', 'tls'),
                            'rules'        => 'required|string',
                        ],
                    ],
                ],
                'sender' => [
                    'label'       => 'Default Sender',
                    'description' => 'Default from address',
                    'fields'      => [
                        [
                            'key'     => 'from_address',
                            'type'    => 'text',
                            'label'   => 'From Email',
                            'default' => env('MAIL_FROM_ADDRESS', 'hello@example.com'),
                            'rules'   => 'required|email',
                        ],
                        [
                            'key'     => 'from_name',
                            'type'    => 'text',
                            'label'   => 'From Name',
                            'default' => env('MAIL_FROM_NAME', config('app.name')),
                            'rules'   => 'required|string',
                        ],
                    ],
                ],
            ],
        ],

        'contact'    => [
            'id'       => 'contact',
            'label'    => 'Contact',
            'icon'     => 'heroicon-o-phone',
            'sections' => [
                'main' => [
                    'label'       => 'Contact Info',
                    'description' => 'Who users can reach out to',
                    'fields'      => [
                        [
                            'key'     => 'email',
                            'type'    => 'text',
                            'label'   => 'Support Email',
                            'default' => 'support@example.com',
                            'rules'   => 'required|email',
                        ],
                        [
                            'key'     => 'phone',
                            'type'    => 'text',
                            'label'   => 'Support Phone',
                            'default' => '',
                            'rules'   => 'nullable|string',
                        ],
                        [
                            'key'     => 'address',
                            'type'    => 'textarea',
                            'label'   => 'Office Address',
                            'default' => '',
                            'rules'   => 'nullable|string',
                        ],
                    ],
                ],
            ],
        ],

        'membership' => [
            'id'       => 'membership',
            'label'    => 'Membership',
            'icon'     => 'heroicon-o-user-group',
            'sections' => [
                'plans'   => [
                    'label'       => 'Membership Plans',
                    'description' => 'Configure membership plan settings',
                    'fields'      => [
                        [
                            'key'           => 'default_plan',
                            'type'          => 'select',
                            'label'         => 'Default Plan',
                            'options_type'  => 'model',
                            'options'       => 'App\Models\Membership\MembershipPlan',
                            'options_label' => 'package_title',
                            'options_value' => 'id',
                            'default'       => '',
                            'rules'         => 'nullable|exists:member_plans,id',
                        ],
                        [
                            'key'     => 'trial_days',
                            'type'    => 'text',
                            'label'   => 'Trial Period (days)',
                            'default' => '14',
                            'rules'   => 'required|numeric|min:0',
                        ],
                        [
                            'key'          => 'auto_approve',
                            'type'         => 'radio',
                            'label'        => 'Auto Approve New Members',
                            'options_type' => 'array',
                            'options'      => ['yes' => 'Yes', 'no' => 'No'],
                            'descriptions' => [
                                'yes' => 'New members will be automatically approved',
                                'no'  => 'New members require manual approval',
                            ],
                            'inline'       => true,
                            'default'      => 'no',
                            'rules'        => 'required|string',
                        ],
                    ],
                ],
                'credits' => [
                    'label'       => 'Credit Settings',
                    'description' => 'Configure credit pricing and bulk discounts',
                    'fields'      => [
                        [
                            'key'         => 'credit_price',
                            'type'        => 'number',
                            'label'       => 'Price per Credit',
                            'placeholder' => '1.00',
                            'default'     => '1.00',
                            'rules'       => 'required|numeric|min:0',
                            'help'        => 'The base price for one credit in your default currency.',
                        ],
                        [
                            'key'          => 'credit_currency',
                            'type'         => 'select',
                            'label'        => 'Credit Currency',
                            'options_type' => 'array',
                            'options'      => [
                                'USD' => 'USD - US Dollar',
                                'EUR' => 'EUR - Euro',
                                'GBP' => 'GBP - British Pound',
                                'CAD' => 'CAD - Canadian Dollar',
                                'AUD' => 'AUD - Australian Dollar',
                                'PKR' => 'PKR - Pakistani Rupee',
                            ],
                            'default'      => 'USD',
                            'rules'        => 'required|string',
                            'help'         => 'Currency used for credit pricing.',
                        ],
                        [
                            'key'     => 'bulk_discounts_enabled',
                            'type'    => 'toggle',
                            'label'   => 'Enable Bulk Discounts',
                            'default' => true,
                            'rules'   => 'boolean',
                            'help'    => 'Allow users to get discounts when purchasing credits in bulk.',
                        ],
                        [
                            'key'            => 'credit_bulk_discounts',
                            'type'           => 'repeater',
                            'label'          => 'Bulk Discount Tiers',
                            'default'        => json_encode([
                                ['quantity' => 10, 'discount' => 5],
                                ['quantity' => 25, 'discount' => 10],
                                ['quantity' => 50, 'discount' => 15],
                                ['quantity' => 100, 'discount' => 20],
                            ]),
                            'rules'          => 'nullable|json',
                            'help'           => 'Add multiple discount tiers. Users will get the highest applicable discount based on their purchase quantity.',
                            'schema'         => [
                                [
                                    'key'         => 'quantity',
                                    'type'        => 'number',
                                    'label'       => 'Minimum Quantity',
                                    'placeholder' => '10',
                                    'rules'       => 'required|numeric|min:1',
                                    'help'        => 'Minimum number of credits to qualify for this discount',
                                ],
                                [
                                    'key'         => 'discount',
                                    'type'        => 'number',
                                    'label'       => 'Discount Percentage',
                                    'placeholder' => '5',
                                    'rules'       => 'required|numeric|min:0|max:100',
                                    'help'        => 'Discount percentage (0-100)',
                                ],
                            ],
                            'addActionLabel' => 'Add Discount Tier',
                            'reorderable'    => true,
                            'collapsible'    => true,
                            'cloneable'      => false,
                            'minItems'       => 0,
                            'maxItems'       => 10,
                        ],
                        [
                            'key'         => 'max_credits_per_purchase',
                            'type'        => 'number',
                            'label'       => 'Maximum Credits per Purchase',
                            'placeholder' => '1000',
                            'default'     => '1000',
                            'rules'       => 'required|numeric|min:1',
                            'help'        => 'Maximum number of credits a user can purchase in a single transaction.',
                        ],
                        [
                            'key'         => 'min_credits_per_purchase',
                            'type'        => 'number',
                            'label'       => 'Minimum Credits per Purchase',
                            'placeholder' => '1',
                            'default'     => '1',
                            'rules'       => 'required|numeric|min:1',
                            'help'        => 'Minimum number of credits a user must purchase in a single transaction.',
                        ],
                        [
                            'key'     => 'credit_expiry_enabled',
                            'type'    => 'toggle',
                            'label'   => 'Enable Credit Expiry',
                            'default' => false,
                            'rules'   => 'boolean',
                            'help'    => 'Enable automatic expiry of credits after a certain period.',
                        ],
                        [
                            'key'         => 'credit_expiry_days',
                            'type'        => 'number',
                            'label'       => 'Credit Expiry Period (Days)',
                            'placeholder' => '365',
                            'default'     => '365',
                            'rules'       => 'nullable|numeric|min:1',
                            'help'        => 'Number of days after which purchased credits expire. Only applies if expiry is enabled.',
                        ],
                        [
                            'key'     => 'credit_refund_enabled',
                            'type'    => 'toggle',
                            'label'   => 'Enable Credit Refunds',
                            'default' => false,
                            'rules'   => 'boolean',
                            'help'    => 'Allow users to request refunds for unused credits.',
                        ],
                        [
                            'key'     => 'credit_transfer_enabled',
                            'type'    => 'toggle',
                            'label'   => 'Enable Credit Transfers',
                            'default' => false,
                            'rules'   => 'boolean',
                            'help'    => 'Allow users to transfer credits to other users.',
                        ],
                    ],
                ],
            ],
        ],

        'payments'   => [
            'id'       => 'payments',
            'label'    => 'Payment Gateways',
            'icon'     => 'heroicon-o-credit-card',
            'sections' => [
                'paypal'  => [
                    'label'       => 'PayPal Settings',
                    'description' => 'Configure PayPal payment gateway',
                    'fields'      => [
                        [
                            'key'          => 'paypal_mode',
                            'type'         => 'radio',
                            'label'        => 'PayPal Mode',
                            'options_type' => 'array',
                            'options'      => ['sandbox' => 'Sandbox (Testing)', 'live' => 'Live (Production)'],
                            'descriptions' => [
                                'sandbox' => 'Use PayPal sandbox for testing payments',
                                'live'    => 'Use PayPal live environment for real payments',
                            ],
                            'inline'       => true,
                            'default'      => 'sandbox',
                            'rules'        => 'required|in:sandbox,live',
                        ],
                        [
                            'key'     => 'paypal_client_id',
                            'type'    => 'text',
                            'label'   => 'PayPal Client ID',
                            'default' => env('PAYPAL_CLIENT_ID'),
                            'rules'   => 'nullable|string|max:255',
                        ],
                        [
                            'key'     => 'paypal_client_secret',
                            'type'    => 'password',
                            'label'   => 'PayPal Client Secret',
                            'default' => env('PAYPAL_CLIENT_SECRET'),
                            'rules'   => 'nullable|string|max:255',
                        ],
                        [
                            'key'     => 'paypal_sandbox_client_id',
                            'type'    => 'text',
                            'label'   => 'PayPal Sandbox Client ID',
                            'default' => env('PAYPAL_SANDBOX_CLIENT_ID'),
                            'rules'   => 'nullable|string|max:255',
                        ],
                        [
                            'key'     => 'paypal_sandbox_client_secret',
                            'type'    => 'password',
                            'label'   => 'PayPal Sandbox Client Secret',
                            'default' => env('PAYPAL_SANDBOX_CLIENT_SECRET'),
                            'rules'   => 'nullable|string|max:255',
                        ],
                        [
                            'key'          => 'paypal_currency',
                            'type'         => 'select',
                            'label'        => 'PayPal Currency',
                            'options_type' => 'array',
                            'options'      => [
                                'USD' => 'USD - US Dollar',
                                'EUR' => 'EUR - Euro',
                                'GBP' => 'GBP - British Pound',
                                'CAD' => 'CAD - Canadian Dollar',
                                'AUD' => 'AUD - Australian Dollar',
                                'JPY' => 'JPY - Japanese Yen',
                                'PKR' => 'PKR - Pakistani Rupee',
                            ],
                            'default'      => 'USD',
                            'rules'        => 'required|string',
                        ],
                        [
                            'key'     => 'paypal_webhook_url',
                            'type'    => 'text',
                            'label'   => 'PayPal Webhook URL',
                            'default' => '/webhook/paypal',
                            'rules'   => 'nullable|string',
                        ],
                        [
                            'key'     => 'paypal_enabled',
                            'type'    => 'toggle',
                            'label'   => 'Enable PayPal',
                            'default' => false,
                            'rules'   => 'boolean',
                        ],
                    ],
                ],
                'stripe'  => [
                    'label'       => 'Stripe Settings',
                    'description' => 'Configure Stripe payment gateway',
                    'fields'      => [
                        [
                            'key'          => 'stripe_mode',
                            'type'         => 'radio',
                            'label'        => 'Stripe Mode',
                            'options_type' => 'array',
                            'options'      => ['test' => 'Test Mode', 'live' => 'Live Mode'],
                            'descriptions' => [
                                'test' => 'Use Stripe test keys for testing payments',
                                'live' => 'Use Stripe live keys for real payments',
                            ],
                            'inline'       => true,
                            'default'      => 'test',
                            'rules'        => 'required|in:test,live',
                        ],
                        [
                            'key'     => 'stripe_publishable_key',
                            'type'    => 'text',
                            'label'   => 'Stripe Publishable Key (Live)',
                            'default' => env('STRIPE_KEY'),
                            'rules'   => 'nullable|string|max:255',
                        ],
                        [
                            'key'     => 'stripe_secret_key',
                            'type'    => 'password',
                            'label'   => 'Stripe Secret Key (Live)',
                            'default' => env('STRIPE_SECRET'),
                            'rules'   => 'nullable|string|max:255',
                        ],
                        [
                            'key'     => 'stripe_test_publishable_key',
                            'type'    => 'text',
                            'label'   => 'Stripe Test Publishable Key',
                            'default' => env('STRIPE_TEST_KEY'),
                            'rules'   => 'nullable|string|max:255',
                        ],
                        [
                            'key'     => 'stripe_test_secret_key',
                            'type'    => 'password',
                            'label'   => 'Stripe Test Secret Key',
                            'default' => env('STRIPE_TEST_SECRET'),
                            'rules'   => 'nullable|string|max:255',
                        ],
                        [
                            'key'     => 'stripe_webhook_secret',
                            'type'    => 'password',
                            'label'   => 'Stripe Webhook Secret',
                            'default' => env('STRIPE_WEBHOOK_SECRET'),
                            'rules'   => 'nullable|string|max:255',
                        ],
                        [
                            'key'          => 'stripe_currency',
                            'type'         => 'select',
                            'label'        => 'Stripe Currency',
                            'options_type' => 'array',
                            'options'      => [
                                'usd' => 'USD - US Dollar',
                                'eur' => 'EUR - Euro',
                                'gbp' => 'GBP - British Pound',
                                'cad' => 'CAD - Canadian Dollar',
                                'aud' => 'AUD - Australian Dollar',
                                'jpy' => 'JPY - Japanese Yen',
                                'pkr' => 'PKR - Pakistani Rupee',
                            ],
                            'default'      => 'usd',
                            'rules'        => 'required|string',
                        ],
                        [
                            'key'     => 'stripe_webhook_url',
                            'type'    => 'text',
                            'label'   => 'Stripe Webhook URL',
                            'default' => '/webhook/stripe',
                            'rules'   => 'nullable|string',
                        ],
                        [
                            'key'     => 'stripe_enabled',
                            'type'    => 'toggle',
                            'label'   => 'Enable Stripe',
                            'default' => false,
                            'rules'   => 'boolean',
                        ],
                    ],
                ],
                'general' => [
                    'label'       => 'General Payment Settings',
                    'description' => 'Configure general payment options',
                    'fields'      => [
                        [
                            'key'          => 'default_payment_gateway',
                            'type'         => 'select',
                            'label'        => 'Default Payment Gateway',
                            'options_type' => 'array',
                            'options'      => [
                                'stripe' => 'Stripe',
                                'paypal' => 'PayPal',
                            ],
                            'default'      => 'stripe',
                            'rules'        => 'required|in:stripe,paypal',
                        ],
                        [
                            'key'     => 'payment_tax_rate',
                            'type'    => 'text',
                            'label'   => 'Tax Rate (%)',
                            'default' => '0',
                            'rules'   => 'required|numeric|min:0|max:100',
                        ],
                        [
                            'key'     => 'payment_processing_fee',
                            'type'    => 'text',
                            'label'   => 'Processing Fee (%)',
                            'default' => '0',
                            'rules'   => 'required|numeric|min:0|max:100',
                        ],
                        [
                            'key'     => 'payment_minimum_amount',
                            'type'    => 'text',
                            'label'   => 'Minimum Payment Amount',
                            'default' => '1.00',
                            'rules'   => 'required|numeric|min:0',
                        ],
                    ],
                ],
            ],
        ],

        // footer
        'footer'     => [
            'id'       => 'footer',
            'label'    => 'Footer',
            'icon'     => 'heroicon-o-cog',
            'sections' => [
                'footer_general'      => [
                    'label'       => 'General',
                    'description' => 'Basic app settings',
                    'fields'      => [
                        [
                            'key'     => 'footer_app_about',
                            'type'    => 'textarea',
                            'label'   => 'About',
                            'default' => 'Welcome to our app',
                            'rules'   => 'nullable|string|max:500',
                        ],
                        [
                            'key'           => 'light_logo',
                            'type'          => 'file',
                            'label'         => 'Light Logo',
                            'rules'         => 'nullable|image|max:5120', // Increased to 5MB for better quality
                            'image_options' => [
                                'image_editor' => true,
                                'resize_mode' => 'contain',
                                'resize_target_width' => 180,
                                'resize_upscale' => true,
                                'accepted_types' => ['image/png', 'image/webp', 'image/svg+xml', 'image/jpeg'], // Lossless formats for quality
                                'previewable' => true,
                                'downloadable' => true,
                                'openable' => true,
                                'deletable' => true,

                            ],
                        ],
                        [
                            'key' => 'dark_logo',
                            'type' => 'file',
                            'label' => 'Dark Logo',
                            'rules' => 'nullable|image|max:5120', // Increased to 5MB for better quality
                            'image_options' => [
                                'image_editor' => true,
                                'resize_mode' => 'contain',
                                'resize_upscale' => true,
                                'resize_target_width' => 180,
                                'accepted_types' => ['image/png', 'image/webp', 'image/svg+xml'], // Lossless formats for quality
                                'previewable' => true,
                                'downloadable' => true,
                                'openable' => true,
                                'deletable' => true,
                            ],
                        ],
                        [
                            'key'     => 'facebook',
                            'type'    => 'text',
                            'label'   => 'Facebook',
                            'default' => 'https://www.facebook.com/',
                            'rules'   => 'nullable|string|max:100',
                        ],
                        [
                            'key'     => 'twitter',
                            'type'    => 'text',
                            'label'   => 'Twitter',
                            'default' => 'https://twitter.com/',
                            'rules'   => 'nullable|string|max:100',
                        ],
                        [
                            'key'     => 'instagram',
                            'type'    => 'text',
                            'label'   => 'Instagram',
                            'default' => 'https://www.instagram.com/',
                            'rules'   => 'nullable|string|max:100',
                        ],
                        [
                            'key'     => 'linkedin',
                            'type'    => 'text',
                            'label'   => 'LinkedIn',
                            'default' => 'https://www.linkedin.com/',
                            'rules'   => 'nullable|string|max:100',
                        ],
                        [
                            'key'     => 'copyright_text',
                            'type'    => 'text',
                            'label'   => 'Copyright Text',
                            'default' => '© ' . date('Y') . ' ESP Marketplace. All Rights Reserved.',
                            'rules'   => 'nullable|string|max:100',
                        ],
                    ],

                ],

                'footer_menus'        => [
                    'label'       => 'Footer Menus',
                    'description' => 'Footer menus',
                    'fields'      => [
                        [
                            'key'     => 'enable_footer_menu_1',
                            'type'    => 'toggle',
                            'label'   => 'Enable Footer Menu 1',
                            'default' => false,
                            'rules'   => 'boolean',
                        ],
                        [
                            'key'           => 'choose_menu_1',
                            'type'          => 'select',
                            'label'         => 'Choose Menu 1',
                            'options_type'  => 'model',
                            'options'       => 'Biostate\\FilamentMenuBuilder\\Models\\Menu',
                            'options_label' => 'name',
                            'options_value' => 'slug',
                            'rules'         => 'nullable|string|max:100',
                        ],
                        [
                            'key'     => 'enable_footer_menu_2',
                            'type'    => 'toggle',
                            'label'   => 'Enable Footer Menu 2',
                            'default' => false,
                            'rules'   => 'boolean',
                        ],
                        [
                            'key'           => 'choose_menu_2',
                            'type'          => 'select',
                            'label'         => 'Choose Menu 2',
                            'options_type'  => 'model',
                            'options'       => 'Biostate\\FilamentMenuBuilder\\Models\\Menu',
                            'options_label' => 'name',
                            'options_value' => 'slug',
                            'rules'         => 'nullable|string|max:100',
                        ],
                    ],
                ],

                'footer_subscription' => [
                    'label'       => 'Subscription',
                    'description' => 'Footer subscription',
                    'fields'      => [
                        [
                            'key'     => 'enable_footer_subscription',
                            'type'    => 'toggle',
                            'label'   => 'Enable Footer Subscription',
                            'default' => false,
                            'rules'   => 'boolean',
                        ],
                        [
                            'key'     => 'footer_subscription_heading',
                            'type'    => 'text',
                            'label'   => 'Footer Subscription Heading',
                            'default' => 'Subscribe to our newsletter',
                            'rules'   => 'nullable|string|max:100',
                        ],
                        [
                            'key'     => 'footer_subscription_sub_heading',
                            'type'    => 'text',
                            'label'   => 'Footer Subscription Sub Heading',
                            'default' => 'Get new job opportunities delivered directly to your inbox!',
                            'rules'   => 'nullable|string|max:100',
                        ],
                        [
                            'key'     => 'footer_subscription_helper_text',
                            'type'    => 'text',
                            'label'   => 'Footer Subscription Helper Text',
                            'default' => 'We respect your privacy. Unsubscribe anytime.',
                            'rules'   => 'nullable|string|max:100',
                        ],
                    ],
                ],
            ],
        ],

    ],

    /*
    |--------------------------------------------------------------------------
    | Options Callbacks
    |--------------------------------------------------------------------------
    | Define callback functions that return options for select fields
    */

    'callbacks' => [
        'getTimezoneOptions' => function () {
            return array_combine(\DateTimeZone::listIdentifiers(), \DateTimeZone::listIdentifiers());
        },
    ],

];
