<?php

namespace Database\Factories\Classified;

use App\Models\Classified\Classified;
use App\Models\Location\City;
use App\Models\Location\Country;
use App\Models\Location\State;
use App\Models\User\User;
use App\Enums\ClassifiedStatus;
use Illuminate\Database\Eloquent\Factories\Factory;

class ClassifiedFactory extends Factory
{
    protected $model = Classified::class;

    public function definition(): array
    {
        // Get Canada and its locations
        $country = Country::where('country_code', 'CA')->first();
        $state = $country ? State::where('country_id', $country->id)->inRandomOrder()->first() : null;
        $city = $state ? City::where('state_id', $state->id)->inRandomOrder()->first() : null;

        // Get a random existing user
        $user = User::inRandomOrder()->first();

        // Fallback to create a user if none exist
        if (!$user) {
            $user = User::factory()->create();
        }

        $price = $this->faker->randomFloat(2, 10, 10000);
        $discount_price = $price * (1 - $this->faker->randomFloat(2, 0.10, 0.30));
        return [
            'title' => $this->faker->sentence(),
            'description' => $this->faker->paragraphs(3, true),
            'price' => $price,
            'discount_price' => $discount_price,
            'website_link' => $this->faker->url(),
            'is_featured' => $this->faker->boolean(20), // 20% chance of being featured
            'location' => $city ? $city->city_name : ($state ? $state->state_name : $this->faker->city()),
            'address' => $this->faker->streetAddress(),
            'manufacture_date' => $this->faker->dateTimeBetween('-10 years', 'now'),
            'discount_start' => $this->faker->dateTimeBetween('-4 days', '-1 days'),
            'discount_end' => $this->faker->dateTimeBetween('+1 days', '+4 days'),
            'country_id' => $country?->id,
            'state_id' => $state?->id,
            'city_id' => $city?->id,
            'postal_code' => $this->faker->postcode(),
            'latitude' => $this->faker->latitude(41.7, 83.0),  // Latitude range for Canada
            'longitude' => $this->faker->longitude(-141.0, -52.6),  // Longitude range for Canada
            'user_id' => $user->id,
            'classified_status' => $this->faker->randomElement(ClassifiedStatus::cases()),
            'created_by' => $user->id, // Using the same user for both relationships
        ];
    }

    public function published(): static
    {
        return $this->state(fn(array $attributes) => [
            'classified_status' => ClassifiedStatus::Published,
        ]);
    }

    public function draft(): static
    {
        return $this->state(fn(array $attributes) => [
            'classified_status' => ClassifiedStatus::Draft,
        ]);
    }

    public function pending(): static
    {
        return $this->state(fn(array $attributes) => [
            'classified_status' => ClassifiedStatus::Pending,
        ]);
    }

    public function featured(): static
    {
        return $this->state(fn(array $attributes) => [
            'featured' => true,
        ]);
    }

    /**
     * Configure the factory to use a specific user
     */
    public function forUser(User $user): static
    {
        return $this->state(fn(array $attributes) => [
            'user_id' => $user->id,
            'created_by' => $user->id,
        ]);
    }
}
