<?php

namespace Database\Factories\Job;

use App\Enums\JobShift;
use App\Enums\JobStatus;
use App\Enums\JobWorkMode;
use App\Enums\PreferredGender;
use App\Enums\SalaryPeriod;
use App\Models\Currency;
use App\Models\Job\Job;
use App\Models\Job\JobCareerLevel;
use App\Models\Job\JobIndustry;
use App\Models\Job\JobRole;
use App\Models\Job\JobSkill;
use App\Models\Job\JobType;
use App\Models\Location\City;
use App\Models\Location\Country;
use App\Models\Location\State;
use App\Models\User\User;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Str;

class JobFactory extends Factory
{
    protected $model = Job::class;

    public function definition(): array
    {
        $title = $this->faker->jobTitle();
        $country = Country::inRandomOrder()->first() ?? Country::factory()->create();
        $state = State::where('country_id', $country->id)->inRandomOrder()->first() ?? State::factory()->create(['country_id' => $country->id]);
        $city = City::where('state_id', $state->id)->inRandomOrder()->first() ?? City::factory()->create(['state_id' => $state->id]);
        $employer = User::role('Service')->whereHas('profile')->inRandomOrder()->first()
            ?? User::whereHas('profile')->inRandomOrder()->first()
            ?? User::inRandomOrder()->first();

        // Get child job role (where parent_id is not null)
        $childJobRole = JobRole::whereNotNull('parent_id')->inRandomOrder()->first();

        // If no child roles exist, create one
        if (! $childJobRole) {
            $childJobRole = JobRole::factory()->child()->create();
        }

        $jobIndustry = JobIndustry::inRandomOrder()->first() ?? JobIndustry::factory()->create();
        $currency = Currency::inRandomOrder()->first() ?? Currency::factory()->create();
        $careerLevel = JobCareerLevel::inRandomOrder()->first() ?? JobCareerLevel::factory()->create();
        $jobType = JobType::inRandomOrder()->first() ?? JobType::factory()->create();

        return [
            'uuid' => Str::uuid(),
            'job_title' => $title,
            'slug' => Str::slug($title),
            'job_description' => $this->faker->paragraphs(3, true),
            'user_id' => $employer->id,
            'job_role_id' => $childJobRole->id,
            'job_industry_id' => $jobIndustry->id,
            'available_positions' => $this->faker->numberBetween(1, 10),
            'preferred_gender' => $this->faker->randomElement(PreferredGender::cases()),
            'min_salary' => $min = $this->faker->numberBetween(30000, 80000),
            'max_salary' => $this->faker->numberBetween($min, $min + 50000),
            'currency_id' => $currency->id,
            'job_career_level_id' => $careerLevel->id,
            'job_type_id' => $jobType->id,
            'salary_period' => $this->faker->randomElement(SalaryPeriod::cases()),
            'job_shift' => $this->faker->randomElement(JobShift::cases()),
            'work_mode' => $this->faker->randomElement(JobWorkMode::cases()),
            'apply_by_date' => $this->faker->dateTimeBetween('now', '+30 days'),
            'deactivate_apply_by' => $this->faker->boolean(),
            'key_responsibilities' => $this->faker->paragraphs(3, true),
            'required_qualifications' => $this->faker->paragraphs(2, true),
            'bonus_and_benefits' => $this->faker->paragraphs(2, true),
            'country_id' => $country->id,
            'state_id' => $state->id,
            'city_id' => $city->id,
            'created_by' => $employer->id,
            'published_at' => $this->faker->dateTimeBetween('-30 days', 'now'),
            'expired_at' => $this->faker->dateTimeBetween('+30 days', '+60 days'),
            'job_status' => JobStatus::Published,
        ];
    }

    /**
     * Configure the factory to attach skills to the job.
     */
    public function withSkills(int $count = 3): static
    {
        return $this->afterCreating(function (Job $job) use ($count) {
            // Get existing skills or create new ones if not enough exist
            $existingSkills = JobSkill::inRandomOrder()->limit($count)->get();

            if ($existingSkills->count() < $count) {
                $additionalSkills = JobSkill::factory()
                    ->count($count - $existingSkills->count())
                    ->create();
                $skills = $existingSkills->merge($additionalSkills);
            } else {
                $skills = $existingSkills;
            }

            $job->job_skills()->attach($skills);
        });
    }
}
