<?php

namespace Database\Factories\Job;

use App\Models\Job\JobRole;
use App\Models\User\User;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Str;

class JobRoleFactory extends Factory
{
    protected $model = JobRole::class;

    protected static array $parentRoles = [
        'Software Engineering',
        'Project Management',
        'Business Analysis',
        'Marketing',
        'Sales',
        'Human Resources',
        'Product Management',
        'Data Science',
        'Design',
        'Finance'
    ];

    protected static array $childRoleSuffixes = [
        'Junior',
        'Senior',
        'Lead',
        'Associate',
        'Principal',
        'Head of',
        'Director of',
        'VP of',
        'Chief',
        'Specialist'
    ];

    public function definition(): array
    {
        $name = $this->faker->unique()->randomElement(static::$parentRoles);
        return [
            'uuid' => Str::uuid(),
            'job_role_name' => $name,
            'slug' => Str::slug($name),
            'job_role_description' => $this->faker->paragraph(),
            'parent_id' => null, // Default to parent role
            'is_active' => $this->faker->boolean(80),
            'created_by' => User::factory()
        ];
    }

    /**
     * Configure the model factory to create a parent role.
     */
    public function parent(): static
    {
        return $this->state(function (array $attributes) {
            return [
                'parent_id' => null,
            ];
        });
    }

    /**
     * Configure the model factory to create a child role.
     * 
     * @param JobRole|null $parent Optional parent role to use
     */
    public function child(?JobRole $parent = null): static
    {
        return $this->state(function (array $attributes) use ($parent) {
            // If no parent provided, find or create one
            if (!$parent) {
                $parent = JobRole::whereNull('parent_id')->inRandomOrder()->first();

                if (!$parent) {
                    $parent = JobRole::factory()->parent()->create();
                }
            }

            $suffix = $this->faker->randomElement(static::$childRoleSuffixes);
            $childName = "{$suffix} {$parent->job_role_name}";

            return [
                'job_role_name' => $childName,
                'slug' => Str::slug($childName),
                'parent_id' => $parent->id,
            ];
        });
    }
}
