<?php

declare(strict_types=1);

namespace Database\Factories\Membership;

use App\Models\Membership\Payment;
use App\Models\Membership\Purchase;
use App\Models\User\User;
use App\Enums\PaymentType;
use App\Enums\PaymentMethod;
use App\Enums\PaymentStatus;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Str;

/**
 * @extends Factory<Payment>
 */
class PaymentFactory extends Factory
{
    protected $model = Payment::class;

    public function definition(): array
    {
        return [
            'uuid' => Str::uuid(),
            'purchase_id' => Purchase::factory(),
            'user_id' => User::inRandomOrder()->first()?->id ?? User::factory(),
            'payment_type' => PaymentType::Payment,
            'payment_method' => $this->faker->randomElement(PaymentMethod::cases()),
            'amount' => $this->faker->randomFloat(2, 10, 500),
            'currency' => 'USD',
            'status' => $this->faker->randomElement(PaymentStatus::cases()),
            'transaction_id' => $this->faker->unique()->regexify('[A-Z0-9]{10}'),
            'service_response' => [
                'transaction_id' => $this->faker->regexify('[A-Z0-9]{20}'),
                'status' => 'completed',
                'fee' => $this->faker->randomFloat(2, 0.5, 5),
                'gateway_reference' => $this->faker->regexify('[A-Z0-9]{15}'),
            ],
            'payment_date' => $this->faker->dateTimeBetween('-1 year', 'now'),
            'notes' => $this->faker->optional()->sentence(),
            'created_by' => User::inRandomOrder()->first()?->id ?? 1,
        ];
    }

    public function successful(): static
    {
        return $this->state(function (array $attributes) {
            return [
                'status' => PaymentStatus::Paid,
                'service_response' => array_merge($attributes['service_response'] ?? [], [
                    'status' => 'completed',
                    'success' => true,
                ]),
            ];
        });
    }

    public function failed(): static
    {
        return $this->state(function (array $attributes) {
            return [
                'status' => PaymentStatus::Failed,
                'service_response' => array_merge($attributes['service_response'] ?? [], [
                    'status' => 'failed',
                    'error_code' => $this->faker->randomElement(['card_declined', 'insufficient_funds', 'expired_card']),
                    'error_message' => $this->faker->sentence(),
                ]),
            ];
        });
    }

    public function pending(): static
    {
        return $this->state(function (array $attributes) {
            return [
                'status' => PaymentStatus::Pending,
                'service_response' => array_merge($attributes['service_response'] ?? [], [
                    'status' => 'pending',
                    'pending_reason' => 'processing',
                ]),
            ];
        });
    }

    public function stripe(): static
    {
        return $this->state(function (array $attributes) {
            return [
                'payment_method' => PaymentMethod::Stripe,
                'transaction_id' => 'pi_' . $this->faker->regexify('[a-zA-Z0-9]{24}'),
                'service_response' => [
                    'id' => 'pi_' . $this->faker->regexify('[a-zA-Z0-9]{24}'),
                    'object' => 'payment_intent',
                    'status' => 'succeeded',
                    'amount' => $attributes['amount'] * 100, // Stripe uses cents
                    'currency' => strtolower($attributes['currency']),
                ],
            ];
        });
    }

    public function paypal(): static
    {
        return $this->state(function (array $attributes) {
            return [
                'payment_method' => PaymentMethod::Paypal,
                'transaction_id' => $this->faker->regexify('[A-Z0-9]{17}'),
                'service_response' => [
                    'id' => $this->faker->regexify('[A-Z0-9]{17}'),
                    'status' => 'COMPLETED',
                    'payer' => [
                        'email_address' => $this->faker->email(),
                        'payer_id' => $this->faker->regexify('[A-Z0-9]{13}'),
                    ],
                ],
            ];
        });
    }
}
