<?php

declare(strict_types=1);

namespace Database\Factories\Membership;

use App\Models\Membership\Purchase;
use App\Models\Membership\Payment;
use App\Models\Membership\Subscription;
use App\Models\User\User;
use App\Enums\PurchaseItemType;
use App\Enums\PaymentType;
use App\Enums\PaymentMethod;
use App\Enums\PaymentStatus;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Str;

/**
 * @extends Factory<Purchase>
 */
class PurchaseFactory extends Factory
{
    protected $model = Purchase::class;

    public function definition(): array
    {
        $amount = $this->faker->randomFloat(2, 10, 500);
        $taxAmount = $amount * 0.1; // 10% tax
        $discountAmount = $this->faker->boolean(30) ? $amount * 0.05 : 0; // 30% chance of 5% discount
        $totalAmount = $amount + $taxAmount - $discountAmount;

        return [
            'uuid' => Str::uuid(),
            'user_id' => User::inRandomOrder()->first()?->id ?? User::factory(),
            'item_type' => $this->faker->randomElement(PurchaseItemType::cases()),
            'amount' => $amount,
            'currency' => 'USD',
            'quantity' => $this->faker->numberBetween(1, 5),
            'total_amount' => $totalAmount,
            'tax_amount' => $taxAmount,
            'discount_amount' => $discountAmount,
            'status' => $this->faker->boolean(85), // 85% chance of successful purchase
            'purchase_date' => $this->faker->dateTimeBetween('-1 year', 'now'),
            'notes' => $this->faker->optional()->sentence(),
            'metadata' => [
                'source' => $this->faker->randomElement(['web', 'api', 'admin']),
                'user_agent' => $this->faker->userAgent(),
                'ip_address' => $this->faker->ipv4(),
            ],
            'created_by' => User::inRandomOrder()->first()?->id ?? 1,
        ];
    }

    public function forSubscription(Subscription $subscription): static
    {
        return $this->state(function (array $attributes) use ($subscription) {
            return [
                'user_id' => $subscription->user_id,
                'item_type' => PurchaseItemType::Membership,
                'purchaseable_type' => Subscription::class,
                'purchaseable_id' => $subscription->id,
                'amount' => $this->calculateSubscriptionAmount($subscription),
                'purchase_date' => $subscription->purchase_date,
                'created_by' => $subscription->created_by ?? 1,
            ];
        });
    }

    public function withPayment(): static
    {
        return $this->afterCreating(function (Purchase $purchase) {
            Payment::factory()->create([
                'purchase_id' => $purchase->id,
                'user_id' => $purchase->user_id,
                'amount' => $purchase->total_amount,
                'currency' => $purchase->currency,
                'payment_date' => $purchase->purchase_date,
                'status' => $purchase->status ? PaymentStatus::Paid : PaymentStatus::Failed,
                'created_by' => $purchase->created_by,
            ]);
        });
    }

    public function successful(): static
    {
        return $this->state(function (array $attributes) {
            return [
                'status' => true,
            ];
        });
    }

    public function failed(): static
    {
        return $this->state(function (array $attributes) {
            return [
                'status' => false,
            ];
        });
    }

    private function calculateSubscriptionAmount(Subscription $subscription): float
    {
        $membershipPlan = $subscription->membershipPlan;
        if (!$membershipPlan) {
            return 99.99; // Default amount
        }

        return match ($subscription->payment_term) {
            \App\Enums\MembershipPaymentTerm::Monthly => $membershipPlan->price_monthly ?? 29.99,
            \App\Enums\MembershipPaymentTerm::Quarterly => $membershipPlan->price_quarterly ?? 79.99,
            \App\Enums\MembershipPaymentTerm::SemiYearly => $membershipPlan->price_half_yearly ?? 149.99,
            \App\Enums\MembershipPaymentTerm::Yearly => $membershipPlan->price_yearly ?? 299.99,
            default => 99.99,
        };
    }
}
