<?php

namespace Database\Factories\User;

use App\Enums\InquiryStatus;
use App\Enums\InquiryType;
use App\Models\User\Contact;
use App\Models\User\Inquiry;
use App\Models\User\User;
use Illuminate\Database\Eloquent\Factories\Factory;

class InquiryFactory extends Factory
{
    protected $model = Inquiry::class;

    public function definition(): array
    {
        return [
            'contact_id' => Contact::factory(),
            'user_id' => $this->faker->boolean(70) ? User::factory() : null, // 70% chance of having a user
            'form_data' => $this->generateFormData(),
            'inquiry_status' => $this->faker->randomElement(InquiryStatus::cases()),
            'inquiry_type' => $this->faker->randomElement(InquiryType::cases()),
            'inquiryable_type' => null,
            'inquiryable_id' => null
        ];
    }

    private function generateFormData(): array
    {
        $data = [
            'name' => $this->faker->name(),
            'email' => $this->faker->safeEmail(),
            'phone' => $this->faker->phoneNumber(),
            'message' => $this->faker->paragraph(3),
        ];

        // Randomly add additional fields
        if ($this->faker->boolean(50)) {
            $data['company'] = $this->faker->company();
        }

        if ($this->faker->boolean(40)) {
            $data['address'] = $this->faker->address();
        }

        if ($this->faker->boolean(30)) {
            $data['budget'] = $this->faker->randomElement(['$1,000 - $5,000', '$5,000 - $15,000', '$15,000+', 'Not sure']);
        }

        if ($this->faker->boolean(25)) {
            $data['timeline'] = $this->faker->randomElement(['ASAP', 'Within 1 month', '1-3 months', '3+ months']);
        }

        return $data;
    }

    public function forUserProfile(): static
    {
        return $this->state(fn(array $attributes) => [
            'inquiry_type' => InquiryType::Profile,
            'form_data' => array_merge($this->generateFormData(), [
                'inquiry_about' => 'Your company profile',
                'interested_services' => $this->faker->randomElements([
                    'Swimming Pool Construction',
                    'Pool Maintenance',
                    'Equipment Installation',
                    'Consultation',
                ], $this->faker->numberBetween(1, 3)),
            ]),
        ]);
    }

    public function forJob(): static
    {
        return $this->state(fn(array $attributes) => [
            'inquiry_type' => InquiryType::Jobs,
            'form_data' => array_merge($this->generateFormData(), [
                'inquiry_about' => 'Job opportunity',
                'experience_years' => $this->faker->numberBetween(0, 20),
                'availability' => $this->faker->randomElement(['Immediate', '2 weeks notice', '1 month notice']),
                'expected_salary' => $this->faker->randomElement(['$40,000 - $60,000', '$60,000 - $80,000', '$80,000+']),
            ]),
        ]);
    }

    public function forClassified(): static
    {
        return $this->state(fn(array $attributes) => [
            'inquiry_type' => InquiryType::Classified,
            'form_data' => array_merge($this->generateFormData(), [
                'inquiry_about' => 'Classified ad item',
                'interested_in' => $this->faker->randomElement(['Purchase', 'More information', 'Price negotiation']),
                'pickup_delivery' => $this->faker->randomElement(['Pickup', 'Delivery', 'Either']),
            ]),
        ]);
    }

    public function forProduct(): static
    {
        return $this->state(fn(array $attributes) => [
            'inquiry_type' => InquiryType::Product,
            'form_data' => array_merge($this->generateFormData(), [
                'inquiry_about' => 'Product information',
                'quantity' => $this->faker->numberBetween(1, 10),
                'delivery_location' => $this->faker->city(),
            ]),
        ]);
    }

    public function forArticle(): static
    {
        return $this->state(fn(array $attributes) => [
            'inquiry_type' => InquiryType::Article,
            'form_data' => array_merge($this->generateFormData(), [
                'inquiry_about' => 'Article content',
                'question_type' => $this->faker->randomElement(['General question', 'Request for clarification', 'Additional information']),
            ]),
        ]);
    }

    public function pending(): static
    {
        return $this->state(fn(array $attributes) => [
            'inquiry_status' => InquiryStatus::Pending,
        ]);
    }

    public function contacted(): static
    {
        return $this->state(fn(array $attributes) => [
            'inquiry_status' => InquiryStatus::Contacted,
        ]);
    }
}
