<?php

namespace Database\Seeders;

use App\Models\Classified\Classified;
use App\Models\Classified\ClassifiedCategory;
use App\Models\Classified\ClassifiedDelivery;
use App\Models\Classified\ClassifiedTag;
use App\Models\Classified\ClassifiedType;
use App\Models\Seo;
use App\Models\User\User;
use Illuminate\Database\Seeder;

class ClassifiedSeeder extends Seeder
{
    public function run(): void
    {
        // Get existing users
        $authors = User::all();

        if ($authors->isEmpty()) {
            // Fallback to create a user if none exist
            $authors = collect([User::factory()->create()]);
        }

        // Get existing records
        $types = ClassifiedType::all();
        $categories = ClassifiedCategory::all();
        $tags = ClassifiedTag::all();
        $deliveries = ClassifiedDelivery::all();

        // Ensure we have records to work with
        if ($types->isEmpty() || $categories->isEmpty() || $tags->isEmpty()) {
            throw new \Exception('Please run JobCategorySeeder first to create required records');
        }

        // Create classifieds in batches to avoid memory issues
        $totalClassifieds = 100;
        $batchSize = 100;
        $batches = ceil($totalClassifieds / $batchSize);

        if ($this->command) {
            $this->command->getOutput()->writeln("Creating {$totalClassifieds} classifieds in {$batches} batches...");
        }

        for ($batch = 1; $batch <= $batches; $batch++) {
            if ($this->command) {
                $this->command->getOutput()->writeln("Processing batch {$batch} of {$batches}...");
            }

            // Create classifieds and attach relationships
            Classified::factory()
                ->count($batchSize)
                ->state(fn () => ['created_by' => $authors->random()->id])
                ->create()
                ->each(function ($classified) use ($categories, $tags, $types, $deliveries) {
                    // Attach 1-2 types
                    $classified->types()->attach(
                        $types->random(rand(1, 2))->pluck('id')->toArray()
                    );

                    // Attach 1-3 categories
                    $classified->categories()->attach(
                        $categories->random(rand(1, 3))->pluck('id')->toArray()
                    );

                    // Attach 2-5 tags
                    $classified->tags()->attach(
                        $tags->random(rand(2, 5))->pluck('id')->toArray()
                    );

                    // Attach 1 delivery
                    $classified->deliveries()->attach(
                        $deliveries->random(1)->pluck('id')->toArray()
                    );

                    // Create SEO record for the classified
                    $classified->seo()->create([
                        'title' => fake()->sentence(6),
                        'description' => fake()->paragraph(2),
                    ]);
                });
        }

        if ($this->command) {
            $this->command->getOutput()->writeln("Successfully created {$totalClassifieds} classifieds!");
        }
    }
}
