<?php

namespace Database\Seeders;

use App\Models\User\Inquiry;
use App\Models\User\Contact;
use App\Models\User\User;
use App\Models\User\UserProfile;
use App\Models\Job\Job;
use App\Models\Classified\Classified;
use App\Models\Product\Product;
use App\Models\Blog\Post;
use Illuminate\Database\Seeder;

class InquirySeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('Creating inquiries for user profiles...');
        $this->createInquiriesForUserProfiles();

        $this->command->info('Creating inquiries for jobs...');
        $this->createInquiriesForJobs();

        $this->command->info('Creating inquiries for classifieds...');
        $this->createInquiriesForClassifieds();

        $this->command->info('Creating inquiries for products...');
        $this->createInquiriesForProducts();

        $this->command->info('Creating inquiries for posts/articles...');
        $this->createInquiriesForPosts();

        $this->command->info('Inquiry seeder completed successfully!');
    }

    private function createNewContact(): Contact
    {
        return Contact::factory()->create([
            'created_by' => null, // Don't assign a user to the contact
        ]);
    }

    private function createInquiriesForUserProfiles(): void
    {
        $userProfiles = UserProfile::all();

        if ($userProfiles->isEmpty()) {
            $this->command->warn('No user profiles found. Skipping user profile inquiries.');
            return;
        }

        $totalInquiries = 0;
        foreach ($userProfiles as $profile) {
            // Create 2-5 inquiries per profile as requested
            $inquiryCount = rand(2, 5);

            for ($i = 0; $i < $inquiryCount; $i++) {
                $contact = $this->createNewContact();

                Inquiry::factory()
                    ->forUserProfile()
                    ->create([
                        'contact_id' => $contact->id,
                        'user_id' => $profile->user_id, // Use the profile's user_id
                        'inquiryable_type' => UserProfile::class,
                        'inquiryable_id' => $profile->id,
                    ]);
                $totalInquiries++;
            }
        }

        $this->command->info("Created {$totalInquiries} inquiries for {$userProfiles->count()} user profiles.");
    }

    private function createInquiriesForJobs(): void
    {
        $jobs = Job::all();

        if ($jobs->isEmpty()) {
            $this->command->warn('No jobs found. Skipping job inquiries.');
            return;
        }

        foreach ($jobs as $job) {
            $contact = $this->createNewContact();

            // Use user_id first, fallback to created_by
            $userId = $job->user_id ?? $job->created_by;

            Inquiry::factory()
                ->forJob()
                ->create([
                    'contact_id' => $contact->id,
                    'user_id' => $userId,
                    'inquiryable_type' => Job::class,
                    'inquiryable_id' => $job->id,
                ]);
        }

        $this->command->info("Created inquiries for {$jobs->count()} jobs.");
    }

    private function createInquiriesForClassifieds(): void
    {
        $classifieds = Classified::all();

        if ($classifieds->isEmpty()) {
            $this->command->warn('No classifieds found. Skipping classified inquiries.');
            return;
        }

        foreach ($classifieds as $classified) {
            $contact = $this->createNewContact();

            // Use user_id first, fallback to created_by
            $userId = $classified->user_id ?? $classified->created_by;

            Inquiry::factory()
                ->forClassified()
                ->create([
                    'contact_id' => $contact->id,
                    'user_id' => $userId,
                    'inquiryable_type' => Classified::class,
                    'inquiryable_id' => $classified->id,
                ]);
        }

        $this->command->info("Created inquiries for {$classifieds->count()} classifieds.");
    }

    private function createInquiriesForProducts(): void
    {
        $products = Product::all();

        if ($products->isEmpty()) {
            $this->command->warn('No products found. Skipping product inquiries.');
            return;
        }

        foreach ($products as $product) {
            $contact = $this->createNewContact();

            Inquiry::factory()
                ->forProduct()
                ->create([
                    'contact_id' => $contact->id,
                    'user_id' => $product->created_by, // Use created_by as the author
                    'inquiryable_type' => Product::class,
                    'inquiryable_id' => $product->id,
                ]);
        }

        $this->command->info("Created inquiries for {$products->count()} products.");
    }

    private function createInquiriesForPosts(): void
    {
        $posts = Post::all();

        if ($posts->isEmpty()) {
            $this->command->warn('No posts/articles found. Skipping post inquiries.');
            return;
        }

        foreach ($posts as $post) {
            $contact = $this->createNewContact();

            Inquiry::factory()
                ->forArticle()
                ->create([
                    'contact_id' => $contact->id,
                    'user_id' => $post->created_by, // Use created_by as the author
                    'inquiryable_type' => Post::class,
                    'inquiryable_id' => $post->id,
                ]);
        }

        $this->command->info("Created inquiries for {$posts->count()} posts/articles.");
    }
}
