<?php

namespace Database\Seeders;

use App\Enums\LeadFormFieldDataType;
use App\Enums\LeadUserStatus;
use App\Models\Lead\Lead;
use App\Models\Lead\LeadForm;
use App\Models\Lead\FieldLeadForm;
use App\Models\Lead\LeadFormFields;
use App\Models\Lead\LeadFormVisit;
use App\Models\User\MemberCategory;
use App\Models\User\User;
use Illuminate\Database\Seeder;

class LeadSeeder extends Seeder
{
    public function run(): void
    {
        // Create admin user if not exists
        $admin = User::firstOrCreate(
            ['email' => 'admin@admin.com'],
            [
                'name' => 'Admin User',
                'password' => bcrypt('admin123'),
            ]
        );

        // Get all users with Service role
        $serviceUsers = User::whereHas('roles', function ($query) {
            $query->where('name', 'Service');
        })->get();

        // Create member categories if they don't exist
        if (MemberCategory::count() < 3) {
            MemberCategory::factory(6)->create();
        }

        // Get all member categories
        $memberCategories = MemberCategory::all();

        // Get all lead form fields
        $leadFormFields = LeadFormFields::all();

        // Make sure we have form fields to use
        if ($leadFormFields->isEmpty()) {
            $this->call(LeadFormFieldSeeder::class);
            $leadFormFields = LeadFormFields::all();
        }

        // Create 5 lead forms
        LeadForm::factory(5)
            ->create(['created_by' => $admin->id])
            ->each(function ($leadForm) use ($admin, $serviceUsers, $memberCategories, $leadFormFields) {
                // Link all form fields to this form
                foreach ($leadFormFields as $index => $field) {
                    FieldLeadForm::create([
                        'lead_form_id' => $leadForm->id,
                        'field_id' => $field->id,
                        'is_required' => in_array($field->field_id, ['first_name', 'email', 'phone']),
                        'field_order' => $index + 1,
                        'is_visible' => true,
                    ]);
                }

                // Add a submit button as the last field if it doesn't exist
                $submitField = $leadFormFields->where('field_id', 'submit')->first();
                if (!$submitField) {
                    // Create a submit button field
                    $submitField = LeadFormFields::create([
                        'field_name' => 'Submit',
                        'field_id' => 'submit',
                        'data_type' => LeadFormFieldDataType::SubmitButton,
                        'is_default' => true,
                        'created_by' => $admin->id,
                    ]);

                    // Link the submit button to the form
                    FieldLeadForm::create([
                        'lead_form_id' => $leadForm->id,
                        'field_id' => $submitField->id,
                        'is_required' => true,
                        'field_order' => $leadFormFields->count() + 1,
                        'is_visible' => true,
                    ]);
                }

                // Attach 1-3 random member categories with pivot values
                $randomCategories = $memberCategories->random(rand(1, 3));
                foreach ($randomCategories as $category) {
                    $leadForm->memberCategories()->attach($category->id, [
                        'credits' => fake()->randomFloat(2, 10, 100),
                        'radius' => fake()->randomElement([5, 10, 15, 20, 25, 30, 50]),
                    ]);
                }

                // Create 200 leads for each form (1000 total)
                Lead::factory(200)
                    ->create([
                        'lead_form_id' => $leadForm->id,
                        'created_by' => $admin->id,
                    ])
                    ->each(function ($lead) use ($serviceUsers) {
                        // Assign 1-3 random service users to each lead
                        $randomUsers = $serviceUsers->random(rand(1, min(3, $serviceUsers->count())));

                        foreach ($randomUsers as $user) {
                            $lead->users()->attach($user->id, [
                                'lead_status' => fake()->randomElement(LeadUserStatus::cases())->value,
                            ]);
                        }
                    });

                // Create between 20-50 visits for each form
                $visitsCount = rand(20, 50);
                LeadFormVisit::factory()
                    ->count($visitsCount)
                    ->create([
                        'lead_form_id' => $leadForm->id,
                    ]);
            });
    }
}
