<?php

namespace Database\Seeders;

use App\Models\User\MemberCategory;
use App\Models\User\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Str;

class MemberCategorySeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get admin user or create a fallback user
        $admin = User::firstOrCreate(
            ['email' => 'admin@admin.com'],
            [
                'name' => 'Admin User',
                'password' => bcrypt('admin123'),
            ]
        );

        // Create parent categories if not enough exist
        if (MemberCategory::count() < 5) {
            $this->command->info('Creating member categories hierarchy...');
            $this->createCategoryHierarchy($admin->id);
        } else {
            $this->command->info('Sufficient member categories already exist.');
        }
    }

    /**
     * Create a hierarchy of member categories with parents, children, and sub-children
     */
    private function createCategoryHierarchy(int $createdBy): void
    {
        // Define main parent categories
        $parentCategories = [
            'Well Services' => [
                'description' => 'Services related to water wells',
                'children' => [
                    'Well Installation' => [
                        'description' => 'Services for installing new wells',
                        'children' => [
                            'Residential Well Installation' => 'Installation of wells for residential properties',
                            'Commercial Well Installation' => 'Installation of wells for commercial properties',
                            'Agricultural Well Installation' => 'Installation of wells for agricultural use'
                        ]
                    ],
                    'Well Maintenance' => [
                        'description' => 'Services for maintaining existing wells',
                        'children' => [
                            'Well Cleaning' => 'Cleaning services for wells',
                            'Well Repair' => 'Repair services for wells',
                            'Well Inspection' => 'Regular inspection services for wells'
                        ]
                    ]
                ]
            ],
            'Water Testing' => [
                'description' => 'Services related to water quality testing',
                'children' => [
                    'Drinking Water Testing' => [
                        'description' => 'Testing services for drinking water',
                        'children' => [
                            'Bacterial Testing' => 'Testing for bacterial contamination in water',
                            'Chemical Testing' => 'Testing for chemical contamination in water',
                            'Mineral Testing' => 'Testing for mineral content in water'
                        ]
                    ],
                    'Environmental Water Testing' => [
                        'description' => 'Testing services for environmental water',
                        'children' => [
                            'Surface Water Testing' => 'Testing for surface water quality',
                            'Groundwater Testing' => 'Testing for groundwater quality',
                            'Wastewater Testing' => 'Testing for wastewater quality'
                        ]
                    ]
                ]
            ],
            'Pump Services' => [
                'description' => 'Services related to water pumps',
                'children' => [
                    'Pump Installation' => [
                        'description' => 'Services for installing pumps',
                        'children' => [
                            'Submersible Pump Installation' => 'Installation of submersible pumps',
                            'Jet Pump Installation' => 'Installation of jet pumps',
                            'Hand Pump Installation' => 'Installation of hand pumps'
                        ]
                    ],
                    'Pump Repair' => [
                        'description' => 'Services for repairing pumps',
                        'children' => [
                            'Pump Motor Repair' => 'Repair services for pump motors',
                            'Pump Pressure Repair' => 'Repair services for pump pressure issues',
                            'Pump Replacement' => 'Services for replacing pumps'
                        ]
                    ]
                ]
            ]
        ];

        // Create the category hierarchy
        foreach ($parentCategories as $parentName => $parentData) {
            $parent = $this->createOrUpdateCategory(
                $parentName,
                $parentData['description'],
                null,
                $createdBy
            );

            if (isset($parentData['children']) && is_array($parentData['children'])) {
                foreach ($parentData['children'] as $childName => $childData) {
                    $child = $this->createOrUpdateCategory(
                        $childName,
                        $childData['description'],
                        $parent->id,
                        $createdBy
                    );

                    if (isset($childData['children']) && is_array($childData['children'])) {
                        foreach ($childData['children'] as $subChildName => $subChildDescription) {
                            $this->createOrUpdateCategory(
                                $subChildName,
                                $subChildDescription,
                                $child->id,
                                $createdBy
                            );
                        }
                    }
                }
            }
        }
    }

    /**
     * Create or update a member category
     */
    private function createOrUpdateCategory(
        string $name,
        string $description,
        ?int $parentId,
        int $createdBy
    ): MemberCategory {
        return MemberCategory::firstOrCreate(
            ['category_name' => $name],
            [
                'uuid' => Str::uuid(),
                'category_description' => $description,
                'parent_id' => $parentId,
                'created_by' => $createdBy,
                'is_active' => true,
            ]
        );
    }
}
