<?php

declare(strict_types=1);

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Membership\Subscription;
use App\Models\Membership\Purchase;
use App\Models\Membership\Payment;
use App\Enums\PurchaseItemType;
use App\Enums\PaymentType;
use App\Enums\PaymentMethod;
use App\Enums\PaymentStatus;
use App\Enums\MembershipPaymentTerm;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class PurchaseSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $this->command->info('Emptying purchases and payments tables...');

        // Empty the tables (this will also handle foreign key constraints properly)
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        DB::table('payments')->truncate();
        DB::table('purchases')->truncate();
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        // die();
        $this->command->info('Tables emptied successfully.');
        $this->command->info('Creating purchases for existing subscriptions...');

        // Get all subscriptions that don't already have purchases
        $subscriptions = Subscription::with(['user', 'membershipPlan'])
            ->whereDoesntHave('purchases')
            ->get();

        if ($subscriptions->isEmpty()) {
            $this->command->info('No subscriptions found without purchases.');
            return;
        }

        $this->command->info("Found {$subscriptions->count()} subscriptions to create purchases for.");

        $bar = $this->command->getOutput()->createProgressBar($subscriptions->count());
        $bar->start();

        $successCount = 0;
        $errorCount = 0;
        $errors = [];

        foreach ($subscriptions as $subscription) {
            DB::beginTransaction();
            try {
                // Calculate amounts
                $amount = $this->calculateSubscriptionAmount($subscription);
                $taxAmount = $amount * 0.1; // 10% tax
                $totalAmount = $amount + $taxAmount;

                // Create purchase
                $purchase = Purchase::create([
                    'uuid' => Str::uuid(),
                    'user_id' => $subscription->user_id,
                    'item_type' => PurchaseItemType::Subscription,
                    'purchaseable_type' => Subscription::class,
                    'purchaseable_id' => $subscription->id,
                    'amount' => $amount,
                    'currency' => 'USD',
                    'quantity' => 1,
                    'total_amount' => $totalAmount,
                    'tax_amount' => $taxAmount,
                    'discount_amount' => 0,
                    'notes' => "Purchase for {$subscription->membershipPlan?->package_title} subscription",
                    'created_by' => $subscription->created_by ?? 1,
                ]);

                // Create payment
                $payment = Payment::create([
                    'uuid' => Str::uuid(),
                    'purchase_id' => $purchase->id,
                    'user_id' => $subscription->user_id,
                    'payment_type' => PaymentType::Payment,
                    'payment_method' => $this->getRandomPaymentMethod(),
                    'amount' => $totalAmount,
                    'currency' => 'USD',
                    'status' => PaymentStatus::Paid,
                    'transaction_id' => 'seed_' . Str::random(20),
                    'service_response' => [
                        'status' => 'completed',
                        'transaction_id' => 'txn_' . Str::random(15),
                        'fee' => round($totalAmount * 0.029 + 0.30, 2), // Typical payment fee
                        'seeded' => true,
                    ],
                    'payment_date' => $purchase->purchase_date,
                    'notes' => "Payment for subscription #{$subscription->id}",
                    'created_by' => $subscription->created_by ?? 1,
                ]);

                DB::commit();
                $successCount++;
            } catch (\Exception $e) {
                DB::rollBack();
                $errorCount++;
                $errors[] = "Subscription {$subscription->id}: {$e->getMessage()}";
            }

            $bar->advance();
        }

        $bar->finish();
        $this->command->newLine(2);

        // Display results
        $this->command->info("Successfully created {$successCount} purchases with payments.");

        if ($errorCount > 0) {
            $this->command->error("Failed to create {$errorCount} purchases:");
            foreach ($errors as $error) {
                $this->command->error("  - {$error}");
            }
        }

        // Show summary
        if ($successCount > 0) {
            $this->command->info("\nPurchase Summary:");
            $recentPurchases = Purchase::with(['user', 'purchaseable', 'payments'])
                ->whereIn('purchaseable_id', $subscriptions->pluck('id'))
                ->where('purchaseable_type', Subscription::class)
                ->latest()
                ->take(5)
                ->get();

            $this->command->table(
                ['User', 'Plan', 'Amount', 'Payment Status', 'Date'],
                $recentPurchases->map(function ($purchase) {
                    $subscription = $purchase->purchaseable;
                    return [
                        $purchase->user->name ?? 'Unknown',
                        $subscription?->membershipPlan?->package_title ?? 'Unknown Plan',
                        '$' . number_format((float) $purchase->total_amount, 2),
                        $purchase->latestPayment()?->status?->getLabel() ?? 'No Payment',
                        $purchase->purchase_date?->format('Y-m-d') ?? 'N/A',
                    ];
                })->toArray()
            );

            if ($successCount > 5) {
                $this->command->info("... and " . ($successCount - 5) . " more purchases created.");
            }
        }
    }

    /**
     * Calculate the subscription amount based on payment term and membership plan
     */
    private function calculateSubscriptionAmount(Subscription $subscription): float
    {
        $membershipPlan = $subscription->membershipPlan;
        if (!$membershipPlan) {
            return 99.99; // Default amount
        }

        return match ($subscription->payment_term) {
            MembershipPaymentTerm::Monthly => (float) ($membershipPlan->price_monthly ?? 29.99),
            MembershipPaymentTerm::Quarterly => (float) ($membershipPlan->price_quarterly ?? 79.99),
            MembershipPaymentTerm::SemiYearly => (float) ($membershipPlan->price_half_yearly ?? 149.99),
            MembershipPaymentTerm::Yearly => (float) ($membershipPlan->price_yearly ?? 299.99),
            default => 99.99,
        };
    }

    /**
     * Get a random payment method for seeded payments
     */
    private function getRandomPaymentMethod(): PaymentMethod
    {
        return fake()->randomElement(PaymentMethod::cases());
    }
}
