<?php

namespace Database\Seeders;

use App\Models\Review;
use App\Models\User\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class ReviewSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get all users with the Service role
        $serviceUsers = User::role('Service')->get();

        if ($serviceUsers->isEmpty()) {
            $this->command->info('No users with the Service role found. Skipping review seeding.');
            return;
        }

        $this->command->info('Starting to seed reviews for ' . $serviceUsers->count() . ' service providers...');

        // Get a pool of potential reviewers
        $potentialReviewers = User::whereDoesntHave('roles', function ($query) {
            $query->where('name', 'Service');
        })->limit(100)->get();

        if ($potentialReviewers->isEmpty()) {
            $this->command->info('No reviewers found. Creating anonymous reviews.');
        }

        // For each service provider, create 50-100 reviews
        foreach ($serviceUsers as $serviceUser) {
            $reviewCount = rand(50, 100);
            $this->command->info("Creating {$reviewCount} reviews for {$serviceUser->name}");

            // Create reviews in chunks for better memory management
            $chunks = ceil($reviewCount / 10);
            $chunkSize = ceil($reviewCount / $chunks);

            for ($chunk = 0; $chunk < $chunks; $chunk++) {
                $batchSize = min($chunkSize, $reviewCount - ($chunk * $chunkSize));
                $this->command->info("  Processing batch " . ($chunk + 1) . " of {$chunks} ({$batchSize} reviews)");

                DB::beginTransaction();
                try {
                    for ($i = 0; $i < $batchSize; $i++) {
                        // Pick a random reviewer if available, otherwise null for anonymous
                        $reviewer = $potentialReviewers->isNotEmpty() ? $potentialReviewers->random() : null;

                        // Create the review
                        $review = Review::factory()->create([
                            'review_user_id' => $reviewer?->id,
                            'reviewable_type' => User::class,
                            'reviewable_id' => $serviceUser->id,
                        ]);

                        // Calculate the main review_rating as average of detailed ratings
                        // $averageRating = round($review->average_rating);
                        // $review->update(['review_rating' => $averageRating]);
                    }
                    DB::commit();
                } catch (\Exception $e) {
                    DB::rollBack();
                    $this->command->error("Error in batch " . ($chunk + 1) . ": " . $e->getMessage());
                }
            }

            $this->command->info("Completed {$reviewCount} reviews for {$serviceUser->name}");
        }

        $this->command->info('Finished seeding reviews for service providers.');
    }
}
