<?php

namespace Database\Seeders;

use App\Models\User\User;
use App\Models\User\MemberCategory;
use App\Models\User\UserProfile;
use App\Models\User\UserLocation;
use App\Models\User\MyRole;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class UserProfileSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get existing users (limit to 200 users)
        $users = User::take(200)->get();

        if ($users->isEmpty()) {
            $this->command->error('No users found. Please run the UserSeeder first.');
            return;
        }

        // Make sure we have member categories with appropriate hierarchy
        if (MemberCategory::count() < 5) {
            $this->command->info('Not enough member categories found. Running MemberCategorySeeder...');
            $this->call(MemberCategorySeeder::class);
        }

        // Get all member categories after potentially seeding them
        $memberCategories = MemberCategory::all();

        if ($memberCategories->isEmpty()) {
            $this->command->warn('No member categories found even after seeding. Skipping member category attachment.');
        } else {
            $this->command->info('Found ' . $memberCategories->count() . ' member categories.');
        }

        // Get the Service role
        $serviceRole = MyRole::where('name', 'Service')->first();

        if (!$serviceRole) {
            $this->command->warn('Service role not found. Make sure to run the RoleSeeder first.');
        } else {
            $this->command->info('Found Service role. Will assign to all users.');
        }

        // Create profiles and locations for each user
        $this->command->info('Creating user profiles and locations for ' . $users->count() . ' users...');

        $progressBar = $this->command->getOutput()->createProgressBar($users->count());
        $progressBar->start();

        $users->each(function ($user) use ($memberCategories, $serviceRole, $progressBar) {
            $this->createUserProfile($user);
            $this->createUserLocation($user);

            // Assign Service role to the user
            if ($serviceRole) {
                $user->assignRole($serviceRole->name);
            }

            if ($memberCategories->isNotEmpty()) {
                $this->attachMemberCategories($user, $memberCategories);
            }

            $progressBar->advance();
        });

        $progressBar->finish();
        $this->command->info("\nUser profiles seeding completed.");
    }

    /**
     * Create a user profile for the given user
     */
    private function createUserProfile(User $user): void
    {
        // Create user profile only if it doesn't exist
        if (!$user->profile) {
            $profile = UserProfile::factory()->make();
            $profile->user_id = $user->id;
            $profile->save();
        }
    }

    /**
     * Create a primary location for the given user
     */
    private function createUserLocation(User $user): void
    {
        // Create primary location if the user doesn't have one
        if (!$user->primaryLocation) {
            $location = UserLocation::factory()->primary()->make();
            $location->user_id = $user->id;
            $location->created_by = $user->id;
            $location->save();
        }
    }

    /**
     * Attach member categories to the user
     * 
     * This attaches at least two subcategories along with their parent categories
     */
    private function attachMemberCategories(User $user, $allCategories): void
    {
        try {
            // Get leaf categories (those without children)
            $leafCategories = $allCategories->filter(function ($category) use ($allCategories) {
                return !$allCategories->contains('parent_id', $category->id);
            });

            if ($leafCategories->isEmpty()) {
                // Fallback to random categories if no leaf categories
                $categories = $allCategories->random(min(2, $allCategories->count()));
            } else {
                // Pick at least 2 random leaf (child) categories
                $categories = $leafCategories->random(min(2, $leafCategories->count()));
            }

            $categoriesToAttach = collect();

            // For each selected category, also include all parent categories
            foreach ($categories as $category) {
                $categoriesToAttach->push($category);

                // Add parent category chain
                $parentId = $category->parent_id;
                while ($parentId) {
                    $parent = $allCategories->firstWhere('id', $parentId);
                    if ($parent) {
                        $categoriesToAttach->push($parent);
                        $parentId = $parent->parent_id;
                    } else {
                        break;
                    }
                }
            }

            // Get unique category IDs to prevent duplicate attachments
            $categoryIds = $categoriesToAttach->unique('id')->pluck('id')->toArray();

            // Attach categories to user
            $user->memberCategories()->sync($categoryIds);
        } catch (\Exception $e) {
            Log::error('Error attaching member categories to user: ' . $e->getMessage());
        }
    }
}
