<?php

namespace Database\Seeders;

use App\Models\User\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;

class UserSeeder extends Seeder
{
    public function run(): void
    {
        // Create the admin user
        $adminUser = User::create([
            'name' => 'admin',
            'email' => 'admin@admin.com',
            'password' => Hash::make('admin123'),
            'email_verified_at' => now(),
        ]);

        // Assign the Super Admin role
        $superAdminRole = Role::where('name', 'Super Admin')->first();
        if ($superAdminRole) {
            $adminUser->assignRole($superAdminRole);
            $this->command->info('Super Admin user created successfully with email: admin@admin.com');
        } else {
            $this->command->error('Super Admin role not found. Please make sure RoleSeeder runs before UserSeeder.');
            return;
        }

        // Create 500 additional users
        $this->command->info('Creating 500 additional users...');

        // Get all available roles
        $roles = Role::all();

        if ($roles->isEmpty()) {
            $this->command->error('No roles found. Please make sure RoleSeeder runs before UserSeeder.');
            return;
        }

        // Define role distribution weights (higher number = more users with that role)
        $roleWeights = [
            'Subscriber' => 20,    // 20% subscribers
            'Service' => 60,       // 60% service providers
            'Supplier' => 10,      // 10% suppliers
            'Manufacturer' => 10,  // 10% manufacturers
        ];

        // Create role distribution array
        $roleDistribution = [];
        foreach ($roleWeights as $roleName => $weight) {
            $role = $roles->where('name', $roleName)->first();
            if ($role) {
                for ($i = 0; $i < $weight; $i++) {
                    $roleDistribution[] = $role;
                }
            }
        }

        // Create users in batches for better performance
        $batchSize = 50;
        $totalUsers = 500;
        $batches = ceil($totalUsers / $batchSize);

        for ($batch = 0; $batch < $batches; $batch++) {
            $usersInThisBatch = min($batchSize, $totalUsers - ($batch * $batchSize));

            // Create batch of users
            $users = User::factory($usersInThisBatch)->create();

            // Assign roles to each user
            foreach ($users as $user) {
                if (!empty($roleDistribution)) {
                    $randomRole = $roleDistribution[array_rand($roleDistribution)];
                    $user->assignRole($randomRole);
                }
            }

            $this->command->info("Created batch " . ($batch + 1) . " of $batches (Users: " . (($batch * $batchSize) + $usersInThisBatch) . "/$totalUsers)");
        }

        $this->command->info('Successfully created 500 additional users with assigned roles!');
        $this->command->info('Total users in system: ' . User::count());
    }
}
