# Bug Fix: field_options.forEach is not a function

## Problem

When rendering forms with checkbox, select, or radio fields, the SDK threw an error:
```
options.forEach is not a function
```

## Root Cause

The `field_options` field in the database was stored as a JSON string:
```
"{"tea":"Tea","coffee":"Coffee","cola":"Cola"}"
```

But the API Resource (`LeadFormFieldResource.php`) was returning it as-is without parsing it. According to the API documentation, `field_options` should be returned as either:
- `null` for fields without options
- An array like `["Tea", "Coffee", "Cola"]` for fields with options

The SDK expected an array but received a string, causing the `.forEach()` call to fail.

## Solution

### 1. Backend Fix (Root Cause) ✅

**File:** `app/Http/Resources/V1/Lead/LeadForm/LeadFormFieldResource.php`

Updated the resource to properly decode JSON strings and convert objects to arrays:

```php
// Parse field_options if it's a JSON string
$fieldOptions = $this->pivot->field_options ?? null;
if (is_string($fieldOptions)) {
    $decoded = json_decode($fieldOptions, true);
    // If it's an object like {"tea":"Tea"}, convert to array ["Tea", "Coffee"]
    $fieldOptions = $decoded ? array_values($decoded) : null;
}
```

**Result:** API now returns `field_options` as proper arrays conforming to the API documentation.

### 2. SDK Defensive Code (Safety Net) ✅

**File:** `public/js/sdk/embedForms.js`

Added defensive parsing in the `renderField()` method to handle various formats gracefully:

```javascript
// Safely parse field_options - handle string, object, or array
let options = config.field_options || [];
if (typeof options === 'string') {
    try {
        const parsed = JSON.parse(options);
        options = Array.isArray(parsed) ? parsed : Object.values(parsed);
    } catch (e) {
        this.log(`Warning: Could not parse field_options for field ${field.id}`);
        options = [];
    }
} else if (typeof options === 'object' && !Array.isArray(options) && options !== null) {
    options = Object.values(options);
}

// Ensure it's always an array
if (!Array.isArray(options)) {
    options = [];
}
```

**Result:** SDK now handles edge cases and won't crash even if the API returns unexpected formats.

## Testing

### Before Fix:
```html
<div class="esp-message esp-message-error">
    options.forEach is not a function
</div>
```

### After Fix:
Form renders correctly with all checkbox/select/radio options displayed properly.

### Test Case:
- Form UUID: `289c3d6b-a15e-4512-95d2-d4dcce0e7328`
- Field: "Checklist" (checkbox type)
- Options: Tea, Coffee, Cola

## Files Changed

1. ✅ `app/Http/Resources/V1/Lead/LeadForm/LeadFormFieldResource.php` - API fix
2. ✅ `public/js/sdk/embedForms.js` - SDK defensive code

## Impact

- **All checkbox fields** will now render correctly
- **All select/dropdown fields** will now render correctly
- **All radio button fields** will now render correctly
- **Backward compatible** with both old and new API responses

## Cache Clear Required

After applying this fix, clear Laravel caches:
```bash
php artisan route:clear
php artisan config:clear
```

## Status

✅ **FIXED** - Both backend and frontend updated to handle field_options correctly.
