# Feature: Custom Submit Button Configuration

## Overview

The SDK now supports using a `submit_button` field type from the form configuration to customize the submit button text and help text, instead of using a hardcoded "Submit" button.

## Feature Details

### How It Works

1. **SDK scans for submit_button field**: When rendering a form, the SDK looks for a field with `field_type === 'submit_button'`
2. **Extracts configuration**: Uses the field's `config` object to get:
   - `field_placeholder` → Submit button text (e.g., "Send Request")
   - `help_text` → Optional help text displayed below the button (e.g., "Click to send your request.")
3. **Ignores field_order**: Submit button is always placed at the end of the form, regardless of its `field_order` value
4. **Falls back to default**: If no `submit_button` field exists, uses "Submit" as default text

### Field Configuration Example

```json
{
    "id": 5,
    "uuid": "5dc45be6-a72c-4274-9b48-62555422bd68",
    "field_name": "Submit",
    "field_id": "submit",
    "field_type": "submit_button",
    "data_type": "submit_button",
    "is_default": 1,
    "config": {
        "field_placeholder": "Send Request",
        "toggle_yes_text": null,
        "toggle_no_text": null,
        "field_columns": 1,
        "help_text": "Click to send your request.",
        "field_options": null,
        "is_required": 1,
        "is_visible": 1,
        "field_order": 5
    }
}
```

### Rendered Output

**With submit_button field:**
```html
<div class="esp-form-actions">
    <button type="submit" class="esp-form-submit">Send Request</button>
    <small class="esp-submit-help">Click to send your request.</small>
</div>
```

**Without submit_button field (fallback):**
```html
<div class="esp-form-actions">
    <button type="submit" class="esp-form-submit">Submit</button>
</div>
```

## Implementation Details

### 1. Form Rendering Logic (`renderForm` method)

**Changes:**
- Searches for `submit_button` field in the fields array
- Extracts `field_placeholder` for button text and `help_text` for help message
- Filters out `submit_button` from regular field rendering
- Places submit button at the end with custom configuration
- Stores original button text in `formInstance.submitButtonText` for later use

**Code:**
```javascript
// Find submit button field configuration (if exists)
const submitButtonField = fields.find(f =>
    f.field_type === 'submit_button' && f.config?.is_visible !== false
);

// Get submit button configuration
const submitButtonText = submitButtonField?.config?.field_placeholder || 'Submit';
const submitButtonHelpText = submitButtonField?.config?.help_text || '';

// Exclude submit_button from regular field rendering
const sortedFields = [...fields]
    .filter(f => f.field_type !== 'submit_button')
    .sort((a, b) => ...);
```

### 2. Submit Handler (`handleSubmit` method)

**Changes:**
- Uses stored `formInstance.submitButtonText` instead of hardcoded "Submit"
- Restores original button text after submission completes

**Code:**
```javascript
const originalButtonText = formInstance.submitButtonText || 'Submit';
// ... submission logic ...
submitButton.textContent = originalButtonText; // Restore original text
```

### 3. CSS Styling (`embedForms.css`)

**Changes:**
- Updated `.esp-form-actions` to use flexbox column layout
- Added `.esp-submit-help` class for help text styling

**Code:**
```css
.esp-form-actions {
    display: flex;
    flex-direction: column;
    align-items: flex-start;
    gap: 0.5rem;
}

.esp-submit-help {
    display: block;
    font-size: 0.875rem;
    color: var(--esp-color-text-light);
}
```

## Files Modified

1. ✅ `public/js/sdk/embedForms.js:195-264` - Updated `renderForm()` method
2. ✅ `public/js/sdk/embedForms.js:589-634` - Updated `handleSubmit()` method
3. ✅ `public/js/sdk/embedForms.css:334-363` - Added styling for help text

## Benefits

✅ **Customizable button text**: Admins can set context-specific button text (e.g., "Get Quote", "Send Inquiry", "Request Info")
✅ **Additional guidance**: Help text provides extra instructions to users
✅ **Consistent placement**: Submit button always at the end, regardless of field order
✅ **Backward compatible**: Falls back to "Submit" if no custom button field is configured
✅ **Dynamic restoration**: Button text correctly restores after submission

## Testing

### Test Case 1: Form with Custom Submit Button
- Form has `submit_button` field with `field_placeholder: "Send Request"`
- **Expected**: Button displays "Send Request" instead of "Submit"

### Test Case 2: Form with Help Text
- Form has `submit_button` field with `help_text: "Click to send your request."`
- **Expected**: Help text appears below the button in gray color

### Test Case 3: Form without Submit Button Field
- Form does not include a `submit_button` field
- **Expected**: Button displays default "Submit" text

### Test Case 4: Submit Button Visibility
- Form has `submit_button` field with `is_visible: false`
- **Expected**: Button displays default "Submit" text (custom config ignored)

### Test Case 5: Submission States
- Click submit button with custom text "Send Request"
- **Expected**: Button changes to "Submitting..." during submission, then restores to "Send Request" after completion

## Status

✅ **IMPLEMENTED** - Custom submit button configuration is now fully functional.

## Example Usage in Admin Panel

To customize the submit button text in your lead forms:

1. Add or edit a field in your Lead Form
2. Set `field_type` to `submit_button`
3. Set `field_placeholder` to your desired button text (e.g., "Send Request")
4. Optionally add `help_text` for additional instructions
5. The button will automatically use this configuration when rendered

---

**Version:** 1.1.0
**Date:** 2025-01-12
**Status:** ✅ Complete
