# ESP Lead Form SDK Setup Guide

## Quick Setup (3 Steps)

### Step 1: Access the Test Page

Open this URL in your browser:
```
http://esp_directory.test/js/sdk/test.html
```

This diagnostic page will help you test your API connection and credentials.

### Step 2: Get Your API Credentials

#### Option A: Via Admin Panel (Recommended)

1. **Login to Admin Panel**
   - Go to: `http://esp_directory.test/espadmin`
   - Login with your admin credentials

2. **Create API Client**
   - Navigate to: **API Clients** in the sidebar
   - Click **"New API Client"**
   - Fill in the details:
     - **Name**: "WordPress Site" (or your site name)
     - **Type**: Select "Third Party"
     - **Is Active**: Check this box
   - Click **Save**
   - **Copy the Bearer Token** that appears (you'll need this!)

3. **Get Form UUID**
   - Navigate to: **Lead Forms** in the sidebar
   - Find an active form (or create a new one)
   - Click on the form to view details
   - **Copy the UUID** from the URL or form details

#### Option B: Via Artisan Commands (Quick Test)

Run these commands in your terminal:

```bash
cd /Users/shahnawazjahania/Herd/esp_directory

# Create a test user (if you don't have one)
php artisan tinker
```

Then in Tinker:
```php
// Create or get a user
$user = \App\Models\User\User::first() ?? \App\Models\User\User::factory()->create([
    'name' => 'Test User',
    'email' => 'test@example.com',
    'password' => bcrypt('password')
]);

// Create an API token
$token = $user->createToken('Test API Token')->plainTextToken;
echo "Your API Token: " . $token . "\n";

// Create a test lead form (if you don't have one)
$form = \App\Models\Lead\LeadForm::create([
    'title' => 'Test Contact Form',
    'description' => 'A test form for SDK testing',
    'is_active' => true,
    'enable_captcha' => false,
    'match_criteria' => 'category',
    'reception_mode' => 'auto',
    'submit_message' => 'Thank you for your submission!',
    'created_by' => $user->id
]);

// Attach some default fields
$nameField = \App\Models\Lead\LeadFormFields::where('field_name', 'first_name')->first();
$emailField = \App\Models\Lead\LeadFormFields::where('field_name', 'email')->first();
$messageField = \App\Models\Lead\LeadFormFields::where('field_name', 'message')->first();

if ($nameField) {
    $form->formFields()->attach($nameField->id, [
        'field_placeholder' => 'Enter your name',
        'is_required' => true,
        'is_visible' => true,
        'field_order' => 1,
        'field_columns' => 12
    ]);
}

if ($emailField) {
    $form->formFields()->attach($emailField->id, [
        'field_placeholder' => 'Enter your email',
        'is_required' => true,
        'is_visible' => true,
        'field_order' => 2,
        'field_columns' => 12
    ]);
}

if ($messageField) {
    $form->formFields()->attach($messageField->id, [
        'field_placeholder' => 'Enter your message',
        'is_required' => false,
        'is_visible' => true,
        'field_order' => 3,
        'field_columns' => 12
    ]);
}

echo "Form UUID: " . $form->uuid . "\n";
```

### Step 3: Test the SDK

1. **Open Test Page**
   ```
   http://esp_directory.test/js/sdk/test.html
   ```

2. **Enter Your Credentials**
   - API Base URL: `http://esp_directory.test`
   - API Bearer Token: (paste the token from Step 2)
   - Lead Form UUID: (paste the form UUID from Step 2)

3. **Run Tests**
   - Click "🚀 Run All Tests"
   - All three tests should pass ✓

4. **Test the Demo**
   - Open: `http://esp_directory.test/js/sdk/demo.html`
   - Update only two attributes in the HTML:
     - `data-form-id` → your form UUID
     - `data-api-token` → your bearer token
   - **No need to set `data-api-url`** - SDK auto-detects it!
   - Reload the page - your form should appear!

---

## Common Issues & Solutions

### Issue: "NetworkError when attempting to fetch resource"

**Causes:**
- Using placeholder values (`your-form-uuid-here`, `your-api-token-here`)
- Invalid API token
- Form doesn't exist or is inactive
- CORS issues

**Solutions:**
1. Use the test page (`test.html`) to diagnose the exact issue
2. Verify your API token is correct
3. Verify your form UUID is correct
4. Check that the form is active in admin panel
5. Check browser console for detailed error messages

### Issue: "Authentication failed"

**Solutions:**
1. Generate a new API token
2. Make sure you copied the entire token (they're long!)
3. Check that the API client is active

### Issue: "Form not found or is inactive"

**Solutions:**
1. Verify the form UUID is correct
2. Check that the form's "Is Active" checkbox is checked
3. View the form in admin panel to confirm it exists

### Issue: Form renders but submit fails

**Solutions:**
1. Check that the form has at least one field attached
2. Verify the field configurations
3. Check browser console for validation errors
4. Ensure the API token has proper permissions

---

## Integration Examples

### WordPress Shortcode

After testing, add this to your WordPress `functions.php`:

```php
function esp_lead_form_shortcode($atts) {
    $atts = shortcode_atts(array(
        'id' => '',
        'token' => '',
    ), $atts);

    return sprintf(
        '<div class="esp-lead-form" data-form-id="%s" data-api-token="%s"></div>',
        esc_attr($atts['id']),
        esc_attr($atts['token'])
    );
}
add_shortcode('esp_form', 'esp_lead_form_shortcode');
```

Use in posts/pages:
```
[esp_form id="your-form-uuid" token="your-token"]
```

**✨ SDK Auto-Detection:** The SDK automatically detects its API URL from the script source!

### Plain HTML (Simple - Recommended)

```html
<!DOCTYPE html>
<html>
<head>
    <link rel="stylesheet" href="http://esp_directory.test/js/sdk/embedForms.css">
    <script src="http://esp_directory.test/js/sdk/embedForms.js"></script>
</head>
<body>
    <!-- SDK auto-detects API URL - no data-api-url needed! -->
    <div class="esp-lead-form"
         data-form-id="your-form-uuid"
         data-api-token="your-token">
    </div>
</body>
</html>
```

### Plain HTML (With Override - Advanced)

Only use `data-api-url` if you need to point to a different API server:

```html
<!DOCTYPE html>
<html>
<head>
    <link rel="stylesheet" href="http://esp_directory.test/js/sdk/embedForms.css">
    <script src="http://esp_directory.test/js/sdk/embedForms.js"></script>
</head>
<body>
    <!-- Override auto-detected URL with custom API endpoint -->
    <div class="esp-lead-form"
         data-form-id="your-form-uuid"
         data-api-token="your-token"
         data-api-url="https://api-staging.example.com">
    </div>
</body>
</html>
```

---

## Next Steps

1. ✅ Complete the setup steps above
2. ✅ Test with the diagnostic page
3. ✅ View your form in the demo page
4. 📝 Customize the styling (see [javascript-sdk.md](../../../docs/api/javascript-sdk.md))
5. 🚀 Integrate into your WordPress site

---

## Need Help?

- **Test Page**: `http://esp_directory.test/js/sdk/test.html`
- **Demo Page**: `http://esp_directory.test/js/sdk/demo.html`
- **Full Documentation**: `/docs/api/javascript-sdk.md`
- **API Documentation**: `/docs/api/crm-endpoints.md`
