/**
 * ESP Lead Form Embed SDK
 *
 * A lightweight JavaScript SDK for embedding ESP lead forms on external websites.
 * Requires Laravel Sanctum Bearer token for API authentication.
 *
 * Features:
 * - Automatic form rendering from API
 * - Form validation and submission
 * - Visit tracking with unique visitor identification
 * - Session-based duplicate visit prevention
 * - Supports data-enable-tracking attribute for opt-in analytics
 *
 * @version 1.1.0
 * @author ESP Directory
 * @license MIT
 */
(function (window, document) {
    'use strict';

    /**
     * Main SDK Class
     */
    class ESPLeadFormSDK {
        constructor() {
            this.forms = [];
            this.config = {
                apiBaseUrl: this.detectApiBaseUrl(),
                apiVersion: 'v1',
                debug: false
            };
            // File upload state management
            this.fileUploads = {}; // Stores selected files per field
            this.fileValidationSettings = {}; // Store validation settings per field from API

            // Default file validation settings (used as fallback)
            this.defaultFileValidation = {
                max_files: 5,
                max_size_kb: 10240, // 10 MB
                allowed_types: ['image'],
                accept: 'image/*',
                mime_types: ['image/jpeg', 'image/png', 'image/gif', 'image/webp'],
                extensions: ['jpg', 'jpeg', 'png', 'gif', 'webp']
            };
            this.log(`SDK initialized with API base URL: ${this.config.apiBaseUrl}`);
            this.init();
        }

        /**
         * Detect API base URL from script source
         * Auto-detects the domain where this SDK is hosted
         * @returns {string} The detected API base URL
         */
        detectApiBaseUrl() {
            // Try to get URL from the script tag that loaded this SDK
            const scripts = document.getElementsByTagName('script');

            for (let script of scripts) {
                if (script.src && script.src.includes('embedForms.js')) {
                    try {
                        const url = new URL(script.src);
                        const detectedUrl = url.origin;
                        this.log(`Auto-detected API URL from script source: ${detectedUrl}`);
                        return detectedUrl;
                    } catch (e) {
                        console.warn('[ESP Lead Form SDK] Failed to parse script URL:', e);
                    }
                }
            }

            // Fallback to current page origin (useful for same-domain embedding)
            const fallbackUrl = window.location.origin;
            this.log(`Using fallback API URL (current page origin): ${fallbackUrl}`);
            return fallbackUrl;
        }

        /**
         * Initialize the SDK
         */
        init() {
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', () => this.discoverForms());
            } else {
                this.discoverForms();
            }
        }

        /**
         * Discover all form containers on the page
         */
        discoverForms() {
            const containers = document.querySelectorAll('.esp-lead-form');

            if (containers.length === 0) {
                this.log('No ESP lead form containers found on page');
                return;
            }

            this.log(`Found ${containers.length} form container(s)`);

            containers.forEach(container => {
                this.initializeForm(container);
            });
        }

        /**
         * Initialize a single form
         * @param {HTMLElement} container - The form container element
         */
        async initializeForm(container) {
            const formId = container.getAttribute('data-form-id');
            const customApiUrl = container.getAttribute('data-api-url');
            const apiUrl = customApiUrl || this.config.apiBaseUrl;
            const apiToken = container.getAttribute('data-api-token');
            const enableTracking = container.getAttribute('data-enable-tracking') !== 'false';

            // Log which URL is being used
            if (customApiUrl) {
                this.log(`Using custom API URL from data-api-url: ${apiUrl}`);
            } else {
                this.log(`Using auto-detected API URL: ${apiUrl}`);
            }

            // Validation
            if (!formId) {
                this.showError(container, 'Missing data-form-id attribute');
                return;
            }

            if (!apiToken) {
                this.showError(container, 'Missing data-api-token attribute');
                return;
            }

            // Store form instance
            const formInstance = {
                id: formId,
                container: container,
                apiUrl: apiUrl,
                apiToken: apiToken,
                enableTracking: enableTracking,
                formData: null
            };

            this.forms.push(formInstance);

            // Show loading state
            container.innerHTML = '<div class="esp-form-loading">Loading form...</div>';

            try {
                // Fetch form configuration
                const formData = await this.fetchForm(apiUrl, apiToken, formId);
                formInstance.formData = formData;

                // Render the form
                this.renderForm(formInstance);

                // Track visit
                if (enableTracking) {
                    this.trackVisit(formInstance);
                }

            } catch (error) {
                this.showError(container, error.message);
            }
        }

        /**
         * Fetch form configuration from API
         * @param {string} apiUrl - Base API URL
         * @param {string} apiToken - Sanctum Bearer token
         * @param {string} formId - Form UUID
         * @returns {Promise<Object>} Form data
         */
        async fetchForm(apiUrl, apiToken, formId) {
            const url = `${apiUrl}/api/${this.config.apiVersion}/lead-forms/${formId}`;

            this.log(`Fetching form: ${url}`);

            const response = await fetch(url, {
                method: 'GET',
                headers: {
                    'Authorization': `Bearer ${apiToken}`,
                    'Accept': 'application/json',
                    'Content-Type': 'application/json'
                }
            });

            if (!response.ok) {
                if (response.status === 401) {
                    throw new Error('Authentication failed. Please check your API token.');
                }
                if (response.status === 404) {
                    throw new Error('Form not found or is inactive.');
                }
                if (response.status === 429) {
                    throw new Error('Rate limit exceeded. Please try again later.');
                }
                throw new Error(`Failed to load form: ${response.statusText}`);
            }

            const result = await response.json();

            if (!result.success || !result.data) {
                throw new Error('Invalid API response format');
            }

            return result.data;
        }

        /**
         * Render the form HTML
         * @param {Object} formInstance - Form instance object
         */
        renderForm(formInstance) {
            const { container, formData } = formInstance;

            // Check if this is a wizard form
            if (formData.form_layout === 'wizard' && formData.steps && formData.steps.length > 0) {
                this.renderWizardForm(formInstance);
                return;
            }

            // Render simple form (original behavior)
            this.renderSimpleForm(formInstance);
        }

        /**
         * Render a simple single-page form
         * @param {Object} formInstance - Form instance object
         */
        renderSimpleForm(formInstance) {
            const { container, formData } = formInstance;
            const fields = formData.fields || [];

            // Find submit button field configuration (if exists)
            const submitButtonField = fields.find(f => f.field_type === 'submit_button' && f.config?.is_visible !== false);

            // Get submit button configuration
            const submitButtonText = submitButtonField?.config?.field_placeholder || 'Submit';
            const submitButtonHelpText = submitButtonField?.config?.help_text || '';

            // Store original button text for later use
            formInstance.submitButtonText = submitButtonText;

            // Sort fields by field_order (excluding submit_button - it goes at the end)
            const sortedFields = [...fields]
                .filter(f => f.field_type !== 'submit_button')
                .sort((a, b) => {
                    const orderA = a.config?.field_order ?? 0;
                    const orderB = b.config?.field_order ?? 0;
                    return orderA - orderB;
                });

            // Process description with shortcodes
            const processedDescription = formData.description ? this.processShortcodes(formData.description) : '';

            // Create form HTML
            let formHtml = `
                <div class="esp-form-wrapper">
                    ${formData.title ? `<h2 class="esp-form-title">${this.escapeHtml(formData.title)}</h2>` : ''}
                    ${processedDescription ? `<p class="esp-form-description">${processedDescription}</p>` : ''}
                    <form class="esp-form" data-form-instance-id="${this.forms.length - 1}">
                        <input type="hidden" name="form_id" value="${formInstance.id}" />
                        ${this.renderAddressHiddenFields()}
                        <div class="esp-form-fields">
            `;

            // Render each field (excluding submit_button)
            sortedFields.forEach(field => {
                if (field.config?.is_visible !== false) {
                    formHtml += this.renderField(field);
                }
            });

            // Close fields container and add messages
            formHtml += `
                        </div>
                        <div class="esp-form-messages"></div>
            `;

            // Add submit button with help text if available
            formHtml += `
                        <div class="esp-form-actions">
                            <button type="submit" class="esp-form-submit">${this.escapeHtml(submitButtonText)}</button>
            `;

            // Add help text below button if provided
            if (submitButtonHelpText) {
                formHtml += `
                            <small class="esp-submit-help">${this.escapeHtml(submitButtonHelpText)}</small>
                `;
            }

            formHtml += `
                        </div>
                    </form>
                </div>
            `;

            container.innerHTML = formHtml;

            // Attach event listeners
            const form = container.querySelector('.esp-form');
            form.addEventListener('submit', (e) => this.handleSubmit(e, formInstance));

            // Setup file upload listeners
            this.setupFileUploadListeners(container);

            // Setup conditional fields (dependent fields logic)
            this.setupConditionalFields(container, sortedFields);
        }

        /**
         * Render a wizard-style multi-step form
         * @param {Object} formInstance - Form instance object
         */
        renderWizardForm(formInstance) {
            const { container, formData } = formInstance;

            // Filter active steps only
            const activeSteps = formData.steps.filter(step => step.is_active !== false);

            if (activeSteps.length === 0) {
                this.showError(container, 'No active steps found in wizard form');
                return;
            }

            // Initialize wizard state
            formInstance.wizard = {
                isWizard: true,
                activeSteps: activeSteps,
                currentStepIndex: 0,
                visitedSteps: new Set([0]),
                totalSteps: activeSteps.length,
                collectedData: {} // Store data from all steps
            };

            // Find submit button configuration
            const allFields = activeSteps.flatMap(step => step.fields || []);
            const submitButtonField = allFields.find(f => f.field_type === 'submit_button' && f.config?.is_visible !== false);
            formInstance.submitButtonText = submitButtonField?.config?.field_placeholder || 'Submit';

            // Render wizard HTML
            this.renderWizardHTML(formInstance);

            // Attach event listeners
            const form = container.querySelector('.esp-form');
            const prevButton = container.querySelector('.esp-wizard-prev');
            const nextButton = container.querySelector('.esp-wizard-next');

            if (prevButton) {
                prevButton.addEventListener('click', () => this.handleWizardPrevious(formInstance));
            }

            if (nextButton) {
                nextButton.addEventListener('click', () => this.handleWizardNext(formInstance));
            }

            form.addEventListener('submit', (e) => this.handleSubmit(e, formInstance));

            // Setup file upload listeners
            this.setupFileUploadListeners(container);

            // Setup conditional fields for current step
            const currentStepFields = formInstance.wizard.activeSteps[formInstance.wizard.currentStepIndex].fields || [];
            this.setupConditionalFields(container, currentStepFields);
        }

        /**
         * Render wizard HTML structure
         * @param {Object} formInstance - Form instance object
         */
        renderWizardHTML(formInstance) {
            const { container, formData, wizard } = formInstance;
            const currentStep = wizard.activeSteps[wizard.currentStepIndex];

            // Process shortcodes in description
            const processedDescription = formData.description ? this.processShortcodes(formData.description) : '';

            let formHtml = `
                <div class="esp-form-wrapper esp-wizard-form">
                    ${formData.title ? `<h2 class="esp-form-title">${this.escapeHtml(formData.title)}</h2>` : ''}
                    ${processedDescription ? `<p class="esp-form-description">${processedDescription}</p>` : ''}

                    ${this.renderProgressIndicator(formInstance)}

                    <form class="esp-form esp-wizard" data-form-instance-id="${this.forms.length - 1}">
                        <input type="hidden" name="form_id" value="${formInstance.id}" />
                        ${this.renderAddressHiddenFields()}

                        ${this.renderHiddenFieldsForCollectedData(wizard)}

                        <div class="esp-wizard-step active" data-step-index="${wizard.currentStepIndex}">
                            ${this.renderStepHeader(currentStep)}
                            <div class="esp-step-fields">
                                ${this.renderStepFields(currentStep, wizard)}
                            </div>
                        </div>

                        <div class="esp-form-messages"></div>

                        ${this.renderWizardNavigation(formInstance)}
                    </form>
                </div>
            `;

            container.innerHTML = formHtml;
        }

        /**
         * Render hidden fields for previously collected data
         * @param {Object} wizard - Wizard state object
         * @returns {string} Hidden fields HTML
         */
        renderHiddenFieldsForCollectedData(wizard) {
            let hiddenFieldsHtml = '';

            for (const [fieldName, value] of Object.entries(wizard.collectedData)) {
                if (Array.isArray(value)) {
                    // Handle array values (checkboxes)
                    // Check if fieldName already ends with []
                    const nameWithBrackets = fieldName.endsWith('[]') ? fieldName : `${fieldName}[]`;
                    value.forEach(v => {
                        hiddenFieldsHtml += `<input type="hidden" name="${nameWithBrackets}" value="${this.escapeHtml(v)}" />`;
                    });
                } else {
                    hiddenFieldsHtml += `<input type="hidden" name="${fieldName}" value="${this.escapeHtml(value)}" />`;
                }
            }

            return hiddenFieldsHtml;
        }

        /**
         * Get URL parameter value
         * @param {string} param - Parameter name
         * @returns {string|null} Parameter value or null if not found
         */
        getUrlParameter(param) {
            const urlParams = new URLSearchParams(window.location.search);
            return urlParams.get(param);
        }

        /**
         * Process shortcodes in text (e.g., {{address}})
         * @param {string} text - Text containing shortcodes
         * @returns {string} Text with shortcodes replaced
         */
        processShortcodes(text) {
            if (!text) {
                return text;
            }

            // Replace {{address}} shortcode
            const address = this.getUrlParameter('address');
            if (address && address.trim() !== '') {
                text = text.replace(/\{\{address\}\}/g, this.escapeHtml(address));
            } else {
                // Remove the shortcode if address is not available
                text = text.replace(/\{\{address\}\}/g, '');
            }

            return text;
        }

        /**
         * Render address hidden fields from URL parameters
         * @returns {string} Hidden fields HTML for address data
         */
        renderAddressHiddenFields() {
            const addressParams = ['address', 'lat', 'lng', 'city', 'state', 'state_code', 'country', 'country_code'];
            let hiddenFieldsHtml = '';

            addressParams.forEach(param => {
                const value = this.getUrlParameter(param);

                // Only add hidden field if value exists and is not empty
                if (value && value.trim() !== '') {
                    hiddenFieldsHtml += `<input type="hidden" name="${param}" value="${this.escapeHtml(value)}" />`;
                }
            });

            return hiddenFieldsHtml;
        }

        /**
         * Render progress indicator - Horizontal step list matching host
         * @param {Object} formInstance - Form instance object
         * @returns {string} Progress HTML
         */
        renderProgressIndicator(formInstance) {
            const { wizard } = formInstance;
            let stepsHtml = '';

            wizard.activeSteps.forEach((step, index) => {
                const stepNumber = index + 1;
                const isActive = index === wizard.currentStepIndex;
                const isCompleted = wizard.visitedSteps.has(index) && index < wizard.currentStepIndex;
                const statusClass = isActive ? 'active' : (isCompleted ? 'completed' : '');

                stepsHtml += `
                    <div class="esp-progress-step ${statusClass}">
                        <span class="esp-step-number-box">
                            <span class="esp-step-number">${stepNumber}</span>
                            <span class="esp-step-check">
                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" width="12" height="12">
                                    <path fill-rule="evenodd" d="M16.704 4.153a.75.75 0 01.143 1.052l-8 10.5a.75.75 0 01-1.127.075l-4.5-4.5a.75.75 0 011.06-1.06l3.894 3.893 7.48-9.817a.75.75 0 011.05-.143z" clip-rule="evenodd" />
                                </svg>
                            </span>
                        </span>
                        <p class="esp-step-label">${this.escapeHtml(step.name || step.title || `Step ${stepNumber}`)}</p>
                    </div>
                `;

                // Add separator arrow if not the last step
                if (index < wizard.activeSteps.length - 1) {
                    stepsHtml += `
                        <span class="esp-step-separator">
                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                <path fill-rule="evenodd" d="M7.21 14.77a.75.75 0 01.02-1.06L11.168 10 7.23 6.29a.75.75 0 111.04-1.08l4.5 4.25a.75.75 0 010 1.08l-4.5 4.25a.75.75 0 01-1.06-.02z" clip-rule="evenodd" />
                            </svg>
                        </span>
                    `;
                }
            });

            return `
                <div class="esp-wizard-progress">
                    <div class="esp-progress-steps">
                        ${stepsHtml}
                    </div>
                </div>
            `;
        }

        /**
         * Render step header (title, subtitle, image)
         * @param {Object} step - Step object
         * @returns {string} Step header HTML
         */
        renderStepHeader(step) {
            let headerHtml = '<div class="esp-step-header">';

            // Icon container (matching host design)
            headerHtml += '<span class="esp-step-icon">';
            if (step.image) {
                headerHtml += `<img src="${this.escapeHtml(step.image)}" alt="${this.escapeHtml(step.title || 'Step')}" />`;
            } else {
                // Default icon SVG if no image provided
                headerHtml += `<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M15.75 6a3.75 3.75 0 11-7.5 0 3.75 3.75 0 017.5 0zM4.501 20.118a7.5 7.5 0 0114.998 0A17.933 17.933 0 0112 21.75c-2.676 0-5.216-.584-7.499-1.632z" />
                </svg>`;
            }
            headerHtml += '</span>';

            // Content wrapper
            headerHtml += '<div class="esp-step-content">';

            if (step.title) {
                headerHtml += `<h5 class="esp-step-title">${this.escapeHtml(step.title)}</h5>`;
            }

            if (step.subtitle) {
                headerHtml += `<p class="esp-step-subtitle">${this.escapeHtml(step.subtitle)}</p>`;
            }

            headerHtml += '</div>'; // Close esp-step-content
            headerHtml += '</div>'; // Close esp-step-header
            return headerHtml;
        }

        /**
         * Render fields for a step
         * @param {Object} step - Step object
         * @param {Object} wizard - Wizard state object (optional, for pre-filling)
         * @returns {string} Fields HTML
         */
        renderStepFields(step, wizard = null) {
            const fields = step.fields || [];
            let fieldsHtml = '';

            // Sort and filter fields
            const sortedFields = [...fields]
                .filter(f => f.field_type !== 'submit_button' && f.config?.is_visible !== false)
                .sort((a, b) => {
                    const orderA = a.config?.field_order ?? 0;
                    const orderB = b.config?.field_order ?? 0;
                    return orderA - orderB;
                });

            sortedFields.forEach(field => {
                // Get pre-filled value if going back to a previous step
                const prefilledValue = wizard && wizard.collectedData ? wizard.collectedData[field.field_id] : null;
                fieldsHtml += this.renderField(field, prefilledValue);
            });

            return fieldsHtml;
        }

        /**
         * Render wizard navigation buttons
         * @param {Object} formInstance - Form instance object
         * @returns {string} Navigation HTML
         */
        renderWizardNavigation(formInstance) {
            const { wizard } = formInstance;
            const isFirstStep = wizard.currentStepIndex === 0;
            const isLastStep = wizard.currentStepIndex === wizard.totalSteps - 1;

            return `
                <div class="esp-wizard-navigation">
                    <button type="button" class="esp-wizard-prev" style="display: ${isFirstStep ? 'none' : 'inline-block'};">
                        Previous
                    </button>
                    <button type="button" class="esp-wizard-next" style="display: ${isLastStep ? 'none' : 'inline-block'};">
                        Next
                    </button>
                    <button type="submit" class="esp-form-submit" style="display: ${isLastStep ? 'inline-block' : 'none'};">
                        ${this.escapeHtml(formInstance.submitButtonText || 'Submit')}
                    </button>
                </div>
            `;
        }

        /**
         * Handle wizard Next button
         * @param {Object} formInstance - Form instance object
         */
        handleWizardNext(formInstance) {
            const { container, wizard } = formInstance;
            const form = container.querySelector('.esp-form');
            const messagesContainer = form.querySelector('.esp-form-messages');

            // Clear previous messages
            messagesContainer.innerHTML = '';
            this.clearFieldErrors(form);

            // Validate current step
            const currentStep = wizard.activeSteps[wizard.currentStepIndex];
            const validation = this.validateCurrentStep(form, currentStep);

            if (!validation.valid) {
                this.showValidationErrors(form, validation.errors);
                return;
            }

            // Collect and store data from current step
            this.collectStepData(form, currentStep, wizard);

            // Move to next step
            wizard.currentStepIndex++;
            wizard.visitedSteps.add(wizard.currentStepIndex);

            // Re-render wizard
            this.renderWizardHTML(formInstance);

            // Re-attach event listeners
            const newForm = container.querySelector('.esp-form');
            const prevButton = container.querySelector('.esp-wizard-prev');
            const nextButton = container.querySelector('.esp-wizard-next');

            if (prevButton) {
                prevButton.addEventListener('click', () => this.handleWizardPrevious(formInstance));
            }

            if (nextButton) {
                nextButton.addEventListener('click', () => this.handleWizardNext(formInstance));
            }

            newForm.addEventListener('submit', (e) => this.handleSubmit(e, formInstance));

            // Setup file upload listeners
            this.setupFileUploadListeners(container);

            // Setup conditional fields for new step
            const newStepFields = formInstance.wizard.activeSteps[formInstance.wizard.currentStepIndex].fields || [];
            this.setupConditionalFields(container, newStepFields);

            // Scroll to top of form
            container.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }

        /**
         * Handle wizard Previous button
         * @param {Object} formInstance - Form instance object
         */
        handleWizardPrevious(formInstance) {
            const { container, wizard } = formInstance;

            // Move to previous step (no validation needed)
            wizard.currentStepIndex--;

            // Re-render wizard
            this.renderWizardHTML(formInstance);

            // Re-attach event listeners
            const form = container.querySelector('.esp-form');
            const prevButton = container.querySelector('.esp-wizard-prev');
            const nextButton = container.querySelector('.esp-wizard-next');

            if (prevButton) {
                prevButton.addEventListener('click', () => this.handleWizardPrevious(formInstance));
            }

            if (nextButton) {
                nextButton.addEventListener('click', () => this.handleWizardNext(formInstance));
            }

            form.addEventListener('submit', (e) => this.handleSubmit(e, formInstance));

            // Setup file upload listeners
            this.setupFileUploadListeners(container);

            // Setup conditional fields for current step
            const currentStepFields = wizard.activeSteps[wizard.currentStepIndex].fields || [];
            this.setupConditionalFields(container, currentStepFields);

            // Scroll to top of form
            container.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }

        /**
         * Collect data from current step and store it
         * @param {HTMLFormElement} form - Form element
         * @param {Object} step - Current step object
         * @param {Object} wizard - Wizard state object
         */
        collectStepData(form, step, wizard) {
            const formData = new FormData(form);
            const fields = step.fields || [];

            // Collect regular fields from step.fields
            fields.forEach(field => {
                // Skip non-input field types
                if (field.field_type === 'submit_button' || field.field_type === 'section_title' || field.field_type === 'hidden') {
                    return;
                }

                const fieldId = field.field_id;

                // Get value from form
                let value = formData.get(fieldId);

                // Handle array values (checkboxes)
                if (formData.getAll(fieldId + '[]').length > 0) {
                    value = formData.getAll(fieldId + '[]');
                }

                // Store the value if it exists
                if (value !== null && value !== '') {
                    wizard.collectedData[fieldId] = value;
                }
            });

            // Also collect dependent fields with nested bracket notation
            // These have names like: parent_field[parent_value][dependent_field]
            for (const [name, value] of formData.entries()) {
                // Check if this is a nested field name
                const nestedPattern = /^([^\[]+)\[([^\]]+)\]\[([^\]]+)\](\[\])?$/;
                const match = name.match(nestedPattern);

                if (match) {
                    // This is a dependent field with nested name
                    // Store it as-is with the full nested name as the key
                    if (!wizard.collectedData[name]) {
                        // Check if it's an array field (ends with [])
                        if (match[4] === '[]') {
                            // Collect all values for this array field
                            const allValues = formData.getAll(name);
                            if (allValues.length > 0) {
                                wizard.collectedData[name] = allValues;
                            }
                        } else {
                            // Single value field
                            if (value !== null && value !== '') {
                                wizard.collectedData[name] = value;
                            }
                        }
                    }
                }
            }

            this.log('Collected step data:', wizard.collectedData);
        }

        /**
         * Validate current step fields only
         * @param {HTMLFormElement} form - Form element
         * @param {Object} step - Current step object
         * @returns {Object} Validation result
         */
        validateCurrentStep(form, step) {
            const errors = {};
            const fields = step.fields || [];
            const formData = new FormData(form);

            // Standard field names that are at top level
            const standardFields = ['first_name', 'last_name', 'email', 'phone', 'message'];

            fields.forEach(field => {
                // Skip non-input field types
                if (field.field_type === 'submit_button' || field.field_type === 'section_title' || field.field_type === 'hidden') {
                    return;
                }

                const config = field.config || {};
                const fieldId = field.field_id;
                const fieldLabel = field.field_name || field.field_id;
                const isRequired = config.is_required || false;

                // Special handling for file uploads
                if (field.field_type === 'file') {
                    const fileInput = form.querySelector(`input[type="file"][data-field-id="${field.id}"]`);
                    const isFileRequired = fileInput && fileInput.getAttribute('data-required') === 'true';

                    if (isFileRequired) {
                        const uploadedFiles = this.fileUploads[field.id] || [];
                        if (uploadedFiles.length === 0) {
                            errors[field.id] = `${fieldLabel} is required`;
                        }
                    }
                    return;
                }

                // Get value from form
                let value = formData.get(fieldId);

                // Handle array values (checkboxes)
                if (formData.getAll(fieldId + '[]').length > 0) {
                    value = formData.getAll(fieldId + '[]');
                }

                // Required validation
                if (isRequired && (!value || value === '' || (Array.isArray(value) && value.length === 0))) {
                    errors[field.id] = `${fieldLabel} is required`;
                    return;
                }

                // Skip further validation if empty and not required
                if (!value || value === '' || (Array.isArray(value) && value.length === 0)) {
                    return;
                }

                // Type-specific validation
                const dataType = field.data_type;

                if (dataType === 'email') {
                    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                    if (!emailRegex.test(value)) {
                        errors[field.id] = 'Please enter a valid email address';
                    }
                }

                if (dataType === 'phone') {
                    const phoneRegex = /^[\d\s\-\+\(\)]+$/;
                    if (!phoneRegex.test(value)) {
                        errors[field.id] = 'Please enter a valid phone number';
                    }
                }

                if (dataType === 'number' || field.field_type === 'number') {
                    if (isNaN(value)) {
                        errors[field.id] = 'Please enter a valid number';
                    }
                }
            });

            return {
                valid: Object.keys(errors).length === 0,
                errors: errors
            };
        }

        /**
         * Render a single form field
         * @param {Object} field - Field configuration
         * @returns {string} Field HTML
         */
        renderField(field) {
            const config = field.config || {};
            const fieldType = field.field_type;
            const fieldId = field.field_id;  // Use field_id for HTML name attribute
            const fieldLabel = config.field_label || field.field_name || field.field_id;  // Use field_label, fallback to field_name, then field_id
            const isRequired = config.is_required || false;
            // File upload fields should always be full width
            const columns = fieldType === 'file' ? 12 : (config.field_columns || 12);
            const placeholder = config.field_placeholder || '';
            const helpText = config.help_text || '';

            // Safely parse field_options - handle string, object, or array
            // Options should now be in format: {"key": "Label", "key2": "Label2"}
            let options = config.field_options || {};
            if (typeof options === 'string') {
                try {
                    options = JSON.parse(options);
                } catch (e) {
                    this.log(`Warning: Could not parse field_options for field ${field.id}: ${e.message}`);
                    options = {};
                }
            }

            // Ensure it's an object (key-value pairs)
            if (typeof options !== 'object' || options === null) {
                options = {};
            }

            // Convert to object if it's an array (legacy support)
            if (Array.isArray(options)) {
                const optionsObj = {};
                options.forEach(opt => {
                    optionsObj[opt] = opt;
                });
                options = optionsObj;
            }

            // Don't render submit button or hidden fields in the grid
            if (fieldType === 'submit_button') {
                return '';
            }

            if (fieldType === 'hidden') {
                return `<input type="hidden" name="${fieldId}" data-field-id="${field.id}" />`;
            }

            // Section title
            if (fieldType === 'section_title') {
                const sectionTitle = config.field_placeholder || fieldLabel;
                const sectionSubtitle = config.help_text || '';

                return `
                    <div class="esp-form-section esp-col-12">
                        <h3 class="esp-section-title">${this.escapeHtml(sectionTitle)}</h3>
                        ${sectionSubtitle ? `<p class="esp-section-subtitle">${this.escapeHtml(sectionSubtitle)}</p>` : ''}
                    </div>
                `;
            }

            // Line separator
            if (fieldType === 'line_separator') {
                return `
                    <div class="esp-line-separator esp-col-12">
                        <hr class="esp-separator-line" />
                    </div>
                `;
            }

            let fieldHtml = `<div class="esp-form-field esp-col-${columns}" data-field-id="${field.id}">`;

            // Determine if we should show a label
            // Don't show label for: single checkbox (no options), toggle (they have inline labels),
            // or checkbox/radio groups with field_placeholder (they render their own header)
            const hasOptions = Object.keys(options).length > 0;
            const showLabel = !(fieldType === 'checkbox' && !hasOptions)
                && fieldType !== 'toggle'
                && !(fieldType === 'checkbox' && hasOptions && config.field_placeholder)
                && !(fieldType === 'radio' && hasOptions && config.field_placeholder);

            // Label
            if (showLabel) {
                fieldHtml += `
                    <label class="esp-field-label" for="esp-field-${field.id}">
                        ${this.escapeHtml(fieldLabel)}
                        ${isRequired ? '<span class="esp-required">*</span>' : ''}
                    </label>
                `;
            }

            // Field input based on type
            switch (fieldType) {
                case 'text':
                case 'email':
                case 'phone':
                case 'number':
                    fieldHtml += this.renderInputField(field, fieldType, fieldId, placeholder, isRequired);
                    break;

                case 'textarea':
                    fieldHtml += this.renderTextarea(field, fieldId, placeholder, isRequired);
                    break;

                case 'select':
                    fieldHtml += this.renderSelect(field, fieldId, options, isRequired);
                    break;

                case 'radio':
                    fieldHtml += this.renderRadio(field, fieldId, fieldLabel, options, isRequired, config);
                    break;

                case 'checkbox':
                    fieldHtml += this.renderCheckbox(field, fieldId, fieldLabel, options, isRequired, config);
                    break;

                case 'toggle':
                    fieldHtml += this.renderToggle(field, fieldId, fieldLabel, config, isRequired);
                    break;

                case 'file':
                    fieldHtml += this.renderFileInput(field, fieldId, isRequired);
                    break;

                case 'geolocation':
                    fieldHtml += this.renderGeolocation(field, fieldId, isRequired);
                    break;

                default:
                    fieldHtml += this.renderInputField(field, 'text', fieldId, placeholder, isRequired);
            }

            // Help text - Skip for checkbox/radio with field_placeholder (already shown in section header)
            const skipHelpText = (fieldType === 'checkbox' || fieldType === 'radio')
                && options.length > 0
                && config.field_placeholder;

            if (helpText && !skipHelpText) {
                fieldHtml += `<small class="esp-field-help">${this.escapeHtml(helpText)}</small>`;
            }

            // Error container
            fieldHtml += `<div class="esp-field-error" style="display:none;"></div>`;

            fieldHtml += `</div>`;

            return fieldHtml;
        }

        /**
         * Render input field (text, email, phone, number)
         */
        renderInputField(field, type, name, placeholder, required) {
            const inputType = type === 'phone' ? 'tel' : type;
            return `
                <input
                    type="${inputType}"
                    id="esp-field-${field.id}"
                    name="${name}"
                    class="esp-field-input"
                    data-field-id="${field.id}"
                    placeholder="${this.escapeHtml(placeholder)}"
                    ${required ? 'required' : ''}
                />
            `;
        }

        /**
         * Render textarea field
         */
        renderTextarea(field, name, placeholder, required) {
            return `
                <textarea
                    id="esp-field-${field.id}"
                    name="${name}"
                    class="esp-field-textarea"
                    data-field-id="${field.id}"
                    placeholder="${this.escapeHtml(placeholder)}"
                    rows="4"
                    ${required ? 'required' : ''}
                ></textarea>
            `;
        }

        /**
         * Render select dropdown
         * Options format: {"key": "Label", "key2": "Label2"}
         */
        renderSelect(field, name, options, required, skipDependents = false) {
            let html = `
                <select
                    id="esp-field-${field.id}"
                    name="${name}"
                    class="esp-field-select"
                    data-field-id="${field.id}"
                    ${required ? 'required' : ''}
                >
                    <option value="">Select an option</option>
            `;

            // Iterate over key-value pairs
            Object.entries(options).forEach(([value, label]) => {
                html += `<option value="${this.escapeHtml(value)}">${this.escapeHtml(label)}</option>`;
            });

            html += `</select>`;

            // Render all dependent fields after the select (only if not a dependent field itself)
            if (!skipDependents) {
                html += this.renderAllDependentFields(field);
            }

            return html;
        }

        /**
         * Render radio buttons
         * Options format: {"key": "Label", "key2": "Label2"}
         */
        renderRadio(field, name, label, options, required, config, skipDependents = false) {
            // Radio buttons with section header and grid layout
            const sectionTitle = config.field_label || '';
            const sectionSubtitle = config.field_placeholder || '';
            const dataColumns = config.data_columns || '1';

            let html = '';

            // Section header - only if label or placeholder exists
            if (sectionTitle || sectionSubtitle) {
                html += `
                    <div class="esp-radio-section-header">
                        ${sectionTitle ? `<label class="esp-field-label">
                            ${this.escapeHtml(sectionTitle)}
                            ${required ? '<span class="esp-required">*</span>' : ''}
                        </label>` : ''}
                        ${sectionSubtitle ? `<p class="esp-radio-section-subtitle">${this.escapeHtml(sectionSubtitle)}</p>` : ''}
                    </div>
                `;
            }

            // Radio buttons grid with container attributes on individual items
            const containerClass = config.container_class || '';
            const containerId = config.container_id || '';

            html += `<div class="esp-field-radio-group" data-columns="${dataColumns}">`;

            // Iterate over key-value pairs
            Object.entries(options).forEach(([value, optionLabel], index) => {
                const itemId = containerId ? ` id="${containerId}-${index}"` : '';
                const itemClass = containerClass ? ` ${containerClass}` : '';

                html += `
                    <div class="esp-radio-item${itemClass}"${itemId}>
                        <label class="esp-radio-label">
                            <input
                                type="radio"
                                name="${name}"
                                value="${this.escapeHtml(value)}"
                                data-field-id="${field.id}"
                                ${required && index === 0 ? 'required' : ''}
                            />
                            <span>${this.escapeHtml(optionLabel)}</span>
                        </label>
                    </div>
                `;
            });

            html += '</div>';

            // Render all dependent fields after the radio group (only if not a dependent field itself)
            if (!skipDependents) {
                html += this.renderAllDependentFields(field);
            }

            return html;
        }

        /**
         * Render checkboxes
         * Options format: {"key": "Label", "key2": "Label2"}
         */
        renderCheckbox(field, name, label, options, required, config, skipDependents = false) {
            const hasOptions = Object.keys(options).length > 0;

            if (!hasOptions) {
                // Single checkbox
                return `
                    <label class="esp-checkbox-label">
                        <input
                            type="checkbox"
                            name="${name}"
                            value="1"
                            data-field-id="${field.id}"
                            ${required ? 'required' : ''}
                        />
                        <span>${this.escapeHtml(label)}</span>
                    </label>
                `;
            }

            // Multiple checkboxes with section header and grid layout
            const sectionTitle = config.field_label || '';
            const sectionSubtitle = config.field_placeholder || '';
            const dataColumns = config.data_columns || '1';

            let html = '';

            // Section header - only if label or placeholder exists
            if (sectionTitle || sectionSubtitle) {
                html += `
                    <div class="esp-checkbox-section-header">
                        ${sectionTitle ? `<label class="esp-field-label">
                            ${this.escapeHtml(sectionTitle)}
                            ${required ? '<span class="esp-required">*</span>' : ''}
                        </label>` : ''}
                        ${sectionSubtitle ? `<p class="esp-checkbox-section-subtitle">${this.escapeHtml(sectionSubtitle)}</p>` : ''}
                    </div>
                `;
            }

            // Checkboxes grid with container attributes on individual items
            const containerClass = config.container_class || '';
            const containerId = config.container_id || '';

            html += `<div class="esp-field-checkbox-group" data-columns="${dataColumns}">`;

            // Iterate over key-value pairs
            Object.entries(options).forEach(([value, optionLabel], index) => {
                const itemId = containerId ? ` id="${containerId}-${index}"` : '';
                const itemClass = containerClass ? ` ${containerClass}` : '';

                // Check if this checkbox has dependent fields
                const hasDependents = !skipDependents && field.dependent_fields && field.dependent_fields[value];
                const hasDependendClass = hasDependents ? ' esp-checkbox-item-has-dependents' : '';

                html += `
                    <div class="esp-checkbox-item${itemClass}${hasDependendClass}"${itemId}>
                        <label class="esp-checkbox-label">
                            <input
                                type="checkbox"
                                name="${name}[]"
                                value="${this.escapeHtml(value)}"
                                data-field-id="${field.id}"
                            />
                            <span>${this.escapeHtml(optionLabel)}</span>
                        </label>
                        ${hasDependents ? this.renderDependentFieldsForOption(field, value) : ''}
                    </div>
                `;
            });

            html += '</div>';

            // Render dependent fields that don't match any specific checkbox value (after the group)
            if (!skipDependents) {
                html += this.renderUnmatchedDependentFields(field, Object.keys(options));
            }

            return html;
        }

        /**
         * Render toggle switch
         */
        renderToggle(field, name, label, config, required, skipDependents = false) {
            const yesText = config.toggle_yes_text || 'Yes';
            const noText = config.toggle_no_text || 'No';

            let html = `
                <div class="esp-toggle-wrapper">
                    <label class="esp-toggle-label">
                        <span>${this.escapeHtml(label)}</span>
                        ${required ? '<span class="esp-required">*</span>' : ''}
                    </label>
                    <div class="esp-toggle-control">
                        <label class="esp-toggle-option">
                            <input
                                type="radio"
                                name="${name}"
                                value="1"
                                data-field-id="${field.id}"
                                ${required ? 'required' : ''}
                            />
                            <span>${this.escapeHtml(yesText)}</span>
                        </label>
                        <label class="esp-toggle-option">
                            <input
                                type="radio"
                                name="${name}"
                                value="0"
                                data-field-id="${field.id}"
                            />
                            <span>${this.escapeHtml(noText)}</span>
                        </label>
                    </div>
                </div>
            `;

            // Render all dependent fields after the toggle (only if not a dependent field itself)
            if (!skipDependents) {
                html += this.renderAllDependentFields(field);
            }

            return html;
        }

        /**
         * Render dependent fields for a specific option value (for checkbox inline rendering)
         * @param {Object} parentField - The parent field object
         * @param {string} optionValue - The option value to match
         * @returns {string} HTML for dependent fields
         */
        renderDependentFieldsForOption(parentField, optionValue) {
            if (!parentField.dependent_fields || !parentField.dependent_fields[optionValue]) {
                return '';
            }

            const dependentFields = parentField.dependent_fields[optionValue];
            let html = `<div class="esp-dependent-fields-wrapper esp-form-fields" data-parent-field-id="${parentField.id}" data-parent-value="${this.escapeHtml(optionValue)}">`;

            dependentFields.forEach(depField => {
                html += this.renderDependentField(depField, parentField, optionValue);
            });

            html += '</div>';
            return html;
        }

        /**
         * Render dependent fields that don't match any checkbox option (unmatched)
         * @param {Object} parentField - The parent field object
         * @param {Array} matchedKeys - Array of option keys that were already rendered inline
         * @returns {string} HTML for unmatched dependent fields
         */
        renderUnmatchedDependentFields(parentField, matchedKeys) {
            if (!parentField.dependent_fields) {
                return '';
            }

            let html = '';
            const unmatchedHtml = [];

            Object.entries(parentField.dependent_fields).forEach(([key, fields]) => {
                if (!matchedKeys.includes(key)) {
                    fields.forEach(depField => {
                        unmatchedHtml.push(this.renderDependentField(depField, parentField, key));
                    });
                }
            });

            if (unmatchedHtml.length > 0) {
                html = '<div class="esp-dependent-fields-wrapper esp-dependent-unmatched esp-form-fields">';
                html += unmatchedHtml.join('');
                html += '</div>';
            }

            return html;
        }

        /**
         * Render all dependent fields after a field (for radio, select, toggle)
         * @param {Object} parentField - The parent field object
         * @returns {string} HTML for all dependent fields
         */
        renderAllDependentFields(parentField) {
            if (!parentField.dependent_fields) {
                return '';
            }

            let html = '<div class="esp-dependent-fields-wrapper esp-form-fields">';

            Object.entries(parentField.dependent_fields).forEach(([key, fields]) => {
                fields.forEach(depField => {
                    html += this.renderDependentField(depField, parentField, key);
                });
            });

            html += '</div>';
            return html;
        }

        /**
         * Render a single dependent field with indentation and styling
         * @param {Object} depField - The dependent field object
         * @param {Object} parentField - The parent field object (optional)
         * @param {string} parentValue - The parent checkbox value (optional)
         * @returns {string} HTML for the dependent field
         */
        renderDependentField(depField, parentField = null, parentValue = null) {
            const config = depField.config || {};
            const fieldType = depField.field_type;
            const fieldId = depField.field_id;
            const fieldLabel = config.field_label || depField.field_name || depField.field_id;
            const isRequired = config.is_required || false;
            const columns = config.field_columns || 12;
            const placeholder = config.field_placeholder || '';
            const helpText = config.help_text || '';

            // Build nested field name if parent context exists
            // Format: parent_field[parent_value][dependent_field]
            let fieldName = fieldId;
            if (parentField && parentValue) {
                const parentFieldId = parentField.field_id;
                fieldName = `${parentFieldId}[${parentValue}][${fieldId}]`;
            }

            let options = config.field_options || {};
            if (typeof options === 'string') {
                try {
                    options = JSON.parse(options);
                } catch (e) {
                    options = {};
                }
            }

            if (typeof options !== 'object' || options === null) {
                options = {};
            }

            if (Array.isArray(options)) {
                const optionsObj = {};
                options.forEach(opt => {
                    optionsObj[opt] = opt;
                });
                options = optionsObj;
            }

            // Don't render submit button or hidden fields
            if (fieldType === 'submit_button' || fieldType === 'hidden') {
                return '';
            }

            // Start the field wrapper with dependent field classes
            // Use depField.id for data-field-id (pivot ID) to match parent field structure
            let fieldHtml = `<div class="esp-form-field esp-dependent-field esp-col-${columns}" data-field-id="${depField.id}">`;

            // Label - Only show if config.field_label is explicitly set
            const hasOptions = Object.keys(options).length > 0;
            const showLabel = config.field_label && // Only show if field_label exists
                !(fieldType === 'checkbox' && !hasOptions) &&
                fieldType !== 'toggle' &&
                !(fieldType === 'checkbox' && hasOptions && config.field_placeholder) &&
                !(fieldType === 'radio' && hasOptions && config.field_placeholder);

            if (showLabel) {
                fieldHtml += `
                    <label class="esp-field-label" for="esp-field-${depField.id}">
                        ${this.escapeHtml(fieldLabel)}
                        ${isRequired ? '<span class="esp-required">*</span>' : ''}
                    </label>
                `;
            }

            // Render field input based on type (skip rendering dependent fields to prevent infinite recursion)
            switch (fieldType) {
                case 'text':
                case 'email':
                case 'phone':
                case 'number':
                    fieldHtml += this.renderInputField(depField, fieldType, fieldName, placeholder, isRequired);
                    break;

                case 'textarea':
                    fieldHtml += this.renderTextarea(depField, fieldName, placeholder, isRequired);
                    break;

                case 'select':
                    fieldHtml += this.renderSelect(depField, fieldName, options, isRequired, true);
                    break;

                case 'radio':
                    fieldHtml += this.renderRadio(depField, fieldName, fieldLabel, options, isRequired, config, true);
                    break;

                case 'checkbox':
                    fieldHtml += this.renderCheckbox(depField, fieldName, fieldLabel, options, isRequired, config, true);
                    break;

                case 'toggle':
                    fieldHtml += this.renderToggle(depField, fieldName, fieldLabel, config, isRequired, true);
                    break;

                default:
                    fieldHtml += this.renderInputField(depField, 'text', fieldName, placeholder, isRequired);
            }

            // Help text
            if (helpText) {
                fieldHtml += `<small class="esp-field-help">${this.escapeHtml(helpText)}</small>`;
            }

            // Error container
            fieldHtml += `<div class="esp-field-error" style="display:none;"></div>`;

            fieldHtml += `</div>`;

            return fieldHtml;
        }

        /**
         * Render file input with drag & drop upload area
         */
        renderFileInput(field, name, required) {
            // Get file validation settings from field config or use defaults
            const validation = field.config?.file_validation || this.defaultFileValidation;

            // Store validation settings for this field
            this.fileValidationSettings[field.id] = validation;

            // Build human-readable file type description
            const typeLabels = {
                'image': 'Images (JPG, PNG, GIF)',
                'pdf': 'PDF Documents',
                'document': 'Documents (DOC, DOCX)',
                'spreadsheet': 'Spreadsheets (XLS, XLSX, CSV)',
                'any': 'Any file type'
            };

            const allowedTypesText = validation.allowed_types
                .map(type => typeLabels[type] || type.toUpperCase())
                .join(', ');

            const maxSizeMB = (validation.max_size_kb / 1024).toFixed(0);
            const helpText = `${allowedTypesText}, up to ${maxSizeMB} MB each. Max ${validation.max_files} file(s).`;

            return `
                <div class="esp-file-upload-wrapper">
                    <!-- File Upload Input -->
                    <div class="mt-3">
                        <label for="esp-field-${field.id}-file-input"
                            class="esp-file-upload-area cursor-pointer border border-dashed border-gray-400 text-center p-4 rounded-xl min-h-[206px] relative flex items-center justify-center flex-col"
                            data-field-id="${field.id}">

                            <input type="file"
                                id="esp-field-${field.id}-file-input"
                                name="${name}"
                                multiple
                                accept="${this.escapeHtml(validation.accept)}"
                                class="hidden"
                                data-field-id="${field.id}"
                                data-required="${required ? 'true' : 'false'}">

                            <div class="flex flex-col items-center">
                                <!-- Upload Icon -->
                                <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 48 48" fill="none">
                                    <rect width="48" height="48" rx="24" fill="#F3F4F6"/>
                                    <g clip-path="url(#clip0_4471_62054)">
                                        <path d="M28 28L24 24M24 24L20 28M24 24V32M32 27.7906C33.1777 27.0949 34.1123 26.0534 34.6691 24.8157C35.2259 23.578 35.3778 22.2068 35.1043 20.8823C34.8308 19.5578 34.1449 18.3486 33.1451 17.4249C32.1452 16.5012 30.8827 15.9082 29.5333 15.7273C28.7745 13.8722 27.4732 12.2823 25.8005 11.1667C24.1279 10.0512 22.1608 9.46012 20.1481 9.46689C18.1354 9.47366 16.1722 10.0781 14.5067 11.2044C12.8411 12.3307 11.5492 13.9295 10.8 15.79C9.51638 16.0764 8.35558 16.7401 7.47235 17.6923C6.58911 18.6444 6.02605 19.8397 5.85943 21.1174C5.6928 22.3951 5.93078 23.6936 6.54229 24.8311C7.1538 25.9685 8.10897 26.8909 9.26667 27.4753" stroke="#6B7280" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                                    </g>
                                    <defs>
                                        <clipPath id="clip0_4471_62054">
                                            <rect width="32" height="32" fill="white" transform="translate(8 8)"/>
                                        </clipPath>
                                    </defs>
                                </svg>

                                <h5 class="sm:text-sm text-xs font-medium leading-5 mt-5 tracking-[-0.096px] text-gray-900 text-center">
                                    Choose a file or drag &amp; drop it here.
                                </h5>
                                <p class="mt-1.5 mb-5 sm:text-sm text-xs font-normal leading-5 tracking-[-0.096px] text-gray-600 text-center">
                                    ${this.escapeHtml(helpText)}
                                </p>
                                <span class="text-gray-600 font-normal sm:text-sm text-xs leading-5 tracking-[-0.096px] border border-solid border-gray-300 rounded-full h-8 w-[104px] flex items-center justify-center">
                                    Browse Files
                                </span>
                            </div>
                        </label>
                    </div>

                    <!-- File Preview Section -->
                    <div id="esp-field-${field.id}-preview" style="display: none;"></div>
                </div>
            `;
        }

        /**
         * Render geolocation field
         */
        renderGeolocation(field, name, required) {
            return `
                <div class="esp-geolocation-wrapper">
                    <input
                        type="text"
                        id="esp-field-${field.id}"
                        name="${name}"
                        class="esp-field-input esp-field-geolocation"
                        data-field-id="${field.id}"
                        placeholder="Detecting location..."
                        readonly
                        ${required ? 'required' : ''}
                    />
                    <button type="button" class="esp-geolocation-btn" onclick="ESPLeadFormSDK.getCurrentLocation(${field.id})">
                        Get Location
                    </button>
                </div>
            `;
        }

        /**
         * Setup file upload event listeners
         * @param {HTMLElement} container - Form container
         */
        setupFileUploadListeners(container) {
            // Find all file input fields
            const fileInputs = container.querySelectorAll('input[type="file"][data-field-id]');

            fileInputs.forEach(input => {
                const fieldId = input.getAttribute('data-field-id');
                const uploadArea = container.querySelector(`.esp-file-upload-area[data-field-id="${fieldId}"]`);

                if (!uploadArea) {
                    return;
                }

                // File input change event
                input.addEventListener('change', (e) => {
                    if (e.target.files && e.target.files.length > 0) {
                        this.handleFileSelection(fieldId, e.target.files);
                        // Clear the input to allow selecting the same file again
                        e.target.value = '';
                    }
                });

                // Drag and drop events
                uploadArea.addEventListener('dragover', (e) => {
                    e.preventDefault();
                    e.stopPropagation();
                    uploadArea.classList.add('border-blue-500', 'bg-blue-50');
                });

                uploadArea.addEventListener('dragleave', (e) => {
                    e.preventDefault();
                    e.stopPropagation();
                    uploadArea.classList.remove('border-blue-500', 'bg-blue-50');
                });

                uploadArea.addEventListener('drop', (e) => {
                    e.preventDefault();
                    e.stopPropagation();
                    uploadArea.classList.remove('border-blue-500', 'bg-blue-50');

                    if (e.dataTransfer.files && e.dataTransfer.files.length > 0) {
                        this.handleFileSelection(fieldId, e.dataTransfer.files);
                    }
                });
            });
        }

        /**
         * Setup conditional fields (dependent fields logic)
         * @param {HTMLElement} container - Form container
         * @param {Array} fields - Array of field objects
         */
        setupConditionalFields(container, fields) {
            // Build a map of fields with dependent fields
            const fieldsWithDependents = new Map();

            fields.forEach(field => {
                if (field.dependent_fields && Object.keys(field.dependent_fields).length > 0) {
                    fieldsWithDependents.set(field.id, {
                        field: field,
                        dependentFields: field.dependent_fields
                    });
                }
            });

            if (fieldsWithDependents.size === 0) {
                return; // No conditional logic needed
            }

            // Initially hide all dependent fields
            this.hideAllDependentFields(container, fieldsWithDependents);

            // Attach change listeners to parent fields
            fieldsWithDependents.forEach((data, parentFieldId) => {
                const { field, dependentFields } = data;
                const parentElements = container.querySelectorAll(`[data-field-id="${parentFieldId}"]`);

                parentElements.forEach(element => {
                    // Skip if this element is inside a dependent fields wrapper (to prevent bubbling issues)
                    if (element.closest('.esp-dependent-fields-wrapper')) {
                        return;
                    }

                    const eventType = this.getFieldChangeEvent(field.field_type);

                    element.addEventListener(eventType, (e) => {
                        // Only handle if the event target is this element (not a child inside dependent fields)
                        if (e.target === element) {
                            this.handleConditionalFieldChange(container, field, dependentFields, e);
                        }
                    });

                    // Trigger initial state based on current value
                    if (element.value || element.checked) {
                        this.handleConditionalFieldChange(container, field, dependentFields, { target: element });
                    }
                });
            });
        }

        /**
         * Get the appropriate change event for a field type
         * @param {string} fieldType - Field type
         * @returns {string} Event name
         */
        getFieldChangeEvent(fieldType) {
            if (fieldType === 'select' || fieldType === 'radio') {
                return 'change';
            }
            if (fieldType === 'checkbox') {
                return 'change';
            }
            if (fieldType === 'toggle') {
                return 'change';
            }
            return 'change';
        }

        /**
         * Handle conditional field change event
         * @param {HTMLElement} container - Form container
         * @param {Object} parentField - Parent field object
         * @param {Object} dependentFields - Dependent fields grouped by value
         * @param {Event} event - Change event
         */
        handleConditionalFieldChange(container, parentField, dependentFields, event) {
            const fieldType = parentField.field_type;
            const selectedValues = this.getSelectedValues(event.target, fieldType);

            // First, hide all dependent field wrappers for this parent
            const allWrappers = container.querySelectorAll(`.esp-dependent-fields-wrapper[data-parent-field-id="${parentField.id}"]`);
            allWrappers.forEach(wrapper => {
                wrapper.classList.add('esp-field-conditional-hidden');
                wrapper.style.display = 'none';
                // Note: We don't clear field values - user selections are preserved when re-showing
            });

            // Then, show wrappers for the selected values
            selectedValues.forEach(value => {
                const wrappersToShow = container.querySelectorAll(
                    `.esp-dependent-fields-wrapper[data-parent-field-id="${parentField.id}"][data-parent-value="${value}"]`
                );
                wrappersToShow.forEach(wrapper => {
                    wrapper.classList.remove('esp-field-conditional-hidden');
                    wrapper.style.display = '';
                });
            });
        }

        /**
         * Get selected values from a field
         * @param {HTMLElement} element - Form element
         * @param {string} fieldType - Field type
         * @returns {Array} Selected values
         */
        getSelectedValues(element, fieldType) {
            if (fieldType === 'checkbox') {
                // For checkbox groups, get all checked values with the same data-field-id
                const fieldId = element.getAttribute('data-field-id');
                const checkboxes = document.querySelectorAll(`input[type="checkbox"][data-field-id="${fieldId}"]:checked`);
                return Array.from(checkboxes).map(cb => cb.value);
            }

            if (fieldType === 'radio') {
                return element.checked ? [element.value] : [];
            }

            if (fieldType === 'select') {
                return element.value ? [element.value] : [];
            }

            if (fieldType === 'toggle') {
                // Toggle fields use 1/0 or true/false
                return element.checked ? ['1'] : ['0'];
            }

            return element.value ? [element.value] : [];
        }

        /**
         * Hide all dependent fields initially
         * @param {HTMLElement} container - Form container
         * @param {Map} fieldsWithDependents - Map of parent fields with their dependents
         */
        hideAllDependentFields(container, fieldsWithDependents) {
            // Hide all dependent field wrappers
            const allWrappers = container.querySelectorAll('.esp-dependent-fields-wrapper');
            allWrappers.forEach(wrapper => {
                wrapper.classList.add('esp-field-conditional-hidden');
                wrapper.style.display = 'none';
            });
        }

        /**
         * Hide a field
         * @param {HTMLElement} container - Form container
         * @param {number} fieldId - Field ID
         */
        hideField(container, fieldId) {
            const fieldContainer = container.querySelector(`.esp-form-field[data-field-id="${fieldId}"]`);
            if (fieldContainer) {
                fieldContainer.classList.add('esp-field-conditional-hidden');
                fieldContainer.style.display = 'none';
            }
        }

        /**
         * Show a field
         * @param {HTMLElement} container - Form container
         * @param {number} fieldId - Field ID
         */
        showField(container, fieldId) {
            const fieldContainer = container.querySelector(`.esp-form-field[data-field-id="${fieldId}"]`);
            if (fieldContainer) {
                fieldContainer.classList.remove('esp-field-conditional-hidden');
                fieldContainer.style.display = '';
            }
        }

        /**
         * Clear field value
         * @param {HTMLElement} container - Form container
         * @param {number} fieldId - Field ID
         */
        clearFieldValue(container, fieldId) {
            const inputs = container.querySelectorAll(`[data-field-id="${fieldId}"]`);
            inputs.forEach(input => {
                if (input.type === 'checkbox' || input.type === 'radio') {
                    input.checked = false;
                } else if (input.tagName === 'SELECT') {
                    input.selectedIndex = 0;
                } else {
                    input.value = '';
                }
            });
        }

        /**
         * Handle form submission
         * @param {Event} event - Submit event
         * @param {Object} formInstance - Form instance
         */
        async handleSubmit(event, formInstance) {
            event.preventDefault();

            const form = event.target;
            const submitButton = form.querySelector('.esp-form-submit');
            const messagesContainer = form.querySelector('.esp-form-messages');

            // Store original button text
            const originalButtonText = formInstance.submitButtonText || 'Submit';

            // Clear previous messages
            messagesContainer.innerHTML = '';
            this.clearFieldErrors(form);

            // For wizard forms, collect data from the last step before submitting
            if (formInstance.wizard && formInstance.wizard.isWizard) {
                const currentStep = formInstance.wizard.activeSteps[formInstance.wizard.currentStepIndex];
                this.collectStepData(form, currentStep, formInstance.wizard);
            }

            // Disable submit button
            submitButton.disabled = true;
            submitButton.textContent = 'Submitting...';

            try {
                // Collect form data
                const formData = this.collectFormData(form, formInstance.formData);

                // Validate
                const validation = this.validateFormData(formData, formInstance.formData);
                if (!validation.valid) {
                    this.showValidationErrors(form, validation.errors);
                    throw new Error('Please fix the validation errors');
                }

                // Submit to API
                const result = await this.submitForm(formInstance, formData);

                // Show success message and hide form
                const successMessage = formInstance.formData.submit_message || result.message || 'Thank you for your submission!';
                this.showSuccessAndHideForm(formInstance.container, messagesContainer, successMessage);

                // Reset form (hidden now but reset for potential reuse)
                form.reset();

                // Clear file uploads
                this.fileUploads = {};

            } catch (error) {
                this.showError(messagesContainer, error.message);
            } finally {
                submitButton.disabled = false;
                submitButton.textContent = originalButtonText;
            }
        }

        /**
         * Collect form data
         * @param {HTMLFormElement} form - Form element
         * @param {Object} formConfig - Form configuration
         * @returns {Object} Collected form data
         */
        collectFormData(form, formConfig) {
            const formData = new FormData(form);
            const data = {
                lead_data: {},
                lead_source: window.location.href,  // Current client website URL
                attachments: []  // Array to hold file attachments
            };

            // Standard field names that go at top level (including address fields)
            const standardFields = [
                'first_name', 'last_name', 'email', 'phone', 'message', 'form_id',
                'address', 'lat', 'lng', 'city', 'state', 'state_code', 'country', 'country_code'
            ];

            /**
             * Parse nested field name like "parent[value][field]" or "parent[value][field][]"
             * Returns: { isNested: true, parentField: "parent", parentValue: "value", fieldName: "field", isArray: false }
             * Or for simple fields: { isNested: false, fieldName: "field", isArray: false }
             */
            const parseFieldName = (name) => {
                // Check if it's an array field
                const isArray = name.endsWith('[]');
                const cleanName = isArray ? name.slice(0, -2) : name;

                // Check for nested pattern: parent[value][field]
                const nestedMatch = cleanName.match(/^([^\[]+)\[([^\]]+)\]\[([^\]]+)\]$/);

                if (nestedMatch) {
                    return {
                        isNested: true,
                        parentField: nestedMatch[1],
                        parentValue: nestedMatch[2],
                        fieldName: nestedMatch[3],
                        isArray: isArray
                    };
                }

                // Simple field
                return {
                    isNested: false,
                    fieldName: cleanName,
                    isArray: isArray
                };
            };

            /**
             * Set a value in a nested object structure
             */
            const setNestedValue = (obj, keys, value, isArray) => {
                const lastKey = keys[keys.length - 1];
                const parent = keys.slice(0, -1).reduce((acc, key) => {
                    if (!acc[key]) {
                        acc[key] = {};
                    }
                    return acc[key];
                }, obj);

                if (isArray) {
                    if (!parent[lastKey]) {
                        parent[lastKey] = [];
                    }
                    parent[lastKey].push(value);
                } else {
                    parent[lastKey] = value;
                }
            };

            // First pass: identify which parent fields have dependent fields
            const parentsWithDependents = new Set();
            for (const [name] of formData.entries()) {
                const nestedMatch = name.match(/^([^\[]+)\[([^\]]+)\]\[([^\]]+)\](\[\])?$/);
                if (nestedMatch) {
                    parentsWithDependents.add(nestedMatch[1]);
                }
            }

            // Collect all field values
            for (const [name, value] of formData.entries()) {
                const parsed = parseFieldName(name);

                if (parsed.isNested) {
                    // Nested field from dependent fields: parent[value][field]
                    // Determine if parent field should go to lead_data or top level
                    const isStandardParent = standardFields.includes(parsed.parentField);
                    const container = isStandardParent ? data : data.lead_data;

                    // Ensure parent field is an object (not an array) if it has nested dependents
                    if (!container[parsed.parentField]) {
                        container[parsed.parentField] = {};
                    } else if (Array.isArray(container[parsed.parentField])) {
                        // If it's currently an array, convert to object (discard the array values)
                        container[parsed.parentField] = {};
                    }

                    // Set the nested value: container[parentField][parentValue][fieldName] = value
                    setNestedValue(container, [parsed.parentField, parsed.parentValue, parsed.fieldName], value, parsed.isArray);
                } else {
                    // Regular field (non-nested)
                    if (standardFields.includes(parsed.fieldName)) {
                        // Standard field - add to top level
                        if (parsed.isArray) {
                            // Check if this field has dependent fields
                            if (parentsWithDependents.has(parsed.fieldName)) {
                                // Skip - we'll only store the dependent field data, not the parent values
                                // The parent values can be inferred from which dependent field groups exist
                            } else {
                                // Regular array field
                                if (!data[parsed.fieldName]) {
                                    data[parsed.fieldName] = [];
                                }
                                data[parsed.fieldName].push(value);
                            }
                        } else {
                            data[parsed.fieldName] = value;
                        }
                    } else {
                        // Custom field - add to lead_data
                        if (parsed.isArray) {
                            // Check if this field has dependent fields
                            if (parentsWithDependents.has(parsed.fieldName)) {
                                // Skip - we'll only store the dependent field data, not the parent values
                                // The parent values can be inferred from which dependent field groups exist
                            } else {
                                // Regular array field
                                if (!data.lead_data[parsed.fieldName]) {
                                    data.lead_data[parsed.fieldName] = [];
                                }
                                data.lead_data[parsed.fieldName].push(value);
                            }
                        } else {
                            data.lead_data[parsed.fieldName] = value;
                        }
                    }
                }
            }

            // Collect file uploads from fileUploads object
            // Handle both standard forms and wizard forms (which have steps)
            let allFields = [];

            if (formConfig.fields && Array.isArray(formConfig.fields)) {
                // Standard form - fields are in a flat array
                allFields = formConfig.fields;
            } else if (formConfig.steps && Array.isArray(formConfig.steps)) {
                // Wizard form - fields are organized by steps
                formConfig.steps.forEach(step => {
                    if (step.fields && Array.isArray(step.fields)) {
                        allFields = allFields.concat(step.fields);
                    }
                });
            }

            // Filter for file upload fields
            const fileFields = allFields.filter(f => f.field_type === 'file' || f.field_type === 'file_upload');

            // Process each file field
            fileFields.forEach(field => {
                const fieldId = field.id || field.field_id;
                const files = this.fileUploads[fieldId] || [];

                files.forEach(fileData => {
                    // Add file info to attachments (will be converted to base64 during submission)
                    data.attachments.push({
                        field_id: fieldId,
                        field_name: field.field_name || fieldId,
                        name: fileData.name,
                        type: fileData.type,
                        size: fileData.size,
                        file: fileData.file  // Keep reference to actual file object
                    });
                });
            });

            return data;
        }

        /**
         * Validate form data
         * @param {Object} data - Form data
         * @param {Object} formConfig - Form configuration
         * @returns {Object} Validation result
         */
        validateFormData(data, formConfig) {
            const errors = {};
            const fields = formConfig.fields || [];

            // Standard field names that are at top level
            const standardFields = ['first_name', 'last_name', 'email', 'phone', 'message'];

            fields.forEach(field => {
                // Skip non-input field types
                if (field.field_type === 'submit_button' || field.field_type === 'section_title' || field.field_type === 'hidden') {
                    return;
                }

                const config = field.config || {};
                const fieldId = field.field_id;
                const fieldLabel = field.field_name || field.field_id;
                const isRequired = config.is_required || false;

                // Special handling for file uploads
                if (field.field_type === 'file') {
                    if (isRequired) {
                        const uploadedFiles = this.fileUploads[field.id] || [];
                        if (uploadedFiles.length === 0) {
                            errors[field.id] = `${fieldLabel} is required`;
                        }
                    }
                    return;
                }

                // Get value based on whether it's a standard field or custom field
                let value;
                if (standardFields.includes(fieldId)) {
                    // Standard field - check top level
                    value = data[fieldId];
                } else {
                    // Custom field - check lead_data with field_id as key
                    value = data.lead_data[fieldId];
                }

                // Required validation
                if (isRequired && (!value || value === '' || (Array.isArray(value) && value.length === 0))) {
                    errors[field.id] = `${fieldLabel} is required`;
                    return;
                }

                // Skip further validation if empty and not required
                if (!value || value === '' || (Array.isArray(value) && value.length === 0)) {
                    return;
                }

                // Type-specific validation
                const dataType = field.data_type;

                if (dataType === 'email') {
                    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                    if (!emailRegex.test(value)) {
                        errors[field.id] = 'Please enter a valid email address';
                    }
                }

                if (dataType === 'phone') {
                    const phoneRegex = /^[\d\s\-\+\(\)]+$/;
                    if (!phoneRegex.test(value)) {
                        errors[field.id] = 'Please enter a valid phone number';
                    }
                }

                if (dataType === 'number' || field.field_type === 'number') {
                    if (isNaN(value)) {
                        errors[field.id] = 'Please enter a valid number';
                    }
                }
            });

            return {
                valid: Object.keys(errors).length === 0,
                errors: errors
            };
        }

        /**
         * Submit form data to API
         * @param {Object} formInstance - Form instance
         * @param {Object} data - Form data
         * @returns {Promise<Object>} API response
         */
        async submitForm(formInstance, data) {
            const url = `${formInstance.apiUrl}/api/${this.config.apiVersion}/leads`;

            this.log(`Submitting form to: ${url}`);

            // Convert file attachments to base64 before sending
            if (data.attachments && data.attachments.length > 0) {
                const convertedAttachments = await Promise.all(
                    data.attachments.map(attachment => {
                        return new Promise((resolve, reject) => {
                            if (!attachment.file) {
                                resolve(attachment);
                                return;
                            }

                            const reader = new FileReader();
                            reader.onload = () => {
                                resolve({
                                    field_id: attachment.field_id,
                                    field_name: attachment.field_name,
                                    name: attachment.name,
                                    type: attachment.type,
                                    size: attachment.size,
                                    data: reader.result  // base64 data URL
                                });
                            };
                            reader.onerror = reject;
                            reader.readAsDataURL(attachment.file);
                        });
                    })
                );
                data.attachments = convertedAttachments;
            }

            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${formInstance.apiToken}`,
                    'Accept': 'application/json',
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(data)
            });

            const result = await response.json();

            if (!response.ok) {
                if (response.status === 422 && result.errors) {
                    // Validation errors from server
                    const formattedErrors = {};
                    Object.keys(result.errors).forEach(fieldName => {
                        // Try to match field name to field ID
                        const field = formInstance.formData.fields.find(f =>
                            f.field_name === fieldName || f.field_id === fieldName
                        );
                        if (field) {
                            formattedErrors[field.id] = result.errors[fieldName][0];
                        }
                    });

                    if (Object.keys(formattedErrors).length > 0) {
                        this.showValidationErrors(formInstance.container.querySelector('.esp-form'), formattedErrors);
                    }
                }

                throw new Error(result.message || 'Failed to submit form');
            }

            return result;
        }

        /**
         * Format date as MySQL datetime string (YYYY-MM-DD HH:MM:SS)
         * @param {Date} date - Date object to format
         * @returns {string} MySQL datetime formatted string
         */
        formatMySQLDateTime(date) {
            const year = date.getFullYear();
            const month = String(date.getMonth() + 1).padStart(2, '0');
            const day = String(date.getDate()).padStart(2, '0');
            const hours = String(date.getHours()).padStart(2, '0');
            const minutes = String(date.getMinutes()).padStart(2, '0');
            const seconds = String(date.getSeconds()).padStart(2, '0');

            return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
        }

        /**
         * Get or create a unique visitor identifier
         * Persists across page visits using localStorage
         * @returns {string} Visitor identifier
         */
        getOrCreateVisitorId() {
            const storageKey = 'esp_visitor_id';

            try {
                // Try to get existing visitor ID
                let visitorId = localStorage.getItem(storageKey);

                if (!visitorId) {
                    // Generate new visitor ID
                    visitorId = 'visitor_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
                    localStorage.setItem(storageKey, visitorId);
                    this.log(`Generated new visitor ID: ${visitorId}`);
                } else {
                    this.log(`Using existing visitor ID: ${visitorId}`);
                }

                return visitorId;
            } catch (error) {
                // Fallback if localStorage is not available
                this.log('localStorage not available, using session-based ID');
                return 'session_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
            }
        }

        /**
         * Check if this form visit has already been tracked in this session
         * @param {string} formId - Form UUID
         * @returns {boolean} True if already tracked
         */
        hasTrackedThisSession(formId) {
            const storageKey = `esp_tracked_${formId}`;

            try {
                // Check sessionStorage for this form
                const tracked = sessionStorage.getItem(storageKey);
                return tracked === 'true';
            } catch (error) {
                // If sessionStorage not available, assume not tracked
                this.log('sessionStorage not available, will track visit');
                return false;
            }
        }

        /**
         * Mark this form as tracked in this session
         * @param {string} formId - Form UUID
         */
        markAsTracked(formId) {
            const storageKey = `esp_tracked_${formId}`;

            try {
                sessionStorage.setItem(storageKey, 'true');
                this.log(`Marked form ${formId} as tracked in session`);
            } catch (error) {
                // Silent fail
                this.log('Could not mark form as tracked: ' + error.message);
            }
        }

        /**
         * Track form visit using the new CRUD API
         * @param {Object} formInstance - Form instance
         */
        async trackVisit(formInstance) {
            try {
                // Check if already tracked in this session (prevent duplicates)
                if (this.hasTrackedThisSession(formInstance.id)) {
                    this.log(`Form ${formInstance.id} already tracked in this session, skipping`);
                    return;
                }

                // Get visitor identifier
                const visitorId = this.getOrCreateVisitorId();

                // Prepare tracking data with MySQL-compatible datetime
                const trackingData = {
                    source: window.location.hostname, // Client website domain
                    unique_identifier: visitorId,
                    visit_date: this.formatMySQLDateTime(new Date())
                };

                // Use the new CRUD endpoint for visits
                const url = `${formInstance.apiUrl}/api/${this.config.apiVersion}/lead-forms/${formInstance.id}/visits`;

                this.log(`Tracking visit to: ${url}`);
                this.log('Tracking data:', trackingData);

                const response = await fetch(url, {
                    method: 'POST',
                    headers: {
                        'Authorization': `Bearer ${formInstance.apiToken}`,
                        'Accept': 'application/json',
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(trackingData)
                });

                if (response.ok) {
                    const result = await response.json();
                    this.log('Visit tracked successfully:', result);

                    // Mark as tracked in this session
                    this.markAsTracked(formInstance.id);
                } else {
                    const errorText = await response.text();
                    this.log(`Visit tracking failed with status ${response.status}: ${errorText}`);
                }

            } catch (error) {
                // Silent fail for visit tracking - don't break the form
                this.log('Visit tracking failed: ' + error.message);
            }
        }

        /**
         * Show validation errors
         * @param {HTMLFormElement} form - Form element
         * @param {Object} errors - Field errors (fieldId: message)
         */
        showValidationErrors(form, errors) {
            let firstErrorElement = null;

            Object.keys(errors).forEach(fieldId => {
                // Find the field container div (not the input)
                const fieldContainer = form.querySelector(`.esp-form-field[data-field-id="${fieldId}"]`);

                if (fieldContainer) {
                    // Find and display error message
                    const errorContainer = fieldContainer.querySelector('.esp-field-error');
                    if (errorContainer) {
                        errorContainer.textContent = errors[fieldId];
                        errorContainer.style.display = 'block';
                    }

                    // Add error class to container
                    fieldContainer.classList.add('esp-field-has-error');

                    // Find the actual input/select/textarea and add error class
                    const inputElement = fieldContainer.querySelector('.esp-field-input, .esp-field-textarea, .esp-field-select');
                    if (inputElement) {
                        inputElement.classList.add('esp-field-has-error');
                    }

                    // Store first error element for scrolling
                    if (!firstErrorElement) {
                        firstErrorElement = fieldContainer;
                    }
                } else {
                    // Fallback: try to find by any element with field ID (for field groups like radio/checkbox)
                    const fieldElements = form.querySelectorAll(`[data-field-id="${fieldId}"]`);

                    if (fieldElements.length > 0) {
                        // Find the parent field container
                        let container = fieldElements[0];
                        while (container && !container.classList.contains('esp-form-field')) {
                            container = container.parentElement;
                        }

                        if (container) {
                            const errorContainer = container.querySelector('.esp-field-error');
                            if (errorContainer) {
                                errorContainer.textContent = errors[fieldId];
                                errorContainer.style.display = 'block';
                            }
                            container.classList.add('esp-field-has-error');

                            // Store first error element for scrolling
                            if (!firstErrorElement) {
                                firstErrorElement = container;
                            }
                        }
                    }
                }
            });

            // Scroll to first error
            if (firstErrorElement) {
                firstErrorElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }
        }

        /**
         * Clear field errors
         * @param {HTMLFormElement} form - Form element
         */
        clearFieldErrors(form) {
            const errorContainers = form.querySelectorAll('.esp-field-error');
            errorContainers.forEach(container => {
                container.textContent = '';
                container.style.display = 'none';
            });

            const errorFields = form.querySelectorAll('.esp-field-has-error');
            errorFields.forEach(field => {
                field.classList.remove('esp-field-has-error');
            });
        }

        /**
         * Show success message
         * @param {HTMLElement} container - Message container
         * @param {string} message - Success message
         */
        showSuccess(container, message) {
            container.innerHTML = `
                <div class="esp-message esp-message-success">
                    ${this.escapeHtml(message)}
                </div>
            `;
        }

        /**
         * Show success message and hide the form
         * @param {HTMLElement} formContainer - Main form container
         * @param {HTMLElement} messagesContainer - Message container
         * @param {string} message - Success message
         */
        showSuccessAndHideForm(formContainer, messagesContainer, message) {
            // Hide the entire form wrapper (includes form, wizard steps, title, etc.)
            const formWrapper = formContainer.querySelector('.esp-form-wrapper');
            if (formWrapper) {
                formWrapper.style.display = 'none';
            } else {
                // Fallback: hide the form element itself
                const form = formContainer.querySelector('.esp-form');
                if (form) {
                    form.style.display = 'none';
                }
            }

            // Hide form title if it exists
            const formTitle = formContainer.querySelector('.esp-form-title');
            if (formTitle) {
                formTitle.style.display = 'none';
            }

            // Hide form description if it exists
            const formDescription = formContainer.querySelector('.esp-form-description');
            if (formDescription) {
                formDescription.style.display = 'none';
            }

            // Hide wizard progress if it exists
            const wizardProgress = formContainer.querySelector('.esp-wizard-progress');
            if (wizardProgress) {
                wizardProgress.style.display = 'none';
            }

            // Create success message in the main form container (not inside the hidden wrapper)
            const successHtml = `
                <div class="esp-success-card">
                    <div class="esp-success-icon">
                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                            <path fill-rule="evenodd" d="M2.25 12c0-5.385 4.365-9.75 9.75-9.75s9.75 4.365 9.75 9.75-4.365 9.75-9.75 9.75S2.25 17.385 2.25 12zm13.36-1.814a.75.75 0 10-1.22-.872l-3.236 4.53L9.53 12.22a.75.75 0 00-1.06 1.06l2.25 2.25a.75.75 0 001.14-.094l3.75-5.25z" clip-rule="evenodd" />
                        </svg>
                    </div>
                    <h3 class="esp-success-title">Success!</h3>
                    <p class="esp-success-message">${this.escapeHtml(message)}</p>
                </div>
            `;

            // Check if success card already exists, if not create it
            let successContainer = formContainer.querySelector('.esp-success-card');
            if (successContainer) {
                // Update existing success card
                successContainer.outerHTML = successHtml;
            } else {
                // Insert success card into the main container
                formContainer.insertAdjacentHTML('beforeend', successHtml);
            }

            // Scroll to the form container to show success message
            formContainer.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }

        /**
         * Show error message
         * @param {HTMLElement} container - Message container or element
         * @param {string} message - Error message
         */
        showError(container, message) {
            container.innerHTML = `
                <div class="esp-message esp-message-error">
                    ${this.escapeHtml(message)}
                </div>
            `;
        }

        /**
         * Get current geolocation
         * @param {number} fieldId - Field ID
         */
        getCurrentLocation(fieldId) {
            const input = document.querySelector(`#esp-field-${fieldId}`);

            if (!input) return;

            if (!navigator.geolocation) {
                input.value = 'Geolocation not supported';
                return;
            }

            input.value = 'Getting location...';

            navigator.geolocation.getCurrentPosition(
                (position) => {
                    const lat = position.coords.latitude.toFixed(6);
                    const lng = position.coords.longitude.toFixed(6);
                    input.value = `${lat}, ${lng}`;
                },
                (error) => {
                    input.value = 'Location access denied';
                    console.error('Geolocation error:', error);
                }
            );
        }

        /**
         * Validate file type and size against field-specific validation rules
         * @param {File} file - File to validate
         * @param {string} fieldId - Field ID to get validation settings
         * @returns {Object} Validation result with valid flag and error message
         */
        validateFile(file, fieldId) {
            // Get validation settings for this field
            const validation = this.fileValidationSettings[fieldId] || this.defaultFileValidation;
            const maxSizeBytes = validation.max_size_kb * 1024;

            // Check file size first (more common issue)
            if (file.size > maxSizeBytes) {
                const fileSizeMB = (file.size / 1024 / 1024).toFixed(2);
                const maxSizeMB = (validation.max_size_kb / 1024).toFixed(0);
                return {
                    valid: false,
                    error: `File size exceeds ${maxSizeMB} MB limit. File size: ${fileSizeMB} MB`
                };
            }

            // Check file type (MIME type)
            const allowedMimeTypes = validation.mime_types || [];
            const allowedExtensions = validation.extensions || [];

            // Check if 'any' type is allowed
            if (validation.allowed_types && validation.allowed_types.includes('any')) {
                return { valid: true };
            }

            // Extract file extension
            const fileExtension = file.name.split('.').pop().toLowerCase();

            // Check if file type or extension matches
            const mimeMatch = allowedMimeTypes.some(mime => {
                // Support wildcard MIME types like 'image/*'
                if (mime.endsWith('/*')) {
                    const mimePrefix = mime.slice(0, -2);
                    return file.type.startsWith(mimePrefix);
                }
                return file.type === mime;
            });

            const extensionMatch = allowedExtensions.includes(fileExtension);

            if (!mimeMatch && !extensionMatch) {
                // Build friendly error message
                const allowedTypesText = validation.allowed_types
                    .map(type => type.toUpperCase())
                    .join(', ');

                return {
                    valid: false,
                    error: `Invalid file type. Only ${allowedTypesText} files are allowed.`
                };
            }

            return { valid: true };
        }

        /**
         * Handle file selection for a field
         * @param {string} fieldId - Field ID
         * @param {FileList} files - Selected files
         */
        handleFileSelection(fieldId, files) {
            // Initialize file array for this field if not exists
            if (!this.fileUploads[fieldId]) {
                this.fileUploads[fieldId] = [];
            }

            // Get validation settings for this field
            const validation = this.fileValidationSettings[fieldId] || this.defaultFileValidation;
            const maxFiles = validation.max_files || 5;

            const currentFiles = this.fileUploads[fieldId];
            const remainingSlots = maxFiles - currentFiles.length;

            if (remainingSlots <= 0) {
                alert(`Maximum ${maxFiles} file(s) allowed for this field.`);
                return;
            }

            // Process new files
            const filesToAdd = Array.from(files).slice(0, remainingSlots);
            const errors = [];

            filesToAdd.forEach((file) => {
                const validationResult = this.validateFile(file, fieldId);

                if (validationResult.valid) {
                    // Create file data object
                    const fileData = {
                        file: file,
                        name: file.name,
                        size: file.size,
                        type: file.type,
                        preview: null
                    };

                    // Generate preview URL for images
                    if (file.type.startsWith('image/')) {
                        fileData.preview = URL.createObjectURL(file);
                    }

                    currentFiles.push(fileData);
                } else {
                    errors.push(`${file.name}: ${validationResult.error}`);
                }
            });

            // Show errors if any
            if (errors.length > 0) {
                alert('Some files could not be added:\n\n' + errors.join('\n'));
            }

            // Update the file uploads state
            this.fileUploads[fieldId] = currentFiles;

            // Re-render file previews
            this.updateFilePreviews(fieldId);
        }

        /**
         * Remove a file from selection
         * @param {string} fieldId - Field ID
         * @param {number} index - Index of file to remove
         */
        removeFile(fieldId, index) {
            if (!this.fileUploads[fieldId]) {
                return;
            }

            const files = this.fileUploads[fieldId];

            // Revoke object URL to free memory if it's an image
            if (files[index] && files[index].preview) {
                URL.revokeObjectURL(files[index].preview);
            }

            // Remove file from array
            files.splice(index, 1);

            // Update state
            this.fileUploads[fieldId] = files;

            // Re-render file previews
            this.updateFilePreviews(fieldId);

            // Re-enable file input if we're now below max files
            const validation = this.fileValidationSettings[fieldId] || this.defaultFileValidation;
            const maxFiles = validation.max_files || 5;

            const fileInput = document.querySelector(`#esp-field-${fieldId}-file-input`);
            if (fileInput && files.length < maxFiles) {
                fileInput.disabled = false;
            }
        }

        /**
         * Render single file preview HTML
         * @param {Object} fileData - File data object
         * @param {number} index - File index
         * @param {string} fieldId - Field ID
         * @returns {string} File preview HTML
         */
        renderFilePreview(fileData, index, fieldId) {
            const fileSizeKB = (fileData.size / 1024).toFixed(2);
            const isImage = fileData.type.startsWith('image/');

            return `
                <div class="esp-file-preview-item">
                    ${isImage ? `
                        <img src="${fileData.preview}"
                            alt="${this.escapeHtml(fileData.name)}"
                            class="esp-file-preview-image">
                    ` : `
                        <div class="esp-file-preview-icon-wrapper">
                            <svg class="w-12 h-12 text-red-600" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z" />
                            </svg>
                        </div>
                    `}

                    <!-- File Info (only for non-images) -->
                    ${!isImage ? `
                        <div class="esp-file-preview-info">
                            <p class="esp-file-preview-name" title="${this.escapeHtml(fileData.name)}">
                                ${this.escapeHtml(fileData.name)}
                            </p>
                            <p class="esp-file-preview-size">
                                ${fileSizeKB} KB
                            </p>
                        </div>
                    ` : ''}

                    <!-- Delete Button Overlay -->
                    <button
                        type="button"
                        class="esp-file-delete-btn esp-file-remove"
                        data-field-id="${fieldId}"
                        data-index="${index}"
                        title="Remove file">
                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                            <path d="M6.28 5.22a.75.75 0 00-1.06 1.06L8.94 10l-3.72 3.72a.75.75 0 101.06 1.06L10 11.06l3.72 3.72a.75.75 0 101.06-1.06L11.06 10l3.72-3.72a.75.75 0 00-1.06-1.06L10 8.94 6.28 5.22z" />
                        </svg>
                    </button>
                </div>
            `;
        }

        /**
         * Update file previews for a field
         * @param {string} fieldId - Field ID
         */
        updateFilePreviews(fieldId) {
            const previewContainer = document.querySelector(`#esp-field-${fieldId}-preview`);
            if (!previewContainer) {
                return;
            }

            const files = this.fileUploads[fieldId] || [];

            if (files.length === 0) {
                previewContainer.style.display = 'none';
                previewContainer.innerHTML = '';
                return;
            }

            // Get validation settings for this field
            const validation = this.fileValidationSettings[fieldId] || this.defaultFileValidation;
            const maxFiles = validation.max_files || 5;

            // Show preview container
            previewContainer.style.display = 'block';

            // Generate previews HTML
            let previewsHtml = `
                <div class="esp-file-preview-container">
                    <p class="esp-file-preview-header">Selected Files: (${files.length}/${maxFiles})</p>
                    <div class="esp-file-preview-grid">
            `;

            files.forEach((fileData, index) => {
                previewsHtml += this.renderFilePreview(fileData, index, fieldId);
            });

            previewsHtml += `
                    </div>
                </div>
            `;

            previewContainer.innerHTML = previewsHtml;

            // Attach remove button event listeners
            const removeButtons = previewContainer.querySelectorAll('.esp-file-remove');
            removeButtons.forEach(button => {
                button.addEventListener('click', (e) => {
                    e.preventDefault();
                    const btnFieldId = button.getAttribute('data-field-id');
                    const index = parseInt(button.getAttribute('data-index'));
                    this.removeFile(btnFieldId, index);
                });
            });

            // Disable/enable file input based on max files
            const fileInput = document.querySelector(`#esp-field-${fieldId}-file-input`);
            if (fileInput) {
                fileInput.disabled = files.length >= maxFiles;
            }
        }

        /**
         * Escape HTML to prevent XSS
         * @param {string} text - Text to escape
         * @returns {string} Escaped text
         */
        escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        /**
         * Log debug messages
         * @param {string} message - Log message
         */
        log(message) {
            if (this.config && this.config.debug) {
                console.log('[ESP Lead Form SDK]', message);
            }
        }
    }

    // Initialize SDK
    const sdk = new ESPLeadFormSDK();

    // Expose to window for geolocation callback
    window.ESPLeadFormSDK = sdk;

})(window, document);
