<?php

namespace Tests\Feature\Api\V1\Lead;

use App\Models\Lead\Lead;
use App\Models\Lead\LeadAddress;
use App\Models\Lead\LeadForm;
use App\Models\Location\City;
use App\Models\Location\Country;
use App\Models\Location\State;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class LeadControllerTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;

    protected function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();
    }

    public function test_can_create_lead_with_location_data(): void
    {
        $leadForm = LeadForm::factory()->create();

        $locationData = [
            'address' => 'Vancouver, BC, Canada',
            'lat' => 49.2827291,
            'lng' => -123.1207375,
            'city' => 'Vancouver',
            'state' => 'British Columbia',
            'state_code' => 'BC',
            'country' => 'Canada',
            'country_code' => 'CA',
        ];

        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/v1/leads', array_merge([
                'form_id' => $leadForm->uuid,
                'first_name' => 'John',
                'last_name' => 'Doe',
                'email' => 'john@example.com',
                'phone' => '1234567890',
                'message' => 'Test message',
                'lead_source' => 'test',
            ], $locationData));

        $response->assertStatus(201)
            ->assertJsonStructure([
                'success',
                'message',
                'data' => [
                    'id',
                    'uuid',
                    'first_name',
                    'last_name',
                    'email',
                    'phone',
                ],
            ]);

        // Verify lead was created
        $this->assertDatabaseHas('leads', [
            'first_name' => 'John',
            'last_name' => 'Doe',
            'email' => 'john@example.com',
        ]);

        $lead = Lead::where('email', 'john@example.com')->first();

        // Verify address was created
        $this->assertDatabaseHas('lead_addresses', [
            'lead_id' => $lead->id,
            'address_line_1' => 'Vancouver, BC, Canada',
            'latitude' => 49.2827291,
            'longitude' => -123.1207375,
        ]);

        // Verify location entities were created
        $country = Country::where('code', 'CA')->first();
        $this->assertNotNull($country);
        $this->assertEquals('Canada', $country->name);

        $state = State::where('country_id', $country->id)
            ->where('code', 'BC')
            ->first();
        $this->assertNotNull($state);
        $this->assertEquals('British Columbia', $state->name);

        $city = City::where('state_id', $state->id)
            ->where('name', 'Vancouver')
            ->first();
        $this->assertNotNull($city);

        // Verify address has correct foreign keys
        $address = LeadAddress::where('lead_id', $lead->id)->first();
        $this->assertEquals($city->id, $address->city_id);
        $this->assertEquals($state->id, $address->state_id);
        $this->assertEquals($country->id, $address->country_id);
    }

    public function test_can_create_lead_without_location_data(): void
    {
        $leadForm = LeadForm::factory()->create();

        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/v1/leads', [
                'form_id' => $leadForm->uuid,
                'first_name' => 'Jane',
                'last_name' => 'Doe',
                'email' => 'jane@example.com',
                'phone' => '9876543210',
                'message' => 'Test message',
                'lead_source' => 'test',
            ]);

        $response->assertStatus(201);

        // Verify lead was created
        $this->assertDatabaseHas('leads', [
            'first_name' => 'Jane',
            'last_name' => 'Doe',
            'email' => 'jane@example.com',
        ]);

        $lead = Lead::where('email', 'jane@example.com')->first();

        // Verify no address was created
        $this->assertDatabaseMissing('lead_addresses', [
            'lead_id' => $lead->id,
        ]);
    }

    public function test_can_create_lead_with_partial_location_data(): void
    {
        $leadForm = LeadForm::factory()->create();

        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/v1/leads', [
                'form_id' => $leadForm->uuid,
                'first_name' => 'Bob',
                'last_name' => 'Smith',
                'email' => 'bob@example.com',
                'phone' => '5555555555',
                'message' => 'Test message',
                'lead_source' => 'test',
                'lat' => 49.2827291,
                'lng' => -123.1207375,
            ]);

        $response->assertStatus(201);

        $lead = Lead::where('email', 'bob@example.com')->first();

        // Verify address was created with coordinates only
        $this->assertDatabaseHas('lead_addresses', [
            'lead_id' => $lead->id,
            'latitude' => 49.2827291,
            'longitude' => -123.1207375,
        ]);

        $address = LeadAddress::where('lead_id', $lead->id)->first();
        $this->assertNull($address->city_id);
        $this->assertNull($address->state_id);
        $this->assertNull($address->country_id);
    }

    public function test_reuses_existing_location_entities(): void
    {
        // Pre-create location entities
        $country = Country::factory()->create([
            'code' => 'CA',
            'name' => 'Canada',
        ]);

        $state = State::factory()->create([
            'country_id' => $country->id,
            'code' => 'BC',
            'name' => 'British Columbia',
        ]);

        $city = City::factory()->create([
            'state_id' => $state->id,
            'country_id' => $country->id,
            'name' => 'Vancouver',
        ]);

        $countryCount = Country::count();
        $stateCount = State::count();
        $cityCount = City::count();

        $leadForm = LeadForm::factory()->create();

        $response = $this->actingAs($this->user, 'sanctum')
            ->postJson('/api/v1/leads', [
                'form_id' => $leadForm->uuid,
                'first_name' => 'Alice',
                'last_name' => 'Wonder',
                'email' => 'alice@example.com',
                'phone' => '1111111111',
                'message' => 'Test message',
                'lead_source' => 'test',
                'address' => 'Vancouver, BC, Canada',
                'lat' => 49.2827291,
                'lng' => -123.1207375,
                'city' => 'Vancouver',
                'state' => 'British Columbia',
                'state_code' => 'BC',
                'country' => 'Canada',
                'country_code' => 'CA',
            ]);

        $response->assertStatus(201);

        // Verify no duplicate location entities were created
        $this->assertEquals($countryCount, Country::count());
        $this->assertEquals($stateCount, State::count());
        $this->assertEquals($cityCount, City::count());

        // Verify address uses existing entities
        $lead = Lead::where('email', 'alice@example.com')->first();
        $address = LeadAddress::where('lead_id', $lead->id)->first();

        $this->assertEquals($city->id, $address->city_id);
        $this->assertEquals($state->id, $address->state_id);
        $this->assertEquals($country->id, $address->country_id);
    }
}
