<?php

namespace Tests\Feature;

use App\Livewire\Frontend\Widgets\LeadForm;
use App\Models\Lead\Lead;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Livewire\Livewire;
use Tests\TestCase;

class LeadFormSubmissionDebugTest extends TestCase
{
    use RefreshDatabase;

    public function test_lead_form_submits_successfully_with_all_steps(): void
    {
        $this->withoutExceptionHandling();

        // Step 1: Contact Info
        $component = Livewire::test(LeadForm::class)
            ->set('firstName', 'John')
            ->set('lastName', 'Doe')
            ->set('emailAddress', 'john@example.com')
            ->set('phoneNumber', '1234567890')
            ->set('propertyRelationship', 'owner')
            ->set('preferredContactWindow', 'morning')
            ->set('authorizeToContactMe', true);

        // Verify step 1 validation passes
        $component->call('nextStep')
            ->assertHasNoErrors()
            ->assertSet('currentStep', 1);

        // Step 2: Well Drilling Details
        $component
            ->set('primaryUse', ['residential'])
            ->set('waterQualityGoals', ['safe_drinking_water', 'remove_hardness']);

        $component->call('nextStep')
            ->assertHasNoErrors()
            ->assertSet('currentStep', 2);

        // Step 3: Property & Timeline
        $component
            ->set('isExistingWaterWell', '0')
            ->set('informationAboutExistingWell', 'No existing well')
            ->set('isNewConstructionProject', '1')
            ->set('desiredProjectTimeline', '2 weeks')
            ->set('wellSiteLocation', 'Near the driveway');

        $component->call('nextStep')
            ->assertHasNoErrors()
            ->assertSet('currentStep', 3);

        // Step 4: Complete System & Submit
        $component
            ->set('completeEstimate', 1)
            ->set('notes', 'Please call before visiting')
            ->set('propertyLocation', 1);

        // Submit the form
        $component->call('submit')
            ->assertHasNoErrors();

        // Verify lead was created
        $this->assertDatabaseHas('leads', [
            'first_name' => 'John',
            'last_name' => 'Doe',
            'email' => 'john@example.com',
            'phone' => '1234567890',
            'lead_source' => 'frontend:lead_form_widget',
        ]);

        // Verify redirect to lead summary
        $lead = Lead::where('email', 'john@example.com')->first();
        $this->assertNotNull($lead);
        $this->assertNotNull($lead->uuid);
    }

    public function test_step_1_validation_prevents_progress_with_missing_fields(): void
    {
        Livewire::test(LeadForm::class)
            ->set('firstName', '')
            ->call('nextStep')
            ->assertHasErrors(['firstName'])
            ->assertSet('currentStep', 0);
    }

    public function test_step_2_validation_requires_primary_use_and_water_quality_goals(): void
    {
        Livewire::test(LeadForm::class)
            // Fill step 1
            ->set('firstName', 'John')
            ->set('lastName', 'Doe')
            ->set('emailAddress', 'john@example.com')
            ->set('phoneNumber', '1234567890')
            ->set('propertyRelationship', 'owner')
            ->set('preferredContactWindow', 'morning')
            ->set('authorizeToContactMe', true)
            ->call('nextStep')
            ->assertSet('currentStep', 1)
            // Try to advance without setting required fields
            ->call('nextStep')
            ->assertHasErrors(['primaryUse', 'waterQualityGoals'])
            ->assertSet('currentStep', 1);
    }
}
