<?php

namespace Tests\Feature;

use App\Enums\LeadStatus;
use App\Livewire\Frontend\Widgets\LeadForm;
use App\Models\Lead\Lead;
use App\Models\Lead\LeadForm as LeadFormModel;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Livewire\Livewire;
use Tests\TestCase;

class LeadFormSubmissionTest extends TestCase
{
    use RefreshDatabase;

    public function test_lead_form_submits_successfully_with_minimum_required_fields(): void
    {
        $leadForm = LeadFormModel::factory()->create();

        Livewire::test(LeadForm::class, ['lead_form_id' => $leadForm->id])
            ->set('firstName', 'John')
            ->set('lastName', 'Doe')
            ->set('emailAddress', 'john@example.com')
            ->set('phoneNumber', '1234567890')
            ->set('propertyRelationship', 'Owner')
            ->set('preferredContactWindow', 'Morning')
            ->set('authorizeToContactMe', true)
            ->set('primaryUse', ['Residential'])
            ->set('waterQualityGoals', ['Clean Water'])
            ->set('isExistingWaterWell', '0')
            ->set('isNewConstructionProject', '0')
            ->set('desiredProjectTimeline', '1 week')
            ->set('wellSiteLocation', 'Backyard')
            ->set('completeEstimate', 1)
            ->set('propertyLocation', 1)
            ->call('submit')
            ->assertRedirect();

        $this->assertDatabaseHas('leads', [
            'first_name' => 'John',
            'last_name' => 'Doe',
            'email' => 'john@example.com',
            'phone' => '1234567890',
            'lead_form_id' => $leadForm->id,
            'lead_source' => 'frontend:lead_form_widget',
        ]);

        $lead = Lead::where('email', 'john@example.com')->first();
        $this->assertNotNull($lead);
        $this->assertEquals(LeadStatus::Pending, $lead->lead_status);
    }

    public function test_lead_form_creates_lead_address_with_location_data(): void
    {
        $leadForm = LeadFormModel::factory()->create();

        $locationData = [
            'address' => '123 Main St',
            'lat' => '49.2827',
            'lng' => '-123.1207',
            'city' => 'Vancouver',
            'state' => 'British Columbia',
            'state_code' => 'BC',
            'country' => 'Canada',
            'country_code' => 'CA',
        ];

        Livewire::test(LeadForm::class, [
            'lead_form_id' => $leadForm->id,
            'locationData' => $locationData,
        ])
            ->set('address', $locationData['address'])
            ->set('lat', $locationData['lat'])
            ->set('lng', $locationData['lng'])
            ->set('city', $locationData['city'])
            ->set('state', $locationData['state'])
            ->set('state_code', $locationData['state_code'])
            ->set('country', $locationData['country'])
            ->set('country_code', $locationData['country_code'])
            ->set('firstName', 'Jane')
            ->set('lastName', 'Smith')
            ->set('emailAddress', 'jane@example.com')
            ->set('phoneNumber', '9876543210')
            ->set('propertyRelationship', 'Owner')
            ->set('preferredContactWindow', 'Afternoon')
            ->set('authorizeToContactMe', true)
            ->set('primaryUse', ['Commercial'])
            ->set('waterQualityGoals', ['Safe Drinking'])
            ->set('isExistingWaterWell', '1')
            ->set('informationAboutExistingWell', 'Has a 50ft well')
            ->set('isNewConstructionProject', '0')
            ->set('desiredProjectTimeline', '2 weeks')
            ->set('wellSiteLocation', 'Front yard')
            ->set('completeEstimate', 1)
            ->set('propertyLocation', 1)
            ->call('submit')
            ->assertRedirect();

        $lead = Lead::where('email', 'jane@example.com')->first();
        $this->assertNotNull($lead);

        $this->assertDatabaseHas('lead_addresses', [
            'lead_id' => $lead->id,
            'address_line_1' => '123 Main St',
            'latitude' => 49.2827,
            'longitude' => -123.1207,
        ]);
    }

    public function test_lead_form_handles_file_uploads(): void
    {
        Storage::fake('public');

        $leadForm = LeadFormModel::factory()->create();

        $file = UploadedFile::fake()->image('test-document.jpg');

        Livewire::test(LeadForm::class, ['lead_form_id' => $leadForm->id])
            ->set('firstName', 'Bob')
            ->set('lastName', 'Builder')
            ->set('emailAddress', 'bob@example.com')
            ->set('phoneNumber', '5555555555')
            ->set('propertyRelationship', 'Tenant')
            ->set('preferredContactWindow', 'Evening')
            ->set('authorizeToContactMe', true)
            ->set('primaryUse', ['Agricultural'])
            ->set('waterQualityGoals', ['Irrigation'])
            ->set('isExistingWaterWell', '0')
            ->set('isNewConstructionProject', '1')
            ->set('desiredProjectTimeline', '3 weeks')
            ->set('wellSiteLocation', 'Side yard')
            ->set('completeEstimate', 1)
            ->set('propertyLocation', 1)
            ->set('attachments', [$file])
            ->call('submit')
            ->assertRedirect();

        $lead = Lead::where('email', 'bob@example.com')->first();
        $this->assertNotNull($lead);
        $this->assertCount(1, $lead->getMedia('attachments'));
    }

    public function test_lead_form_validates_required_fields(): void
    {
        $leadForm = LeadFormModel::factory()->create();

        Livewire::test(LeadForm::class, ['lead_form_id' => $leadForm->id])
            ->set('firstName', '')
            ->set('lastName', '')
            ->set('emailAddress', '')
            ->set('phoneNumber', '')
            ->call('submit')
            ->assertHasErrors(['firstName', 'lastName', 'emailAddress', 'phoneNumber']);
    }

    public function test_lead_form_stores_metadata_correctly(): void
    {
        $leadForm = LeadFormModel::factory()->create();

        Livewire::test(LeadForm::class, ['lead_form_id' => $leadForm->id])
            ->set('firstName', 'Alice')
            ->set('lastName', 'Wonder')
            ->set('emailAddress', 'alice@example.com')
            ->set('phoneNumber', '1112223333')
            ->set('propertyRelationship', 'Owner')
            ->set('preferredContactWindow', 'Morning')
            ->set('authorizeToContactMe', true)
            ->set('primaryUse', ['Residential', 'Agricultural'])
            ->set('waterQualityGoals', ['Clean Water', 'Safe Drinking'])
            ->set('isExistingWaterWell', '1')
            ->set('informationAboutExistingWell', 'Needs repair')
            ->set('isNewConstructionProject', '0')
            ->set('desiredProjectTimeline', '4 weeks')
            ->set('wellSiteLocation', 'Backyard corner')
            ->set('completeEstimate', 1)
            ->set('notes', 'Please call before visiting')
            ->set('propertyLocation', 1)
            ->call('submit')
            ->assertRedirect();

        $lead = Lead::where('email', 'alice@example.com')->first();
        $this->assertNotNull($lead);

        $this->assertEquals('Owner', $lead->getMeta('property_relationship'));
        $this->assertEquals('Morning', $lead->getMeta('preferred_contact_window'));
        $this->assertTrue($lead->getMeta('authorize_to_contact_me'));
        $this->assertEquals(['Residential', 'Agricultural'], $lead->getMeta('primary_use'));
        $this->assertEquals(['Clean Water', 'Safe Drinking'], $lead->getMeta('water_quality_goals'));
        $this->assertEquals('1', $lead->getMeta('is_existing_water_well'));
        $this->assertEquals('Needs repair', $lead->getMeta('information_about_existing_well'));
        $this->assertEquals('Please call before visiting', $lead->getMeta('notes'));
    }
}
