// import Swiper bundle with all modules installed
let activeProfileCopyBtn = null;
import Swiper from "swiper/bundle";
import "flowbite";
import { Modal } from "flowbite";
// Import lightgallery and its plugins
import lightGallery from "lightgallery";
import lgThumbnail from "lightgallery/plugins/thumbnail";
import lgZoom from "lightgallery/plugins/zoom";
// Import our custom Google Maps utilities
import ESPGoogleMaps from "./google-maps";
import { initPhoneInput } from './phone-input';
import { initLeadFormPhoneInput } from './lead-form-phone';
import Quill from 'quill';



/**
 * Job Application Quill Editors Module
 * Handles rich text editor initialization for job application form
 */
const JobApplicationEditors = {
    editors: {},

    init() {
        // Initialize editors when the document is ready
        document.addEventListener('DOMContentLoaded', () => {
            this.initializeEditors();
        });

        // Re-initialize when Livewire navigates
        document.addEventListener('livewire:navigated', () => {
            setTimeout(() => {
                this.initializeEditors();
            }, 100);
        });

        // Handle Livewire morphed event to sync content
        document.addEventListener('livewire:morphed', () => {
            this.syncContentFromLivewire();
        });

        // Re-initialize after Livewire updates (fallback)
        document.addEventListener('livewire:updated', () => {
            setTimeout(() => {
                this.initializeEditors();
            }, 100);
        });
    },

    initializeEditors() {
        // Find all Quill editor containers
        const editorContainers = document.querySelectorAll('[id^="editor-"]');

        editorContainers.forEach(container => {
            const fieldName = container.dataset.field;
            const placeholder = container.dataset.placeholder || 'Enter your text here...';

            if (!fieldName) {
                console.warn('No field name found for editor container:', container);
                return;
            }

            // Skip if already initialized and container still exists
            if (this.editors[fieldName]) {
                // Check if the editor container still exists in DOM
                const existingContainer = document.getElementById(`editor-${fieldName}`);
                if (existingContainer && existingContainer.querySelector('.ql-editor')) {
                    console.log(`Quill editor already initialized for ${fieldName}`);
                    return;
                }
                // If container doesn't exist anymore, remove the editor reference
                delete this.editors[fieldName];
            }

            try {
                // Clear any existing content
                container.innerHTML = '';

                // Initialize Quill editor with simplified toolbar
                const quill = new Quill(container, {
                    theme: 'snow',
                    modules: {
                        toolbar: [
                            ['bold', 'italic', 'underline'],
                            [{ 'list': 'ordered' }, { 'list': 'bullet' }],
                            ['link']
                        ],
                        history: {
                            delay: 1000,
                            maxStack: 50,
                            userOnly: true
                        }
                    },
                    placeholder: placeholder
                });

                // Store editor instance
                this.editors[fieldName] = quill;

                // Set up content synchronization with Livewire
                this.setupLivewireSync(quill, fieldName);

                console.log(`Quill editor initialized for ${fieldName}`);
            } catch (error) {
                console.error(`Failed to initialize Quill editor for ${fieldName}:`, error);
            }
        });
    },

    setupLivewireSync(quill, fieldName) {
        let debounceTimer;

        // Sync content to Livewire when user types (with debouncing)
        quill.on('text-change', (delta, oldDelta, source) => {
            if (source === 'user') {
                const content = quill.root.innerHTML;

                // Update the hidden textarea immediately
                const hiddenInput = document.getElementById(`hidden-${fieldName}`);
                if (hiddenInput) {
                    hiddenInput.value = content;
                }

                // Debounce the blur event for validation (wait 1 second after user stops typing)
                clearTimeout(debounceTimer);
                debounceTimer = setTimeout(() => {
                    if (hiddenInput) {
                        // Trigger blur event for wire:model.blur validation
                        hiddenInput.dispatchEvent(new Event('blur', { bubbles: true }));
                        // Also trigger input event as fallback
                        hiddenInput.dispatchEvent(new Event('input', { bubbles: true }));
                    }
                }, 1000);
            }
        });

        // Also trigger validation when user leaves the editor
        quill.on('selection-change', (range, oldRange, source) => {
            if (oldRange && !range) {
                // Editor lost focus
                const hiddenInput = document.getElementById(`hidden-${fieldName}`);
                if (hiddenInput) {
                    hiddenInput.value = quill.root.innerHTML;
                    hiddenInput.dispatchEvent(new Event('blur', { bubbles: true }));
                }
            }
        });

        // Load initial content if available
        this.syncContentToQuill(fieldName);
    },

    syncContentFromLivewire() {
        // Sync content from Livewire to all Quill editors after morphing
        Object.keys(this.editors).forEach(fieldName => {
            this.syncContentToQuill(fieldName);
        });
    },

    syncContentToQuill(fieldName) {
        const hiddenInput = document.getElementById(`hidden-${fieldName}`);
        const quill = this.editors[fieldName];

        if (hiddenInput && quill) {
            const livewireContent = hiddenInput.value || '';
            const quillContent = quill.root.innerHTML;

            // Only update if content is different to avoid cursor issues
            if (livewireContent !== quillContent) {
                const selection = quill.getSelection(); // Save cursor position
                quill.root.innerHTML = livewireContent;
                if (selection) {
                    quill.setSelection(selection); // Restore cursor position
                }
            }
        }
    },

    destroyEditors() {
        Object.keys(this.editors).forEach(fieldName => {
            if (this.editors[fieldName]) {
                delete this.editors[fieldName];
            }
        });
        this.editors = {};
    }
};
// Initialize Job Application Editors
JobApplicationEditors.init();

// Add custom styles for Quill editors
document.addEventListener('DOMContentLoaded', () => {
    const style = document.createElement('style');
    style.textContent = `
        .quill-editor .ql-toolbar {
            border-top: 1px solid #ccc;
            border-left: 1px solid #ccc;
            border-right: 1px solid #ccc;
            border-bottom: none;
        }
        .quill-editor .ql-container {
            border-bottom: 1px solid #ccc;
            border-left: 1px solid #ccc;
            border-right: 1px solid #ccc;
            border-top: none;
        }
        .quill-editor .ql-editor {
            min-height: 120px;
        }
    `;
    document.head.appendChild(style);
});

/**
 * ESP Directory Map Module
 * Handles map initialization and updates for the member directory
 */
const ESPDirectoryMap = {
    // Global flag to track if the map has been initialized
    initialized: false,

    // Default map settings
    defaults: {
        mapContainer: '#esp-directory-map',
        defaultLat: 43.653226, // Default to Toronto
        defaultLng: -79.3831843,
        defaultZoom: 10,
        minZoom: 0,
        maxZoom: 21,
        markerIcons: {
            default: {
                url: '/images/map-pin2.svg',
                scaledSize: null // Will be set when Google Maps is loaded
            },
            active: {
                url: '/images/map-pin2.svg',
                scaledSize: null // Will be set when Google Maps is loaded
            }
        },
        mapStyles: [
            {
                featureType: "poi",
                elementType: "labels",
                stylers: [{ visibility: "off" }]
            },
            {
                featureType: "transit",
                elementType: "labels",
                stylers: [{ visibility: "on" }]
            },
            {
                featureType: "landscape.man_made",
                elementType: "labels",
                stylers: [{ visibility: "on" }]
            }
        ],
        mapOptions: {
            mapTypeControl: false,       // Removes Map/Satellite toggle
            streetViewControl: false,    // Removes Street View
            fullscreenControl: false,    // Removes fullscreen button
            zoomControl: false,          // Removes zoom controls
            scrollwheel: true,           // Allows scrolling to zoom
            minZoom: 0,                  // Prevents zooming out beyond level 6
            maxZoom: 21                  // Prevents zooming in beyond level 30
        }
    },

    /**
     * Initialize the map once
     * @returns {google.maps.Map|null} The map instance or null if initialization failed
     */
    init: function () {
        // Check if map is already initialized to prevent duplicate initialization
        if (this.initialized) {
            return ESPGoogleMaps.currentMap;
        }

        // Check for required dependencies
        const mapContainer = document.getElementById('esp-directory-map');
        if (!mapContainer) {
            return null;
        }

        if (!window.google || !window.google.maps) {
            return null;
        }

        // Check if we're on the map view page
        if (!document.querySelector('.result-map-sec')) {
            return null;
        }

        // Set up marker icons with proper scale
        this.defaults.markerIcons.default.scaledSize = new google.maps.Size(40, 50);
        this.defaults.markerIcons.active.scaledSize = new google.maps.Size(50, 62);

        // Initialize the map with default settings
        const map = ESPGoogleMaps.initMemberDirectoryMap({
            mapContainer: this.defaults.mapContainer,
            defaultLat: this.defaults.defaultLat,
            defaultLng: this.defaults.defaultLng,
            defaultZoom: this.defaults.defaultZoom,
            defaultMarkerIcon: this.defaults.markerIcons.default,
            activeMarkerIcon: this.defaults.markerIcons.active,
            mapStyles: this.defaults.mapStyles,
            mapOptions: this.defaults.mapOptions,
            onMapLoad: (map) => {
                // Force a resize event to ensure the map renders correctly
                setTimeout(() => {
                    google.maps.event.trigger(map, 'resize');
                }, 100);

                // Set up event listeners for map interaction
                this.setupEventListeners();

                // Set the initialized flag to prevent re-initialization
                this.initialized = true;
            }
        });

        // Force a resize event after a short delay to fix any rendering issues
        setTimeout(() => {
            if (ESPGoogleMaps.currentMap) {
                google.maps.event.trigger(ESPGoogleMaps.currentMap, 'resize');
            }
        }, 500);

        return map;
    },

    /**
     * Update member markers on the map
     * @param {Array} members Array of member objects with location data
     */
    updateMarkers: function (members) {
        // Ensure the map is initialized before trying to add markers
        if (!ESPGoogleMaps.currentMap) {
            this.init();

            // If map initialization failed, return
            if (!ESPGoogleMaps.currentMap) {
                return;
            }

            // Give the map a moment to initialize fully before adding markers
            setTimeout(() => {
                if (ESPGoogleMaps.currentMap) {
                    ESPGoogleMaps.updateMemberMarkers(members, true, true);
                }
            }, 500);
            return;
        }

        // Update the map markers
        ESPGoogleMaps.updateMemberMarkers(members, true, true);
    },

    /**
     * Set up event listeners for map and member interactions
     */
    setupEventListeners: function () {
        // Track currently hovered card to avoid flicker
        let currentHoveredCardId = null;
        let closeInfoWindowTimeout = null;

        // Define the handler function once
        const cardHoverHandler = (card, isEntering) => {
            const memberId = card.dataset.memberId;

            if (isEntering) {
                // Clear any pending timeouts
                if (closeInfoWindowTimeout) {
                    clearTimeout(closeInfoWindowTimeout);
                    closeInfoWindowTimeout = null;
                }

                currentHoveredCardId = memberId;

                // Show info window
                if (ESPGoogleMaps.currentMap && ESPGoogleMaps.currentMarkers && ESPGoogleMaps.currentMarkers.length > 0) {
                    ESPGoogleMaps.highlightMemberMarker(memberId, false);
                }
            } else {
                if (currentHoveredCardId === memberId) { // Only clear if this was the one being hovered
                    currentHoveredCardId = null;
                }

                // Add a small delay before closing to prevent flicker
                closeInfoWindowTimeout = setTimeout(() => {
                    // Only close if we're not currently hovering a card and window wasn't click-opened
                    if (!currentHoveredCardId && ESPGoogleMaps.currentInfoWindow && !ESPGoogleMaps._infoWindowOpenedByClick) {
                        ESPGoogleMaps.currentInfoWindow.close();
                        ESPGoogleMaps.currentInfoWindow = null;

                        // Reset marker icons
                        if (ESPGoogleMaps.currentMarkers && ESPGoogleMaps.currentMarkers.length > 0) {
                            ESPGoogleMaps.currentMarkers.forEach(m => {
                                if (m.content && m.content.tagName === 'IMG') {
                                    const defaultIconUrl = ESPGoogleMaps.getIconUrl(ESPGoogleMaps.defaultMarkerIcon);
                                    if (defaultIconUrl) m.content.src = defaultIconUrl;
                                }
                            });
                        }
                    }
                }, 100); // Small delay to prevent flickering
            }
        };

        // Function to initialize or re-initialize hover handlers for all member cards
        const initCardHoverHandlers = () => {
            document.querySelectorAll('[data-member-id]').forEach(card => {
                // Remove existing listeners to prevent duplication if this function is called multiple times on the same elements
                // A more robust way would be to use a flag or a different approach if elements are not re-rendered by Livewire.
                // However, since Livewire typically replaces elements, this might be okay.
                // For simplicity, we'll rely on Livewire re-rendering or a full re-init.

                // Add enter handler
                card.removeEventListener('mouseenter', card._mouseEnterHandler);
                card._mouseEnterHandler = () => cardHoverHandler(card, true);
                card.addEventListener('mouseenter', card._mouseEnterHandler);

                // Add leave handler
                card.removeEventListener('mouseleave', card._mouseLeaveHandler);
                card._mouseLeaveHandler = () => cardHoverHandler(card, false);
                card.addEventListener('mouseleave', card._mouseLeaveHandler);
            });
        };

        // Call it once to set up initial cards
        initCardHoverHandlers();

        // Store it on the ESPDirectoryMap object so it can be called from Livewire hook if needed
        this.initCardHoverHandlers = initCardHoverHandlers;

        // Keep the click handler for accessibility and mobile devices (this is a global listener)
        // Check if a similar listener already exists to avoid duplication if setupEventListeners is called multiple times.
        // This simple check might not be enough if the exact same function reference isn't used.
        if (!document._espMemberCardClickListener) {
            document._espMemberCardClickListener = function (event) {
                const memberCard = event.target.closest('[data-member-id]');
                if (memberCard) {
                    const memberId = memberCard.dataset.memberId;
                    if (memberId) {
                        document.querySelectorAll('[data-member-id]').forEach(card => {
                            card.classList.remove('active');
                        });
                        memberCard.classList.add('active');

                        if (!ESPGoogleMaps.currentMap || !ESPGoogleMaps.currentMarkers || ESPGoogleMaps.currentMarkers.length === 0) {
                            return;
                        }
                        ESPGoogleMaps.highlightMemberMarker(memberId, true);
                    }
                }
            };
            document.addEventListener('click', document._espMemberCardClickListener);
        }

        // Listen for map-marker-clicked event to highlight corresponding list item
        // This is a global listener, ensure it's only added once.
        if (!document._espMapMarkerClickedListener) {
            document._espMapMarkerClickedListener = function (e) {
                const { markerId, member } = e.detail;

                if (markerId || member) {
                    // Try to find the member card using UUID first, then fallback to ID
                    let memberCard = null;

                    // First try with the markerId (should be UUID now)
                    if (markerId) {
                        memberCard = document.querySelector(`[data-member-id="${markerId}"]`);
                    }

                    // If not found and we have member object, try with member.uuid
                    if (!memberCard && member && member.uuid) {
                        memberCard = document.querySelector(`[data-member-id="${member.uuid}"]`);
                    }

                    // Last resort: try with member.id (in case data-member-id uses ID instead of UUID)
                    if (!memberCard && member && member.id) {
                        memberCard = document.querySelector(`[data-member-id="${member.id}"]`);
                    }

                    console.log('memberCard found:', memberCard, 'markerId:', markerId, 'member:', member);

                    if (memberCard) {
                        // Check if the member card is inside a scrollable container
                        const scrollContainer = memberCard.closest('.overflow-y-auto');

                        if (scrollContainer) {
                            // Calculate the position to scroll to (center the card in view)
                            const containerRect = scrollContainer.getBoundingClientRect();
                            const cardRect = memberCard.getBoundingClientRect();
                            const scrollTop = scrollContainer.scrollTop;
                            const cardTop = cardRect.top - containerRect.top + scrollTop;
                            const cardHeight = cardRect.height;
                            const containerHeight = containerRect.height;

                            // Center the card in the container
                            const scrollToPosition = cardTop - (containerHeight / 2) + (cardHeight / 2);

                            // Smooth scroll to the calculated position
                            scrollContainer.scrollTo({
                                top: scrollToPosition,
                                behavior: 'smooth'
                            });
                        } else {
                            // Fallback to regular scrollIntoView
                            memberCard.scrollIntoView({ behavior: 'smooth', block: 'center' });
                        }

                        // Remove active and blink classes from all cards
                        document.querySelectorAll('[data-member-id]').forEach(card => {
                            card.classList.remove('active');
                            card.classList.remove('member-card-blink');
                        });

                        // Add active class
                        memberCard.classList.add('active');

                        // Add the blinking border class
                        memberCard.classList.add('member-card-blink');

                        // Remove the blinking class after animation completes (1.5 seconds)
                        setTimeout(() => {
                            memberCard.classList.remove('member-card-blink');
                        }, 1500);
                    }
                }
            };
            document.addEventListener('map-marker-clicked', document._espMapMarkerClickedListener);
        }

        // Add mouse events for map markers to show/hide info windows
        // This part needs to be re-evaluated if markers are dynamically added/removed.
        // ESPGoogleMaps.updateMemberMarkers already adds listeners to new markers.
        // So, this loop here might be redundant or target old markers if not cleared properly.
        // The listeners are added AT THE TIME OF MARKER CREATION in ESPGoogleMaps.updateMemberMarkers.
        // Therefore, this section within setupEventListeners for existing ESPGoogleMaps.currentMarkers might be unnecessary
        // or should be part of a marker refresh/rebind process if markers themselves are not recreated.
        // For now, assuming updateMemberMarkers handles new marker listeners correctly.
    },

    /**
     * Debug utilities for the map
     */
    debug: {
        // Show the current map status
        status: function () {
            return {
                initialized: ESPDirectoryMap.initialized,
                mapObject: ESPGoogleMaps.currentMap,
                markerCount: ESPGoogleMaps.currentMarkers.length
            };
        },

        // Force map re-initialization
        reinitMap: function () {
            ESPDirectoryMap.initialized = false;
            if (ESPGoogleMaps.currentMap) {
                // Try to clean up old map
                try {
                    ESPGoogleMaps.clearMarkers();
                    ESPGoogleMaps.currentMap = null;
                } catch (e) {
                    // Error handling
                }
            }
            // return ESPDirectoryMap.init();
        },

        // Add test markers for debugging
        addTestMarkers: function () {
            const testMarkers = [
                {
                    id: 'test1',
                    uuid: 'test1',
                    name: 'Test Marker 1',
                    title: 'Test Marker 1',
                    city: 'Toronto',
                    state: 'Ontario',
                    country: 'Canada',
                    lat: 43.653226,
                    lng: -79.3831843,
                    rating: 4.5,
                    total_rating_count: 10
                },
                {
                    id: 'test2',
                    uuid: 'test2',
                    name: 'Test Marker 2',
                    title: 'Test Marker 2',
                    city: 'Vancouver',
                    state: 'British Columbia',
                    country: 'Canada',
                    lat: 49.2827,
                    lng: -123.1207,
                    rating: 4.0,
                    total_rating_count: 5
                }
            ];

            ESPDirectoryMap.updateMarkers(testMarkers);
            return testMarkers;
        }
    }
};

// Export the map module to the window for global access
window.ESPDirectoryMap = ESPDirectoryMap;

// Function to initialize sidebar search and filter functionality
window.initSidebarSearch = function () {
    // Find all search sidebar components on the page
    const searchSidebars = document.querySelectorAll('[id^="search-sidebar-"]');
    if (searchSidebars.length > 0) {
        searchSidebars.forEach(sidebar => {
            const componentId = sidebar.dataset.componentId;
            const country = sidebar.dataset.country || '';

            // Only initialize if Google Maps is loaded and we have a component ID
            if (componentId && window.google && window.google.maps) {
                // Initialize the Google Maps autocomplete for this sidebar
                ESPGoogleMaps.initSearchSidebarAutocomplete(componentId, country);
            }
        });
    }
};


// Function to initialize Classifieds Swiper that can be called when components load
window.initClassifiedsSwiper = function () {
    const classifiedSwipers = document.querySelectorAll(".ClassifieldsSwiper");
    if (classifiedSwipers.length > 0) {
        classifiedSwipers.forEach(element => {
            // If there's an existing swiper instance, destroy it first
            if (element.swiper) {
                element.swiper.destroy(true, true);
            }

            // Check if we have slides
            const slides = element.querySelectorAll('.swiper-slide');
            if (slides.length === 0) {
                return;
            }

            // Create a new swiper instance
            try {
                new Swiper(element, {
                    pagination: {
                        el: ".swiper-pagination",
                        clickable: true,
                    },
                });
            } catch (error) {
                console.error('Error initializing Swiper:', error);
            }
        });
    }
};

// Function to initialize Classifieds Swiper that can be called when components load
window.initClassifiedDetailSwiper = function () {
    const classifiedSwipers = document.querySelectorAll(".classifiedDetailSwiper");
    if (classifiedSwipers.length > 0) {
        classifiedSwipers.forEach(element => {
            // If there's an existing swiper instance, destroy it first
            if (element.swiper) {
                element.swiper.destroy(true, true);
            }

            // Check if we have slides
            const slides = element.querySelectorAll('.swiper-slide');
            if (slides.length === 0) {
                return;
            }

            // Create a new swiper instance
            try {
                new Swiper(element, {
                    loop: true,
                    navigation: {
                        nextEl: ".swiper-button-next",
                        prevEl: ".swiper-button-prev",
                    },
                    pagination: {
                        el: ".swiper-pagination23",
                        clickable: true,
                    },
                });
            } catch (error) {
                console.error('Error initializing Swiper:', error);
            }
        });
    }
};
// Function to initialize Flash Classifieds Swiper that can be called when components load
window.initFlashClassifiedsSwiper = function () {
    const classifiedSwipers = document.querySelectorAll(".flashClassifieldsSwiper");
    if (classifiedSwipers.length > 0) {
        classifiedSwipers.forEach(element => {
            // If there's an existing swiper instance, destroy it first
            if (element.swiper) {
                element.swiper.destroy(true, true);
            }

            // Check if we have slides
            const slides = element.querySelectorAll('.swiper-slide');
            if (slides.length === 0) {
                return;
            }

            // Create a new swiper instance
            try {
                new Swiper(element, {
                    pagination: {
                        el: ".swiper-pagination",
                        clickable: true,
                    },
                });
            } catch (error) {
                console.error('Error initializing Swiper:', error);
            }
        });
    }
};

window.initHomeClassifiedsSwiper = function () {
    console.log('initHomeClassifiedsSwiper');
    const classifiedSwipers = document.querySelectorAll(".homeClassifiedsSwiper");
    if (classifiedSwipers.length > 0) {
        classifiedSwipers.forEach(element => {
            // If there's an existing swiper instance, destroy it first
            if (element.swiper) {
                element.swiper.destroy(true, true);
            }

            // Check if we have slides
            const slides = element.querySelectorAll('.swiper-slide');
            if (slides.length === 0) {
                return;
            }

            // Create a new swiper instance
            try {
                new Swiper(element, {
                    slidesPerView: 3,
                    spaceBetween: 24,
                    breakpoints: {
                        // when window width is >= 320px
                        320: {
                            slidesPerView: 1.1,
                            spaceBetween: 12,
                        },
                        // when window width is >= 640px
                        768: {
                            slidesPerView: 2,
                            spaceBetween: 24,
                        },
                        992: {
                            slidesPerView: 3,
                            spaceBetween: 24,
                        },
                    },
                });
            } catch (error) {
                console.error('Error initializing Swiper:', error);
            }
        });
    }
};

window.initHomeJobSwiper = function () {
    const homeJobSwipers = document.querySelectorAll(".homeJobSwiper");
    if (homeJobSwipers.length > 0) {
        homeJobSwipers.forEach(element => {
            // If there's an existing swiper instance, destroy it first
            if (element.swiper) {
                element.swiper.destroy(true, true);
            }

            // Check if we have slides
            const slides = element.querySelectorAll('.swiper-slide');
            if (slides.length === 0) {
                return;
            }

            // Create a new swiper instance
            try {
                new Swiper(element, {
                    slidesPerView: 1.1,
                    spaceBetween: 12,
                    loop: true,
                });
            } catch (error) {
                console.error('Error initializing Swiper:', error);
            }
        });
    }
};
window.initHomeServicesSwiper = function () {
    const homeServicesSwipers = document.querySelectorAll(".homeServicesSwiper");
    if (homeServicesSwipers.length > 0) {
        homeServicesSwipers.forEach(element => {
            // If there's an existing swiper instance, destroy it first
            if (element.swiper) {
                element.swiper.destroy(true, true);
            }

            // Check if we have slides
            const slides = element.querySelectorAll('.swiper-slide');
            if (slides.length === 0) {
                return;
            }

            // Create a new swiper instance
            try {
                new Swiper(element, {
                    slidesPerView: 2.12,
                    spaceBetween: 8,
                    loop: true,
                    pagination: {
                        el: ".swiper-pagination456",
                        clickable: true,
                    },
                });
            } catch (error) {
                console.error('Error initializing Swiper:', error);
            }
        });
    }
};

window.initTopRatedHomeSwiper = function () {
    const swiper = new Swiper(".espSwiper", {
        spaceBetween: 65,
        loop: true,
        navigation: {
            nextEl: ".swiper-button-next3",
            prevEl: ".swiper-button-prev3",
        },
    });
}
// Initialize Home Top Members Tabs (activates tabs without Livewire props)
window.initHomeMemberTabs = function () {
    try {
        const tabList = document.getElementById('home-member-tabs');
        const tabButtons = document.querySelectorAll('#home-member-tabs button[role="tab"]');
        const tabContents = document.querySelectorAll('#home-member-tabs-content > div[role="tabpanel"]');

        if (!tabList || tabButtons.length === 0 || tabContents.length === 0) {
            return;
        }

        const storageKey = 'home-member-tabs-active';
        const activeClasses = (tabList.dataset.tabsActiveClasses || '').split(' ').filter(Boolean);
        const inactiveClasses = (tabList.dataset.tabsInactiveClasses || '').split(' ').filter(Boolean);

        const applyActiveState = (button) => {
            tabButtons.forEach(btn => {
                btn.setAttribute('aria-selected', 'false');
                if (inactiveClasses.length) btn.classList.add(...inactiveClasses);
                if (activeClasses.length) btn.classList.remove(...activeClasses);
            });

            button.setAttribute('aria-selected', 'true');
            if (activeClasses.length) button.classList.add(...activeClasses);
            if (inactiveClasses.length) button.classList.remove(...inactiveClasses);
        };

        const switchPanel = (targetId) => {
            tabContents.forEach(panel => panel.classList.add('hidden'));
            const target = document.querySelector(targetId);
            if (target) target.classList.remove('hidden');
        };

        // Attach click listeners (idempotent)
        tabButtons.forEach(button => {
            const targetId = button.getAttribute('data-tabs-target');
            const handler = (e) => {
                e.preventDefault();
                applyActiveState(button);
                if (targetId) switchPanel(targetId);
                try { sessionStorage.setItem(storageKey, targetId || ''); } catch (e) { }
            };
            button.removeEventListener('click', button._homeMemberTabsHandler);
            button._homeMemberTabsHandler = handler;
            button.addEventListener('click', handler);
        });

        // Ensure initial state: prefer last saved target, else aria-selected="true", else first
        let savedTargetId = null;
        try { savedTargetId = sessionStorage.getItem(storageKey); } catch (e) { savedTargetId = null; }
        let initialButton = null;
        if (savedTargetId) {
            initialButton = Array.from(tabButtons).find(b => b.getAttribute('data-tabs-target') === savedTargetId) || null;
        }
        if (!initialButton) {
            initialButton = Array.from(tabButtons).find(b => b.getAttribute('aria-selected') === 'true') || tabButtons[0];
        }
        if (initialButton) {
            applyActiveState(initialButton);
            const targetId = initialButton.getAttribute('data-tabs-target');
            if (targetId) switchPanel(targetId);
        }
    } catch (e) {
        // Silent error handling
    }
}


// Helpers to mark a tab as loaded and permanently remove its first-load spinner
window.markHomeMemberTabLoaded = function (targetId) {
    if (!targetId) return;
    try {
        const storageKey = 'home-member-tabs-loaded';
        let loaded = [];
        try { loaded = JSON.parse(sessionStorage.getItem(storageKey) || '[]'); } catch (e) { loaded = []; }
        if (!Array.isArray(loaded)) loaded = [];
        if (!loaded.includes(targetId)) {
            loaded.push(targetId);
            try { sessionStorage.setItem(storageKey, JSON.stringify(loaded)); } catch (e) { }
        }

        const panel = document.querySelector(targetId);
        if (panel) {
            const spinner = panel.querySelector('[id="tabLoading"]');
            if (spinner && spinner.parentNode) {
                // Remove the spinner node so it won't appear again for this panel
                spinner.parentNode.removeChild(spinner);
            }
        }
    } catch (e) {
        // Silent error handling
    }
}
// Function to initialize Gallery Swiper that can be called when components load
window.initGallerySwiper = function () {
    const gallerySwipers = document.querySelectorAll(".gallerySwiper");

    if (gallerySwipers.length > 0) {
        gallerySwipers.forEach((element, index) => {
            // If there's an existing swiper instance, destroy it first
            if (element.swiper) {
                element.swiper.destroy(true, true);
            }

            // Check if we have slides
            const slides = element.querySelectorAll('.swiper-slide');

            if (slides.length === 0) {
                return;
            }

            try {
                // Use exact original configuration - simple and working
                const swiper = new Swiper(element, {
                    loop: slides.length > 1, // Only loop if more than 1 slide
                    // navigation: {
                    //     nextEl: ".swiper-button-next",
                    //     prevEl: ".swiper-button-prev",
                    // },
                    // pagination: {
                    //     el: ".swiper-pagination",
                    //     type: "fraction",
                    // },
                    navigation: {
                        nextEl: ".swiper-button-next",
                        prevEl: ".swiper-button-prev",
                    },
                    pagination: {
                        el: ".swiper-pagination23",
                        clickable: true,
                    },
                });
            } catch (error) {
                // Silent error handling
            }
        });
    }
};

// Debug helper for manual testing
window.debugGallerySwiper = function () {
    const elements = document.querySelectorAll('.gallerySwiper');
    elements.forEach((el, index) => {
        // Debug info available via browser dev tools if needed
    });
};

// Function to initialize profile gallery lightGallery
window.initProfileGallery = function () {
    const profileGallery = document.getElementById('profile-gallery');
    if (profileGallery) {
        // Destroy existing lightGallery instance if it exists
        if (profileGallery.lgInstance) {
            profileGallery.lgInstance.destroy();
        }

        try {
            // Initialize lightGallery for profile gallery
            const lgInstance = lightGallery(profileGallery, {
                plugins: [lgZoom, lgThumbnail],
                speed: 500,
                thumbnail: true,
                selector: '.profile-gallery-item',
                download: true,
                counter: true,
                enableDrag: true,
                enableSwipe: true,
                closable: true,
                loop: true,
                escKey: true,
                keyPress: true,
                thumbWidth: 100,
                thumbHeight: 80,
                thumbMargin: 5,
                showThumbByDefault: false,
                toogleThumb: true,
                pullCaptionUp: true,
                enableThumbDrag: true,
                enableThumbSwipe: true,
                swipeThreshold: 50
            });
        } catch (error) {
            // Silent error handling
        }
    }
};
//classified gallery
window.initClassifiedGallery = function () {

    const classifiedGallery = document.getElementById('classified-gallery');
    if (classifiedGallery) {
        // Destroy existing lightGallery instance if it exists
        if (classifiedGallery.lgInstance) {
            classifiedGallery.lgInstance.destroy();
        }

        try {
            // Initialize lightGallery for profile gallery
            const lgInstance = lightGallery(classifiedGallery, {
                plugins: [lgZoom, lgThumbnail],
                speed: 500,
                thumbnail: true,
                selector: '.classified-gallery-item',
                download: true,
                counter: true,
                enableDrag: true,
                enableSwipe: true,
                closable: true,
                loop: true,
                escKey: true,
                keyPress: true,
                thumbWidth: 100,
                thumbHeight: 80,
                thumbMargin: 5,
                showThumbByDefault: false,
                toogleThumb: true,
                pullCaptionUp: true,
                enableThumbDrag: true,
                enableThumbSwipe: true,
                swipeThreshold: 50
            });
        } catch (error) {
            // Silent error handling
        }
    }
};
//project gallery
window.initProjectGallery = function () {
    const projectGallery = document.getElementById('project-gallery');
    if (projectGallery) {
        // Destroy existing lightGallery instance if it exists
        if (projectGallery.lgInstance) {
            projectGallery.lgInstance.destroy();
        }

        try {
            // Initialize lightGallery for profile gallery
            const lgInstance = lightGallery(projectGallery, {
                plugins: [lgZoom, lgThumbnail],
                speed: 500,
                thumbnail: true,
                selector: '.project-gallery-item',
                download: true,
                counter: true,
                enableDrag: true,
                enableSwipe: true,
                closable: true,
                loop: true,
                escKey: true,
                keyPress: true,
                thumbWidth: 100,
                thumbHeight: 80,
                thumbMargin: 5,
                showThumbByDefault: false,
                toogleThumb: true,
                pullCaptionUp: true,
                enableThumbDrag: true,
                enableThumbSwipe: true,
                swipeThreshold: 50
            });
        } catch (error) {
            // Silent error handling
            alert(error);
        }
    }
};
window.initEquipmentGallery = function () {
    const projectGallery = document.getElementById('equipment-gallery');
    if (projectGallery) {
        // Destroy existing lightGallery instance if it exists
        if (projectGallery.lgInstance) {
            projectGallery.lgInstance.destroy();
        }

        try {
            // Initialize lightGallery for profile gallery
            const lgInstance = lightGallery(projectGallery, {
                plugins: [lgZoom, lgThumbnail],
                speed: 500,
                thumbnail: true,
                selector: '.equipment-gallery-item',
                download: true,
                counter: true,
                enableDrag: true,
                enableSwipe: true,
                closable: true,
                loop: true,
                escKey: true,
                keyPress: true,
                thumbWidth: 100,
                thumbHeight: 80,
                thumbMargin: 5,
                showThumbByDefault: false,
                toogleThumb: true,
                pullCaptionUp: true,
                enableThumbDrag: true,
                enableThumbSwipe: true,
                swipeThreshold: 50
            });
        } catch (error) {
            // Silent error handling
            alert(error);
        }
    }
};

//review gallery
window.initReviewGallery = function () {
    const reviewGalleries = document.querySelectorAll('.review-gallery');
    if (reviewGalleries.length > 0) {
        reviewGalleries.forEach((reviewGallery) => {
            // Destroy existing lightGallery instance if it exists
            if (reviewGallery.lgInstance) {
                reviewGallery.lgInstance.destroy();
            }

            try {
                // Initialize lightGallery for each review gallery list
                lightGallery(reviewGallery, {
                    plugins: [lgZoom, lgThumbnail],
                    speed: 500,
                    thumbnail: true,
                    selector: '.review-gallery-item',
                    download: true,
                    counter: true,
                    enableDrag: true,
                    enableSwipe: true,
                    closable: true,
                    loop: true,
                    escKey: true,
                    keyPress: true,
                    thumbWidth: 100,
                    thumbHeight: 80,
                    thumbMargin: 5,
                    showThumbByDefault: false,
                    toogleThumb: true,
                    pullCaptionUp: true,
                    enableThumbDrag: true,
                    enableThumbSwipe: true,
                    swipeThreshold: 50
                });
            } catch (error) {
                // Silent error handling
            }
        });
    }
}
//member feature gallery
window.initMemberFeatureSwiper = function () {
    // Support multiple swiper instances
    document.querySelectorAll(".memberFeatureSwiper").forEach((element) => {
        // Destroy existing instance if reinitializing
        if (element.swiper) {
            element.swiper.destroy(true, true);
        }

        // Read desktop slides from data attribute, default to 4
        const desktopSlides = parseInt(element.dataset.desktopSlides) || 4;

        new Swiper(element, {
            autoplay: {
                delay: 1000,
                disableOnInteraction: false,
            },
            spaceBetween: 16,
            loop: true,
            centeredSlides: true,
            initialSlide: 2,
            navigation: {
                nextEl: ".swiper-button-next2",
                prevEl: ".swiper-button-prev2",
            },
            breakpoints: {
                // when window width is >= 320px
                320: {
                    slidesPerView: 1.5,
                    spaceBetween: 20,
                    centeredSlides: false
                },
                // when window width is >= 480px
                576: {
                    slidesPerView: 3,
                    spaceBetween: 16,
                    centeredSlides: true,
                    initialSlide: 1,
                },
                // when window width is >= 640px
                768: {
                    slidesPerView: 4.25,
                    spaceBetween: 16,
                    initialSlide: 2,
                },
                992: {
                    slidesPerView: 5.25,
                    spaceBetween: 16,
                    initialSlide: 2,
                },
            },
        });
    });

}
//our partners swiper
window.initHomePartnersSwiper = function () {
    // Support multiple swiper instances
    document.querySelectorAll(".purposeSwiper").forEach((element) => {
        // Destroy existing instance if reinitializing
        if (element.swiper) {
            element.swiper.destroy(true, true);
        }

        // Read desktop slides from data attribute, default to 4
        const desktopSlides = parseInt(element.dataset.desktopSlides) || 4;

        new Swiper(element, {
            loop: true,
            effect: "coverflow",
            grabCursor: true,
            centeredSlides: true,
            initialSlide: 3,
            coverflowEffect: {
                rotate: 0,
                stretch: 0,
                depth: 300,
                modifier: 1,
                slideShadows: true,
            },
            navigation: {
                nextEl: ".swiper-button-next",
                prevEl: ".swiper-button-prev",
            },
            breakpoints: {
                // mobile
                0: {
                    slidesPerView: 1.4,
                    spaceBetween: 0,
                    coverflowEffect: {
                        depth: 500,
                    },
                },
                576: {
                    slidesPerView: 2.5,
                    spaceBetween: 0,

                },
                767: {
                    slidesPerView: 2.5,
                    spaceBetween: 0,
                },
                // large tablets
                992: {
                    slidesPerView: 3,
                    spaceBetween: 0,
                },

            },

        });
    });
}
window.initHomeMonthlyPackageSwiper = function () {
    // Support multiple swiper instances
    document.querySelectorAll(".homeMonthlyPackageSwiper").forEach((element) => {
        // Destroy existing instance if reinitializing
        if (element.swiper) {
            element.swiper.destroy(true, true);
        }

        // Read desktop slides from data attribute, default to 4
        const desktopSlides = parseInt(element.dataset.desktopSlides) || 4;

        new Swiper(element, {
            pagination: {
                el: ".swiper-pagination-month",
                clickable: true,
            },
            breakpoints: {
                320: {
                    slidesPerView: 1.1,
                    spaceBetween: 12,
                },
                // when window width is >= 640px
                992: {
                    slidesPerView: 3,
                    spaceBetween: 24,
                },
                1200: {
                    slidesPerView: 4,
                    spaceBetween: 24,
                },
            },

        });
    });
}
window.initHomeYearlyPackageSwiper = function () {
    // Support multiple swiper instances
    document.querySelectorAll(".homeYearlyPackageSwiper").forEach((element) => {
        // Destroy existing instance if reinitializing
        if (element.swiper) {
            element.swiper.destroy(true, true);
        }

        // Read desktop slides from data attribute, default to 4
        const desktopSlides = parseInt(element.dataset.desktopSlides) || 4;

        new Swiper(element, {
            pagination: {
                el: ".swiper-pagination-year",
                clickable: true,
            },
            breakpoints: {
                // when window width is >= 320px
                320: {
                    slidesPerView: 1.1,
                    spaceBetween: 12,
                },
                // when window width is >= 640px
                992: {
                    slidesPerView: 3,
                    spaceBetween: 24,
                },
                1200: {
                    slidesPerView: 4,
                    spaceBetween: 24,
                },
            },

        });
    });
}

//member Featured Gallery
window.initMemberFeatureGallery = function () {
    const projectGallery = document.getElementById('member-feature-gallery');
    if (projectGallery) {
        // Destroy existing lightGallery instance if it exists
        if (projectGallery.lgInstance) {
            projectGallery.lgInstance.destroy();
        }

        try {
            // Initialize lightGallery for profile gallery
            const lgInstance = lightGallery(projectGallery, {
                plugins: [lgZoom, lgThumbnail],
                speed: 500,
                thumbnail: true,
                selector: '.member-feature-gallery-item',
                download: true,
                counter: true,
                enableDrag: true,
                enableSwipe: true,
                closable: true,
                loop: true,
                escKey: true,
                keyPress: true,
                thumbWidth: 100,
                thumbHeight: 80,
                thumbMargin: 5,
                showThumbByDefault: false,
                toogleThumb: true,
                pullCaptionUp: true,
                enableThumbDrag: true,
                enableThumbSwipe: true,
                swipeThreshold: 50
            });
        } catch (error) {
            // Silent error handling
            alert(error);
        }
    }
}

// Function to open profile gallery programmatically
window.openProfileGallery = function () {
    const firstGalleryItem = document.querySelector('#profile-gallery .profile-gallery-item');
    if (firstGalleryItem) {
        firstGalleryItem.click();
    }
};


window.openClassifiedGallery = function () {
    const firstGalleryItem = document.querySelector('#classified-gallery .classified-gallery-item');
    if (firstGalleryItem) {
        firstGalleryItem.click();
    }
};

window.openProjectGallery = function () {
    const firstGalleryItem = document.querySelector('#project-gallery .project-gallery-item');
    if (firstGalleryItem) {
        firstGalleryItem.click();
    }
};

window.openEquipmentGallery = function () {
    const firstGalleryItem = document.querySelector('#equipment-gallery .equipment-gallery-item');
    if (firstGalleryItem) {
        firstGalleryItem.click();
    }
};

// For backward compatibility
window.initMemberDirectoryMap = function () {
    return ESPDirectoryMap.init();
};

window.updateMemberMarkers = function (members) {
    try {
        if (!members) {
            return;
        }

        let membersArray;

        // Check if members is already an array
        if (Array.isArray(members)) {
            membersArray = members;
        }
        // Check if members is an object, convert to array
        else if (typeof members === 'object' && members !== null) {
            // If it's an object, try to convert to array
            // Check if it has array-like properties or convert object values to array
            if (members.length !== undefined) {
                // Object with length property (array-like object)
                membersArray = Array.from(members);
            } else {
                // Regular object - convert values to array
                membersArray = Object.values(members);
            }
        }
        else {
            return;
        }

        // Validate that we have a proper array
        if (!Array.isArray(membersArray)) {
            return;
        }

        return ESPDirectoryMap.updateMarkers(membersArray);
    } catch (error) {
        return null;
    }
};

window.debugMap = ESPDirectoryMap.debug;

/**
 * Copy profile link to clipboard with custom UI feedback for profile sharing widget
 * @param {string} url - The URL to copy to clipboard
 */
function copyProfileLinkToClipboard(url) {
    // Try using the modern clipboard API first
    if (navigator.clipboard && navigator.clipboard.writeText) {
        navigator.clipboard.writeText(url).then(() => {
            showProfileCopyFeedback();
        }).catch(err => {
            fallbackCopyProfileText(url);
        });
    } else {
        // Fallback for older browsers
        fallbackCopyProfileText(url);
    }
}

/**
 * Fallback copy method for profile sharing widget
 * @param {string} text - The text to copy
 */
function fallbackCopyProfileText(text) {
    const textArea = document.createElement('textarea');
    textArea.value = text;
    textArea.style.position = 'fixed';
    textArea.style.left = '-999999px';
    textArea.style.top = '-999999px';
    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();

    try {
        // @ts-ignore - Using deprecated execCommand as fallback for older browsers
        const successful = document.execCommand('copy');
        if (successful) {
            showProfileCopyFeedback();
        } else {
            showProfileCopyError();
        }
    } catch (err) {
        showProfileCopyError();
    }

    document.body.removeChild(textArea);
}

/**
 * Show feedback when profile URL is successfully copied
 */
function showProfileCopyFeedback() {
    const copyBtn = activeProfileCopyBtn;
    const copyText = copyBtn ? copyBtn.querySelector('.copy-link-text') : null;

    if (copyBtn && copyText) {
        const originalText = copyText.textContent;
        copyText.textContent = 'Copied!';
        copyBtn.classList.add('bg-green-500', 'text-white', 'border-green-500');
        copyBtn.classList.remove('text-primary-blue', 'border-primary-blue');

        setTimeout(() => {
            copyText.textContent = originalText;
            copyBtn.classList.remove('bg-green-500', 'text-white', 'border-green-500');
            copyBtn.classList.add('text-primary-blue', 'border-primary-blue');
        }, 2000);
    }
}

/**
 * Show error feedback when profile copy fails
 */
function showProfileCopyError() {
    const copyBtn = activeProfileCopyBtn;
    const copyText = copyBtn ? copyBtn.querySelector('.copy-link-text') : null;

    if (copyBtn && copyText) {
        const originalText = copyText.textContent;
        copyText.textContent = 'Failed to copy!';
        copyBtn.classList.add('bg-red-500', 'text-white', 'border-red-500');
        copyBtn.classList.remove('text-primary-blue', 'border-primary-blue');

        setTimeout(() => {
            copyText.textContent = originalText;
            copyBtn.classList.remove('bg-red-500', 'text-white', 'border-red-500');
            copyBtn.classList.add('text-primary-blue', 'border-primary-blue');
        }, 2000);
    }
}

// Make the function globally accessible
window.copyProfileLinkToClipboard = copyProfileLinkToClipboard;

// Global click event handler for all clickable elements
document.addEventListener("click", function (event) {
    const profileCopyBtn = event.target.closest('.copy-link-btn');
    if (profileCopyBtn) {
        event.preventDefault();
        activeProfileCopyBtn = profileCopyBtn;
        const url = profileCopyBtn.dataset.url;
        if (navigator.clipboard && navigator.clipboard.writeText) {
            navigator.clipboard.writeText(url).then(() => {
                showProfileCopyFeedback();
            }).catch(err => {
                fallbackCopyProfileText(url);
            });
        } else {
            fallbackCopyProfileText(url);
        }
        return;
    }

    // Handle nav-item clicks
    const navItem = event.target.closest('a.nav-item');
    if (navItem) {
        // Remove the active class from all anchors
        document.querySelectorAll('a.nav-item').forEach(item => {
            item.classList.remove('active');
        });
        // Add the active class to the clicked anchor
        navItem.classList.add('active');
    }

    // Handle menu button clicks
    const menuBtn = event.target.closest('.menu-btn');
    if (menuBtn) {
        menuBtn.classList.toggle("active");
        document.body.classList.toggle("active");
    }

    // Handle save button clicks
    const saveBtn = event.target.closest('.save-btn');
    if (saveBtn) {
        saveBtn.classList.toggle("active");
    }

    // Handle bookmark button clicks
    const bookmarkBtn = event.target.closest('.bookmark-btn');
    if (bookmarkBtn) {
        bookmarkBtn.classList.toggle("active");
    }

    // Handle share button clicks
    const shareBtn = event.target.closest('[data-share_type]');
    if (shareBtn) {
        const type = shareBtn.dataset.share_type;
        const url = shareBtn.dataset.url;

        if (type && url) {
            updateShareModal(type, url);
        }
    }

    // Handle copy button clicks in share modal
    const copyBtn = event.target.closest('#copy');
    if (copyBtn) {
        event.preventDefault();
        copyToClipboard();
    }

    // Handle social share button clicks
    const socialShareBtn = event.target.closest('.social-share-btn');
    if (socialShareBtn) {
        event.preventDefault();
        const platform = socialShareBtn.dataset.platform;
        const url = document.getElementById('text')?.textContent?.trim();

        if (platform && url) {
            handleSocialShare(platform, url);
        }
    }
});

// Marquee
window.initMarquee = function () {
    const container = document.querySelector(".partner-widget-marquee-inner");
    const wrapper = document.querySelector(".partner-widget-marquee-wrapper");

    // Only initialize if marquee elements exist on the page
    if (!container || !wrapper) {
        return;
    }

    const clones = container.cloneNode(true);
    container.appendChild(clones);

    let scrollAmount = 0;
    let isPaused = false;

    function marqueeScroll() {
        if (!isPaused) {
            scrollAmount += 1; // speed
            container.style.transform = `translateX(-${scrollAmount}px)`;
            if (scrollAmount >= container.scrollWidth / 2) {
                scrollAmount = 0;
            }
        }
        requestAnimationFrame(marqueeScroll);
    }
    marqueeScroll();

    wrapper.addEventListener("mouseenter", () => isPaused = true);
    wrapper.addEventListener("mouseleave", () => isPaused = false);
}

// Handle call button mouseenter events using event delegation
document.addEventListener("mouseenter", function (event) {
    const callButton = event.target.closest('.call-btn');
    if (callButton) {
        const callUs = callButton.querySelector('.call-us');
        const callNumberBtn = callButton.querySelector('.call-number-btn');

        callButton.classList.add('group');
        if (callUs) callUs.classList.add('hidden');
        if (callNumberBtn) {
            callNumberBtn.classList.remove('hidden');
            callNumberBtn.classList.add('flex');
        }
    }
}, true);

// Handle call button mouseleave events using event delegation
document.addEventListener("mouseleave", function (event) {
    const callButton = event.target.closest('.call-btn');
    if (callButton) {
        const callUs = callButton.querySelector('.call-us');
        const callNumberBtn = callButton.querySelector('.call-number-btn');

        callButton.classList.remove('group');
        if (callUs) callUs.classList.remove('hidden');
        if (callNumberBtn) {
            callNumberBtn.classList.add('hidden');
            callNumberBtn.classList.remove('flex');
        }
    }
}, true);



let lastScrollTop = 0;
const bottomAction = document.querySelector(".bottom-action");

window.addEventListener(
    "scroll",
    function () {
        if (window.innerWidth > 767) return; // Ignore for desktop (larger screens)

        let currentScroll =
            window.pageYOffset || document.documentElement.scrollTop;

        if (currentScroll > lastScrollTop) {
            // Scrolling down
            bottomAction.classList.add("hide");
        } else {
            // Scrolling up
            bottomAction.classList.remove("hide");
        }

        lastScrollTop = currentScroll <= 0 ? 0 : currentScroll; // For Safari
    },
    false
);

// ------------------ Review Write Modal ------------------

window.initReviewWriteModal = function () {
    // The review modal is now fully controlled by Livewire
    // This function is kept for compatibility but functionality is handled via Alpine.js and Livewire
    // console.log('Review Write Modal initialized with Livewire');
}
// End Review Write Modal

// ------------------ Tab Navigation ------------------
function initTabNavigation() {
    const container = document.querySelector(".scroll-container");
    const arrowRight = document.querySelector("#arrowRight");
    const arrowLeft = document.querySelector("#arrowLeft");
    const shadow = document.querySelector(".shadow-box");
    const shadow2 = document.querySelector(".shadow-box2");

    // Check if all required elements exist
    if (!container || !arrowRight || !arrowLeft || !shadow || !shadow2) {
        return;
    }

    // Function to update arrow visibility
    function updateArrowVisibility() {
        const scrollLeft = container.scrollLeft;
        const maxScrollLeft = container.scrollWidth - container.clientWidth;

        // Show/hide left arrow
        if (scrollLeft <= 0) {
            arrowLeft.style.display = "none";
            shadow2.style.display = "none";
        } else {
            arrowLeft.style.display = "flex";
            shadow2.style.display = "flex";
        }

        // Show/hide right arrow
        if (scrollLeft >= maxScrollLeft) {
            arrowRight.style.display = "none";
            shadow.style.display = "none";
        } else {
            arrowRight.style.display = "flex";
            shadow.style.display = "flex";
        }
    }

    // Initial check
    updateArrowVisibility();

    // Scroll event to handle arrow visibility
    container.addEventListener("scroll", updateArrowVisibility);

    // Scroll actions
    arrowRight.addEventListener("click", (e) => {
        e.preventDefault();
        container.scrollBy({
            left: 100,
            behavior: "smooth",
        });
    });

    arrowLeft.addEventListener("click", (e) => {
        e.preventDefault();
        container.scrollBy({
            left: -100,
            behavior: "smooth",
        });
    });
}
// End Tab Navigation

// ------------------ Profile Tab Navigation ------------------
function initProfileTabs() {
    const tabButtons = document.querySelectorAll('#default-styled-tab button[role="tab"]');
    const tabContents = document.querySelectorAll('#default-styled-tab-content > div[role="tabpanel"]');

    if (tabButtons.length === 0 || tabContents.length === 0) {
        return;
    }

    // On initial load or morph, clean up the URL if a tab parameter exists
    const currentUrl = new URL(window.location.href);
    if (currentUrl.searchParams.has('tab')) {
        currentUrl.searchParams.delete('tab');
        window.history.replaceState({}, '', currentUrl.toString());
    }

    tabButtons.forEach(button => {
        button.removeEventListener('click', handleProfileTabClick); // Prevent duplicate listeners
        button.addEventListener('click', handleProfileTabClick);
    });

    function handleProfileTabClick(event) {
        event.preventDefault();

        const clickedButton = event.currentTarget;
        const targetTabId = clickedButton.dataset.tabsTarget;
        const tabKey = clickedButton.id.replace('-styled-tab', '').replace('Overview', 'overview');

        // Update URL with replaceState
        const url = new URL(window.location.href);
        if (tabKey === 'overview') {
            url.searchParams.delete('tab');
        } else {
            url.searchParams.set('tab', tabKey);
        }
        window.history.replaceState({}, '', url.toString());

        // Deactivate all tabs and hide all content
        tabButtons.forEach(btn => {
            btn.setAttribute('aria-selected', 'false');
            btn.classList.remove('bg-black-1200', 'text-white');
            btn.classList.add('text-mid-gray', 'hover:bg-black-1200', 'hover:text-white');
        });
        tabContents.forEach(content => {
            content.classList.add('hidden');
        });

        // Activate clicked tab and show its content
        clickedButton.setAttribute('aria-selected', 'true');
        clickedButton.classList.add('bg-black-1200', 'text-white');
        clickedButton.classList.remove('text-mid-gray', 'hover:bg-black-1200', 'hover:text-white');

        const activeContent = document.querySelector(targetTabId);
        if (activeContent) {
            activeContent.classList.remove('hidden');
        }
    }

    // Ensure the correct tab is active on initial load or Livewire morph
    const initialActiveTabParam = new URL(window.location.href).searchParams.get('tab') || 'overview';
    const initialActiveButton = document.getElementById(`${initialActiveTabParam}-styled-tab`);

    if (initialActiveButton) {
        const targetTabId = initialActiveButton.dataset.tabsTarget;

        tabButtons.forEach(btn => {
            btn.setAttribute('aria-selected', 'false');
            btn.classList.remove('bg-black-1200', 'text-white');
            btn.classList.add('text-mid-gray', 'hover:bg-black-1200', 'hover:text-white');
        });
        tabContents.forEach(content => {
            content.classList.add('hidden');
        });

        initialActiveButton.setAttribute('aria-selected', 'true');
        initialActiveButton.classList.add('bg-black-1200', 'text-white');
        initialActiveButton.classList.remove('text-mid-gray', 'hover:bg-black-1200', 'hover:text-white');

        const activeContent = document.querySelector(targetTabId);
        if (activeContent) {
            activeContent.classList.remove('hidden');
        }
    }
}
// End Profile Tab Navigation

// Import phone input functionality


// Initialize everything when the DOM content is loaded - SINGLE DOMCONTENTLOADED EVENT
document.addEventListener("DOMContentLoaded", function () {
    // Initialize phone input

    initPhoneInput();

    // Initialize share modal
    initShareModal();

    // initReviewWriteModal();
    // ------------------ Profile Tab Scroller initialization ------------------
    // Initialize the profile tab scroller functionality
    initTabNavigation();
    initProfileTabs();

    // Also try to initialize after a longer delay in case of slow loading
    setTimeout(() => {
        initTabNavigation();
        initProfileTabs();
    }, 500);

    // ------------------ Google Maps initialization ------------------
    // Function to check if Google Maps is loaded and initialize everything
    const checkGoogleMapsAndInitialize = function () {
        try {
            // Check if Google Maps API and Places library are both loaded
            if (window.google && window.google.maps && window.google.maps.places) {
                // Initialize member directory map if we're on the map view page
                // ESPDirectoryMap.init() will internally check if it's the correct page.
                ESPDirectoryMap.init();

                window.updateMemberMarkers(window.memberData);

                window.initSidebarSearch();

                // Initialize estimate banner address autocomplete if banner exists
                try {
                    const estimateInput = document.getElementById('estimate-address-input');
                    console.log('Estimate input element:', estimateInput);
                    console.log('ESPGoogleMaps object:', ESPGoogleMaps);
                    console.log('Google Maps Places available:', window.google?.maps?.places);

                    if (estimateInput) {
                        console.log('Initializing autocomplete for estimate banner...');

                        // Check if ESPGoogleMaps is available
                        if (typeof ESPGoogleMaps === 'undefined') {
                            console.error('ESPGoogleMaps is not defined!');
                            return;
                        }

                        // Use generic autocomplete initializer with city, state, country fields
                        const autocompleteInstance = ESPGoogleMaps.initAutocomplete({
                            inputField: '#estimate-address-input',
                            latitudeField: '#estimate-lat',
                            longitudeField: '#estimate-lng',
                            cityField: '#estimate-city',
                            stateField: '#estimate-state',
                            stateCodeField: '#estimate-state-code',
                            countryField: '#estimate-country',
                            countryCodeField: '#estimate-country-code',
                            types: ['geocode'],
                            onPlaceChanged: function (place) {
                                console.log('Place selected:', place);
                                // Update the input field with formatted address
                                const inputEl = document.getElementById('estimate-address-input');
                                if (inputEl && (place?.formatted_address || place?.name)) {
                                    inputEl.value = place.formatted_address || place.name;
                                }
                            }
                        });

                        console.log('Autocomplete instance returned:', autocompleteInstance);

                        if (autocompleteInstance) {
                            console.log('Estimate banner autocomplete initialized successfully');
                        } else {
                            console.error('Failed to initialize estimate banner autocomplete - returned null');
                        }
                    } else {
                        console.error('Estimate input element not found!');
                    }
                } catch (e) {
                    console.error('Error initializing estimate banner autocomplete:', e);
                }

                return true;
            }
            return false;
        } catch (error) {
            console.error('Error in checkGoogleMapsAndInitialize:', error);
            return false;
        }
    };

    // Try to initialize immediately if Google Maps is already loaded
    if (!checkGoogleMapsAndInitialize()) {
        // Set up a MutationObserver to watch for the Google Maps script
        const observer = new MutationObserver(function (mutations) {
            if (checkGoogleMapsAndInitialize()) {
                observer.disconnect();
            }
        });

        // Start observing the document
        observer.observe(document, { childList: true, subtree: true });

        // Set up a global event listener for Google Maps loaded event
        document.addEventListener('google-maps-loaded', function () {
            checkGoogleMapsAndInitialize();
        });

        // Also set up a timer as fallback
        const checkInterval = setInterval(function () {
            if (checkGoogleMapsAndInitialize()) {
                clearInterval(checkInterval);
            }
        }, 1000);

        // Clear the interval after 20 seconds
        setTimeout(function () {
            clearInterval(checkInterval);
        }, 20000);
    }

    var swiper = new Swiper(".popularSwiper", {
        slidesPerView: 1.2,
        spaceBetween: 16,
        pagination: {
            el: ".swiper-pagination",
            clickable: true,
        },
    });

    // ------------------ Swiper initializations ------------------
    // Product Swiper
    if (document.querySelector(".mySwiper")) {
        new Swiper(".mySwiper", {
            slidesPerView: 4,
            spaceBetween: 24,
            navigation: {
                nextEl: ".swiper-button-next",
                prevEl: ".swiper-button-prev",
            },
            breakpoints: {
                // when window width is >= 320px
                320: {
                    slidesPerView: 1.3,
                    spaceBetween: 12,
                    centeredSlides: true,
                    initialSlide: 1,
                    loop: true,
                },
                576: {
                    slidesPerView: 3,
                    spaceBetween: 14,
                    centeredSlides: false,
                    initialSlide: 0,
                    loop: false,
                },
                768: {
                    slidesPerView: 4,
                    spaceBetween: 14,
                    centeredSlides: false,
                    initialSlide: 0,
                    loop: false,
                },
                // when window width is >= 640px
                992: {
                    slidesPerView: 4,
                    spaceBetween: 24,
                    centeredSlides: false,
                    initialSlide: 0,
                    loop: false,
                },
            },
        });
    }

    // Awards Swiper
    if (document.querySelector(".awardsSwiper")) {
        new Swiper(".awardsSwiper", {
            slidesPerView: 1.7,
            spaceBetween: 12,
            loop: true,
        });
    }

    // Projects Swiper
    if (document.querySelector(".ProjectsSwiper")) {
        new Swiper(".ProjectsSwiper", {
            slidesPerView: 2.1,
            spaceBetween: 12,
            loop: true,
            navigation: {
                nextEl: ".swiper-button-next",
                prevEl: ".swiper-button-prev",
            },
            breakpoints: {
                // when window width is >= 320px
                375: {
                    slidesPerView: 1.05,
                    spaceBetween: 8,
                },
                576: {
                    slidesPerView: 1.1,
                    spaceBetween: 12,
                },
                768: {
                    slidesPerView: 1.5,
                    spaceBetween: 12,
                },
                // when window width is >= 640px
                1200: {
                    slidesPerView: 2.1,
                    spaceBetween: 12,
                },
            },
        });
    }

    // Initialize lightgallery
    const gallery = document.getElementById("lightgallery");
    if (gallery) {
        lightGallery(gallery, {
            plugins: [lgZoom, lgThumbnail],
            speed: 500,
            thumbnail: false,
            selector: '.gallery a',
            download: true,
            counter: true,
            enableDrag: true,
            enableSwipe: true,
            closable: true,
            loop: true,
            escKey: true,
            keyPress: true
        });
    }

    // Articles Swiper
    var swiper = new Swiper(".articlesSwiper", {
        slidesPerView: 3,
        spaceBetween: 12,
        pagination: {
            el: ".swiper-pagination",
            clickable: true,
        },
        breakpoints: {
            375: {
                slidesPerView: 1,
                spaceBetween: 12,
            },
            768: {
                slidesPerView: 3,
                spaceBetween: 12,
            },
        },
    });

    // ------------------ Review Write Modal ------------------

    // End Review Write Modal

    //Initial call to initialize any Top Rated Home Swiper elements
    window.initTopRatedHomeSwiper();
    // Initial call to initialize any Home Partners Swiper elements
    window.initHomePartnersSwiper();
    // Initial call to initialize any Home Monthly Package Swiper elements
    window.initHomeMonthlyPackageSwiper();
    // Initial call to initialize any Home Yearly Package Swiper elements
    window.initHomeYearlyPackageSwiper();
    // Initial call to initialize any Home Job Swiper elements
    window.initHomeJobSwiper();
    // Initial call to initialize any Home Services Swiper elements
    window.initHomeServicesSwiper();
    // Initial call to initialize any Home Classifieds Swiper elements
    window.initHomeClassifiedsSwiper();
    // Initial call to initialize any Flash Classifieds Swiper elements
    window.initFlashClassifiedsSwiper();
    // Initialize Home Top Members tabs
    window.initHomeMemberTabs();
    // Initial call to initialize any Classifieds Swiper elements
    window.initClassifiedsSwiper();
    // Initial call to initialize any Classified Detail Swiper elements
    window.initClassifiedDetailSwiper();
    // Initial call to initialize any Gallery Swiper elements
    window.initGallerySwiper();

    // Initial call to initialize profile gallery lightGallery
    window.initProfileGallery();

    // Initial call to initialize classified gallery lightGallery
    window.initClassifiedGallery();

    // Initial call to initialize project gallery lightGallery
    window.initProjectGallery();
    // Initial call to initialize equipment gallery lightGallery
    window.initEquipmentGallery();
    // Initial call to initialize member feature gallery lightGallery
    window.initMemberFeatureSwiper();
    // Initial call to initialize member feature gallery lightGallery
    window.initMemberFeatureGallery();

    // Initial call to initialize marquee
    window.initMarquee();
});

// Lead wizard initializer to re-apply step visibility from data-current-step
window.initLeadWizard = function () {
    const wizard = document.querySelector('.lead-wizard-form');
    if (!wizard) return;

    const steps = wizard.querySelectorAll('.step');
    const stepLinks = document.querySelectorAll('.step-link');
    const current = parseInt(wizard.getAttribute('data-current-step') || '0', 10) || 0;

    // Update step visibility
    steps.forEach((step, i) => {
        step.style.display = i === current ? 'block' : 'none';
    });

    // Update sidebar state
    stepLinks.forEach((link, i) => {
        link.classList.remove('active', 'complete');
        if (i === current) {
            link.classList.add('active');
        } else if (i < current) {
            link.classList.add('complete');
        }
    });
};

// Hook into Livewire's lifecycle events for other components
Livewire.hook('morphed', ({ el, component, url, options, payload, respond, succeed, fail }) => {


    // Debounce or ensure functions are idempotent if called multiple times rapidly
    if (component?.name === 'frontend.member.profile-tabs.review-tab') {
        window.initReviewGallery();
    }
    // Re-apply lead wizard after Livewire morphs
    if (
        component?.name?.includes('frontend.widgets.lead-form') ||
        el?.querySelector('.lead-wizard-form')
    ) {
        setTimeout(() => {
            window.initLeadWizard && window.initLeadWizard();
        }, 50);
    }

    if (component?.name === 'frontend.member.profile-tabs.classified-tab' || component?.name === 'frontend.classified.classified-list') {
        window.initClassifiedsSwiper();
        setTimeout(() => window.initSidebarSearch(), 100);
    }

    // Initialize gallery swiper for profile components or any component containing gallery
    if (component?.name?.includes('profile') || el?.querySelector('.gallerySwiper')) {
        setTimeout(() => window.initGallerySwiper(), 100);
        // Also initialize profile gallery lightGallery
        // Also initialize classified gallery lightGallery
    }
    if (component?.name?.includes('frontend.member.listing.search-result.search-result-list-view') || component?.name?.includes('frontend.member.listing.search-result.search-result-map-view') || component?.name === 'frontend.classified.partials.search-classified-sidebar-v1' || component?.name === 'frontend.member.listing.search-result.member-city-grid' || component?.name === 'frontend.widgets.top-members') {
        setTimeout(() => window.initSidebarSearch(), 100);
        window.initMemberFeatureSwiper();
        window.initMemberFeatureGallery();
    }

    /**
     * Top Rated Home Swiper
     */
    if (component?.name === 'frontend.widgets.top-members') {
        // Reinitialize swiper and tabs after Livewire morphs the DOM
        setTimeout(() => {
            window.initTopRatedHomeSwiper();
            window.initHomeMemberTabs();
            // After content loads, mark the active tab as loaded and remove its spinner
            try {
                const activeButton = document.querySelector('#home-member-tabs button[role="tab"][aria-selected="true"]');
                const targetId = activeButton ? activeButton.getAttribute('data-tabs-target') : null;
                if (targetId) {
                    window.markHomeMemberTabLoaded(targetId);
                }
            } catch (e) {
                // Silent error handling
            }
        }, 100);
    }
    if (component?.name === 'frontend.widgets.job-list') {
        setTimeout(() => {
            window.initHomeJobSwiper();
        }, 100);
    }
    if (component?.name === 'frontend.widgets.services') {
        setTimeout(() => {
            window.initHomeServicesSwiper();
        }, 100);
    }
    if (component?.name === 'frontend.widgets.classified-list') {
        setTimeout(() => {
            window.initHomeClassifiedsSwiper();
        }, 100);
    }
    window.initMarquee();

    /**
     * Top Rated Home Swiper
     */




    if (ESPDirectoryMap.initialized &&
        (component?.name === 'frontend.member.listing.search-result.search-result-map-view' || el?.querySelector('[data-member-id]')) &&
        typeof ESPDirectoryMap.initCardHoverHandlers === 'function') {
        try {
            const memberData = component?.ephemeral?.membersData || component?.membersData;
            if (memberData) {
                window.memberData = memberData;
                window.updateMemberMarkers(memberData);
            }
        } catch (error) {
            // Silent error handling
        }

        setTimeout(() => ESPDirectoryMap.initCardHoverHandlers(), 250);
    }

    initViewTypeSwitchers();

    // Reinitialize tab scroller when Livewire morphs content
    // This is especially important for profile tabs that load content dynamically
    if (window.ProfileTabScroller && document.querySelector('.main-tabs')) {
        setTimeout(() => {
            window.ProfileTabScroller.forceUpdate();
        }, 100);
    }
    // Reinitialize profile tabs after Livewire morphs content
    if (component?.name?.includes('frontend.member.profile') || el?.querySelector('#default-styled-tab')) {
        setTimeout(() => {
            initProfileTabs();
        }, 100);
    }
});

Livewire.hook('commit.prepare', ({ component }) => {
    const componentName = component.name;
    // console.log('before commit payloads are collected and sent to the server...');
});

Livewire.on('openModal', ({ component, arguments: args }) => {
    // console.log(component, args);
})

Livewire.hook('commit', ({ component, commit, respond, succeed, fail }) => {

    const componentName = component.name;

    if (componentName.startsWith('frontend.member.profile-tabs.')) {
        document.getElementById('tabLoading').classList.remove('hidden');
    }

    if (componentName === 'frontend.job.job-list') {
        // console.log('before commit payloads are collected and sent to the server...');
    }

    if (componentName === 'livewire-ui-modal') {

    }

    respond(() => {
        if (componentName.startsWith('frontend.member.profile-tabs.')) {
            document.getElementById('tabLoading').classList.add('hidden');
            // Update tab scroller arrows after profile tab content loads
            setTimeout(() => {
                if (window.ProfileTabScroller) {
                    window.ProfileTabScroller.forceUpdate();
                }
            }, 100);
        }
        if (componentName === 'frontend.job.job-list') {
            // console.log('after response but before DOM diff');
        }

        if (componentName === 'livewire-ui-modal') {
            // console.log(component.ephemeral);
        }
    });
    succeed(() => {
        if (componentName === 'frontend.job.job-list') {
            // console.log('after DOM diff');
        }
        // Reinitialize package swipers after Livewire updates the package list
        if (componentName === 'frontend.widgets.package-list') {
            setTimeout(() => {
                if (window.initHomeMonthlyPackageSwiper) {
                    window.initHomeMonthlyPackageSwiper();
                }
                if (window.initHomeYearlyPackageSwiper) {
                    window.initHomeYearlyPackageSwiper();
                }
            }, 0);
        }
    });
    fail(() => {
        if (componentName === 'frontend.job.job-list') {
            // console.log('on error');
        }
    });
});


/**
 * Livewire Init Hook
 * Runs after Livewire is loaded but before it's initialized
 */
document.addEventListener('livewire:init', () => {
    Livewire.hook('morphed', ({ el, component }) => {
        const componentName = component.name;
        if (componentName.startsWith('frontend.member.listing.search-result.') || componentName === 'frontend.classified.classified-list') {
            // updateSearchResultText(component.ephemeral);
        }
    });

    Livewire.hook('component.init', ({ component }) => {
        const componentName = component.name;
        if (componentName === 'frontend.member.modal.review-write-modal') {
            window.initReviewWriteModal();
        }

        if (componentName.startsWith('frontend.member.listing.search-result.')) {
            // updateSearchResultText(component.ephemeral);
        }
    });

    // Utility: hide all contact overlays (fallback safe)
    function hideAllContactOverlays() {
        document.querySelectorAll('[class*="contact-overlay-"]').forEach(el => {
            el.classList.add('hidden');
        });
    }
    function hideAllVideoOverlays() {
        document.querySelectorAll('[class*="play-video-overlay-"]').forEach(el => {
            el.classList.add('hidden');
        });
    }

    Livewire.hook('request', ({ url, options, payload, respond, succeed, fail }) => {
        const payloadObj = JSON.parse(payload);

        let modalData = parseRequestPayloadForModal(payloadObj);
        if (modalData.component == 'frontend.member.modal.member-contact-modal') {
            // if (modalData.action === 'close') {
            //     // Ensure any lingering overlays are hidden when modal closes
            //     hideAllContactOverlays();
            // }
            const memberSlug = modalData.args.member;
            const modalId = modalData.args.modalId;
            if (memberSlug) {
                let selector = '.contact-overlay-' + memberSlug;
                if (modalId) selector += `[data-modal-id="${modalId}"]`;
                const contactOverlay = document.querySelector(selector);
                if (contactOverlay) contactOverlay.classList.remove('hidden');
            }
        }

        if (modalData.component == 'frontend.team-member.modal.team-member-detail') {
            const memberSlug = modalData.args.member;
            if (memberSlug) {
                const cardOverlay = document.querySelector('.team-detail-overlay-' + memberSlug);
                cardOverlay.classList.remove('hidden');
            }
        }


        if (modalData.component == 'frontend.member.modal.review-write-modal') {
            const memberSlug = modalData.args.member;
            if (memberSlug) {
                const cardOverlay = document.querySelector('.write-review-btn-overlay-' + memberSlug);
                cardOverlay.classList.remove('hidden');
            }
        }

        if (modalData.component == 'frontend.member.modal.play-video-modal') {
            const videoId = modalData.args.video;
            const modalId = modalData.args.modalId;
            if (videoId) {
                hideAllVideoOverlays();
                let selector = '.play-video-overlay-' + videoId;
                if (modalId) selector += `[data-modal-id="${modalId}"]`;
                const cardOverlay = document.querySelector(selector);
                if (cardOverlay) cardOverlay.classList.remove('hidden');
            }
        }

        //  Login Modal
        if (modalData.component == 'frontend.auth.modal.login-modal') {
            const modalId = modalData.args.modalId;
            if (modalId) {
                const cardOverlay = document.querySelector('.login-overlay-' + modalId);
                (cardOverlay) ? cardOverlay.classList.remove('hidden') : null;
            }
        }
        //  Login Modal
        if (modalData.component == 'frontend.auth.modal.register-modal') {
            const modalId = modalData.args.modalId;
            if (modalId) {
                const cardOverlay = document.querySelector('.register-overlay-' + modalId);
                cardOverlay.classList.remove('hidden');
            }
        }

        respond(({ status, response }) => {
            modalData = parseRequestPayloadForModal(payloadObj);

            if (modalData.component == 'frontend.member.modal.member-contact-modal') {

                // if (modalData.action === 'open') {
                //     hideAllContactOverlays();
                // }
                const memberSlug = modalData.args.member;
                const modalId = modalData.args.modalId;
                if (memberSlug) {
                    let selector = '.contact-overlay-' + memberSlug;
                    if (modalId) selector += `[data-modal-id="${modalId}"]`;
                    const contactOverlay = document.querySelector(selector);
                    if (contactOverlay) contactOverlay.classList.add('hidden');
                }
            }

            if (modalData.component == 'frontend.team-member.modal.team-member-detail') {
                const memberSlug = modalData.args.member;
                if (memberSlug) {
                    const cardOverlay = document.querySelector('.team-detail-overlay-' + memberSlug);
                    cardOverlay.classList.add('hidden');
                }
            }
            // Review Write Modal
            if (modalData.component == 'frontend.member.modal.review-write-modal') {
                const memberSlug = modalData.args.member;
                if (memberSlug) {
                    const writeReviewBtnOverlay = document.querySelector('.write-review-btn-overlay-' + memberSlug);
                    writeReviewBtnOverlay.classList.add('hidden');
                }
            }
            // Runs when the response is received...
            if (modalData.component == 'frontend.member.modal.play-video-modal') {
                const videoId = modalData.args.video;
                const modalId = modalData.args.modalId;
                if (videoId) {
                    let selector = '.play-video-overlay-' + videoId;
                    if (modalId) selector += `[data-modal-id="${modalId}"]`;
                    const cardOverlay = document.querySelector(selector);
                    if (cardOverlay) cardOverlay.classList.add('hidden');
                }
            }
            // Login Modal
            if (modalData.component == 'frontend.auth.modal.login-modal') {
                const modalId = modalData.args.modalId;
                if (modalId) {
                    const cardOverlay = document.querySelector('.login-overlay-' + modalId);
                    (cardOverlay) ? cardOverlay.classList.add('hidden') : null;
                }
            }
            // Register Modal
            if (modalData.component == 'frontend.auth.modal.register-modal') {
                const modalId = modalData.args.modalId;
                if (modalId) {
                    console.log('Register Modal', modalId);
                    const cardOverlay = document.querySelector('.register-overlay-' + modalId);
                    cardOverlay.classList.add('hidden');
                }
            }
            // "response" is the raw HTTP response object
            // before await response.text() is run...
        });

        succeed(({ status, json }) => {
            // Runs when the response is received...
            // "json" is the JSON response object...
        })

        fail(({ status, content, preventDefault }) => {
            // Runs when the response has an error status code...
            // "preventDefault" allows you to disable Livewire's
            // default error handling...
            // "content" is the raw response content...
        })

    });
});

function parseRequestPayloadForModal(payloadObj) {
    let modalData = { component: undefined, args: {}, action: undefined };
    (payloadObj.components || []).forEach(componentPayload => {
        // 1. Parse snapshot JSON to get the Livewire component name
        let snapshot = {};
        try {
            snapshot = JSON.parse(componentPayload.snapshot || '{}');
        } catch (e) { }

        const livewireComponent = snapshot?.memo?.name;
        if (livewireComponent == 'livewire-ui-modal') {
            // 2. Get modal call (__dispatch with openModal/closeModal)
            const modalCall = (componentPayload.calls || []).find(
                c => c.method === '__dispatch' && (c.params?.[0] === 'openModal' || c.params?.[0] === 'closeModal' || c.params?.[0] === 'closeModalWithEvents' || c.params?.[0] === 'closeModalWithEventsOutside')
            );

            if (modalCall) {
                const actionName = modalCall.params?.[0];
                const isOpen = actionName === 'openModal';
                const opts = (isOpen ? (modalCall.params?.[1] ?? {}) : {});
                const modalComponent = isOpen ? opts.component : undefined;
                const modalArgs = isOpen ? (opts.arguments ?? {}) : {};
                modalData = {
                    component: modalComponent,
                    args: modalArgs,
                    action: isOpen ? 'open' : 'close'
                };
                if (isOpen && modalComponent == 'frontend.member.modal.member-contact-modal') {
                    const memberSlug = modalArgs.member;
                    const modalId = modalArgs.modalId;
                    if (memberSlug) {
                        let selector = '.contact-overlay-' + memberSlug;
                        if (modalId) selector += `[data-modal-id=\"${modalId}\"]`;
                        const contactOverlay = document.querySelector(selector) || document.querySelector('.contact-overlay-' + memberSlug);
                        if (contactOverlay) contactOverlay.classList.remove('hidden');
                    }
                }
            }
        }
    });
    return modalData;
}
/**
 * Livewire Initialized Hook
 * Runs immediately after Livewire has finished initializing
 */
document.addEventListener('livewire:initialized', () => {
    // Initial load
    initClassifiedsSwiper();

    // Listen for the custom classified-list-updated event
    document.addEventListener('livewire:classified-list-updated', () => {
        setTimeout(() => {
            initClassifiedsSwiper();
        }, 100);
    });

    // Initialize member data for map markers on initial page load
    try {
        const components = Livewire.getByName('frontend.member.listing.search-result.search-result-map-view');

        // Check if components exist and have at least one element
        if (components && components.length > 0) {
            const comp = components[0];

            // Check if component exists and has the expected properties
            if (comp && typeof comp === 'object') {
                // Try to access membersData property with fallback
                const membersData = comp.membersData || comp.get?.('membersData') || null;

                if (membersData) {
                    window.memberData = membersData;
                }
            }
        }
    } catch (error) {
        // Ensure window.memberData is initialized even if component access fails
        if (typeof window.memberData === 'undefined') {
            window.memberData = [];
        }
    }
});

/**
 * Initializes view type switcher buttons for member listing
 * This handles switching between list and map views by updating the URL
 */
function initViewTypeSwitchers() {
    const viewSwitchers = document.querySelectorAll('.view-type-switch');

    if (viewSwitchers.length === 0) return;

    viewSwitchers.forEach(button => {
        // Remove any existing event listeners to prevent duplicates
        const newButton = button.cloneNode(true);
        button.parentNode.replaceChild(newButton, button);

        newButton.addEventListener('click', function () {
            const viewType = this.dataset.viewType;
            const currentUrl = new URL(window.location.href);

            if (viewType === 'list') {
                // For list view: remove view_type parameter
                currentUrl.searchParams.delete('view_type');
            } else if (viewType === 'map') {
                // For map view: set view_type=map
                currentUrl.searchParams.set('view_type', 'map');
            }

            // Navigate to the new URL
            window.location.href = currentUrl.toString();
        });
    });
}

// Initialize view type switchers on page load
document.addEventListener("DOMContentLoaded", function () {
    initViewTypeSwitchers();

});


document.addEventListener('alpine:init', () => {
    // Initialize Home Top Members with tab management
    Alpine.data('initHomeTopMembers', () => ({
        memberListLoading: false,
        loadedTabs: new Set(['all']), // Track which tabs have been loaded (all is loaded by default)
        currentTab: 'all',

        init() {
            this.memberListLoading = false;

            // Wait for Livewire to be fully initialized before setting up tabs
            const initializeTabs = () => {
                const tabButtons = document.querySelectorAll('#home-member-tabs button[role="tab"]');

                if (tabButtons.length === 0) {
                    // If tabs aren't ready yet, try again after a short delay
                    setTimeout(initializeTabs, 100);
                    return;
                }

                // Set up tab click handlers
                tabButtons.forEach(button => {
                    button.addEventListener('click', (e) => {
                        e.preventDefault();
                        const categoryId = button.getAttribute('data-category-id');
                        const targetId = button.getAttribute('data-tabs-target');

                        // Update current tab
                        this.currentTab = categoryId;

                        // Update tab button states
                        this.updateTabStates(button);

                        // Show/hide tab content
                        this.switchTabContent(targetId);

                        // Load data if this is the first time visiting this tab
                        if (!this.loadedTabs.has(categoryId)) {
                            this.loadTabData(categoryId);
                        } else {
                            // Reinitialize swiper for already loaded content
                            setTimeout(() => {
                                window.initTopRatedHomeSwiper();
                            }, 100);
                        }
                    });
                });

                // Initialize swiper for the default "All" tab
                setTimeout(() => {
                    window.initTopRatedHomeSwiper();
                }, 100);
            };

            // Start initialization after Livewire is ready
            if (typeof Livewire !== 'undefined') {
                // If Livewire is already available, init immediately
                setTimeout(initializeTabs, 100);
            } else {
                // Otherwise wait for Livewire to be ready
                document.addEventListener('livewire:initialized', () => {
                    setTimeout(initializeTabs, 100);
                });
            }
        },

        updateTabStates(activeButton) {
            const allButtons = document.querySelectorAll('#home-member-tabs button[role="tab"]');
            allButtons.forEach(btn => {
                btn.classList.remove('text-blue-1500', 'bg-blue-1400/[0.1]');
                btn.classList.add('text-mid-gray', 'bg-gray-1400');
                btn.setAttribute('aria-selected', 'false');
            });

            activeButton.classList.remove('text-mid-gray', 'bg-gray-1400');
            activeButton.classList.add('text-blue-1500', 'bg-blue-1400/[0.1]');
            activeButton.setAttribute('aria-selected', 'true');
        },

        switchTabContent(targetId) {
            const allPanels = document.querySelectorAll('#home-member-tabs-content > div[role="tabpanel"]');
            allPanels.forEach(panel => {
                panel.classList.add('hidden');
            });

            const targetPanel = document.querySelector(targetId);
            if (targetPanel) {
                targetPanel.classList.remove('hidden');
            }
        },

        loadTabData(categoryId) {
            // Show loading state
            this.memberListLoading = true;

            // Find the top-members Livewire component
            const componentElement = document.querySelector('[wire\\:id]');

            if (!componentElement) {
                console.error('Livewire component element not found');
                this.memberListLoading = false;
                return;
            }

            const wireId = componentElement.getAttribute('wire:id');
            if (!wireId) {
                console.error('wire:id attribute not found');
                this.memberListLoading = false;
                return;
            }

            const component = Livewire.find(wireId);

            if (component) {
                // Call Livewire method to load members for this category
                const finalCategoryId = categoryId === 'all' ? null : categoryId;

                // Using $wire to call the method
                Livewire.find(wireId).call('loadCategoryMembers', finalCategoryId)
                    .then(() => {
                        // Mark this tab as loaded
                        this.loadedTabs.add(categoryId);

                        // Hide loading state
                        this.memberListLoading = false;

                        // Reinitialize swiper after content is loaded
                        setTimeout(() => {
                            window.initTopRatedHomeSwiper();
                        }, 200);
                    })
                    .catch((error) => {
                        console.error('Error loading category members:', error);
                        this.memberListLoading = false;
                    });
            } else {
                console.error('Livewire component not found');
                this.memberListLoading = false;
            }
        }
    }));

    // Member Search
    Alpine.data('initMemberSearch', () => ({
        memberListLoading: false,
        mapLoading: false,
        openFilter: false,
        init() {

            // 1) Whenever the search result component sends its request…
            Livewire.hook('commit', ({ component, commit, respond, succeed, fail }) => {
                const componentName = component.name;
                if (componentName.startsWith('frontend.member.listing.search-result.')) {
                    this.memberListLoading = true;
                }

                respond(() => {
                    if (componentName.startsWith('frontend.member.listing.search-result.')) {
                        this.memberListLoading = false;
                    }
                    if (componentName === 'frontend.member.listing.search-result.search-result-map-view') {
                        this.openFilter = false;
                        // Reset map loading state when map view responds
                        this.mapLoading = false;
                    }

                });
                succeed(() => {
                });

                fail(() => {
                    // Reset loading state on failure
                    if (componentName === 'frontend.member.listing.search-result.search-result-map-view') {
                        this.mapLoading = false;
                    }
                });
            });

            // Listen for custom map loading events
            document.addEventListener('livewire:dispatch', (e) => {
                if (e.detail.name === 'map-loading-complete') {
                    this.mapLoading = false;
                }
            });
        }
    }));

    // Initialize Blog page
    Alpine.data('initBlogSearch', () => ({
        articleListLoading: false,
        openFilter: false,
        init() {
            // 1) Whenever the sidebar component sends its request…
            Livewire.hook('commit', ({ component, commit, respond, succeed, fail }) => {
                const componentName = component.name;

                if (componentName === 'frontend.blog.post-search') {
                    this.articleListLoading = true;
                }

                respond(() => {
                    if (componentName === 'frontend.blog.post-search') {
                        this.articleListLoading = false;
                    }
                });
                succeed(() => {
                });

                fail(() => {

                });

            });
        }
    }));

    // Initialize Classified List page
    Alpine.data('initClassifiedSearch', () => ({
        classifiedListLoading: false,
        openFilter: false,
        init() {
            // 1) Whenever the sidebar component sends its request…
            Livewire.hook('commit', ({ component, commit, respond, succeed, fail }) => {
                const componentName = component.name;
                if (componentName === 'frontend.classified.classified-list' || componentName === 'frontend.classified.partials.search-classified-sidebar-v1') {
                    this.classifiedListLoading = true;
                }

                respond(() => {
                    if (componentName === 'frontend.classified.classified-list' || componentName === 'frontend.classified.partials.search-classified-sidebar-v1') {
                        this.classifiedListLoading = false;
                    }
                });
                succeed(() => {
                });

                fail(() => {

                });
            });
        }
    }));


    // Countdown timer for flash sales  Initialize Classified Flash Countdown
    Alpine.data('countdown', (endDate, timezone = null) => ({
        days: 0,
        hours: 0,
        minutes: 0,
        seconds: 0,
        interval: null,
        targetTimestamp: null,
        debugInfo: {
            endDate: endDate,
            timezone: timezone,
            parsedDate: null,
            currentTime: null,
            distance: null
        },

        init() {
            if (!endDate) return;
            this.calculateTargetTimestamp();
            this.updateCountdown();
            this.startCountdown();
        },

        calculateTargetTimestamp() {
            if (!endDate) return;

            // Debug: Log the received endDate
            this.debugInfo.endDate = endDate;
            this.debugInfo.timezone = timezone;

            try {
                let targetDate;

                // Handle different date formats
                if (endDate.includes('T') && endDate.includes('Z')) {
                    // ISO format with Z (UTC): "2025-01-15T14:30:00Z"
                    targetDate = new Date(endDate);
                } else if (endDate.includes('T')) {
                    // ISO format without timezone: "2025-01-15T14:30:00"
                    if (timezone) {
                        targetDate = new Date(endDate + timezone);
                    } else {
                        // Assume UTC if no timezone provided
                        targetDate = new Date(endDate + 'Z');
                    }
                } else if (endDate.includes(' ')) {
                    // Legacy datetime format: "2025-01-15 14:30:00"
                    const isoString = endDate.replace(' ', 'T');

                    if (timezone) {
                        targetDate = new Date(isoString + timezone);
                    } else {
                        // Assume UTC for legacy format
                        targetDate = new Date(isoString + 'Z');
                    }
                } else {
                    // Date-only format: "2025-01-15" (fallback to midnight UTC)
                    if (timezone) {
                        targetDate = new Date(endDate + 'T00:00:00' + timezone);
                    } else {
                        targetDate = new Date(endDate + 'T00:00:00Z');
                    }
                }

                // Validate the parsed date
                if (isNaN(targetDate.getTime())) {
                    console.error('Invalid date format:', endDate);
                    return;
                }

                this.targetTimestamp = targetDate.getTime();
                this.debugInfo.parsedDate = targetDate.toISOString();

                // Debug log
                console.log('Countdown Debug:', {
                    originalEndDate: endDate,
                    parsedDate: targetDate.toISOString(),
                    localDate: targetDate.toString(),
                    targetTimestamp: this.targetTimestamp,
                    currentTime: new Date().getTime(),
                    difference: (this.targetTimestamp - new Date().getTime()) / 1000 / 60 / 60 + ' hours'
                });

            } catch (error) {
                console.error('Error parsing countdown date:', error, endDate);
            }
        },

        startCountdown() {
            if (!this.targetTimestamp) return;

            // Clear any existing interval
            if (this.interval) {
                clearInterval(this.interval);
            }

            // Update immediately
            this.updateCountdown();

            // Update every second
            this.interval = setInterval(() => {
                this.updateCountdown();
            }, 1000);
        },

        updateCountdown() {
            if (!this.targetTimestamp) return;

            const now = new Date().getTime();
            const distance = this.targetTimestamp - now;

            // Update debug info
            this.debugInfo.currentTime = new Date(now).toISOString();
            this.debugInfo.distance = distance;

            if (distance < 0) {
                // Countdown finished
                this.days = 0;
                this.hours = 0;
                this.minutes = 0;
                this.seconds = 0;

                if (this.interval) {
                    clearInterval(this.interval);
                    this.interval = null;
                }
                return;
            }

            // Calculate time components
            this.days = Math.floor(distance / (1000 * 60 * 60 * 24));
            this.hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            this.minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
            this.seconds = Math.floor((distance % (1000 * 60)) / 1000);
        },

        formatTime(value) {
            return String(value).padStart(2, '0');
        },

        // Debug method to check current state
        getDebugInfo() {
            return {
                ...this.debugInfo,
                currentCountdown: {
                    days: this.days,
                    hours: this.hours,
                    minutes: this.minutes,
                    seconds: this.seconds
                },
                intervalActive: !!this.interval
            };
        },

        destroy() {
            if (this.interval) {
                clearInterval(this.interval);
                this.interval = null;
            }
        }
    }));


    // Initialize Job List page
    Alpine.data('initJobSearch', () => ({
        jobListLoading: false,
        openFilter: false,
        init() {
            // 1) Whenever the sidebar component sends its request…
            Livewire.hook('commit', ({ component, commit, respond, succeed, fail }) => {
                const componentName = component.name;
                if (componentName === 'frontend.job.job-list' || componentName === 'frontend.job.partials.search-job-sidebar-v1') {
                    this.jobListLoading = true;
                }

                respond(() => {
                    if (componentName === 'frontend.job.job-list' || componentName === 'frontend.job.partials.search-job-sidebar-v1') {
                        this.jobListLoading = false;
                    }
                });
                succeed(() => {
                });

                fail(() => {

                });
            });
        }
    }));


    //city grid
    Alpine.data('initMemberCitySearch', () => ({
        memberListLoading: false,
        mapLoading: false,
        openFilter: false,
        init() {

            // 1) Whenever the search result component sends its request…
            Livewire.hook('commit', ({ component, commit, respond, succeed, fail }) => {
                const componentName = component.name;
                if (componentName.startsWith('frontend.member.listing.search-result.member-city-grid')) {
                    this.memberListLoading = true;
                }

                respond(() => {
                    if (componentName.startsWith('frontend.member.listing.search-result.member-city-grid')) {
                        this.memberListLoading = false;
                    }
                    if (componentName === 'frontend.member.listing.search-result.member-city-grid') {
                        // Reset map loading state when map view responds
                        this.mapLoading = false;
                    }

                });
                succeed(() => {
                });

                fail(() => {
                    // Reset loading state on failure
                    if (componentName === 'frontend.member.listing.search-result.member-city-grid') {
                        this.mapLoading = false;
                    }
                });
            });

            // Listen for custom map loading events
            document.addEventListener('livewire:dispatch', (e) => {
                if (e.detail.name === 'map-loading-complete') {
                    this.mapLoading = false;
                }
            });
        }
    }));

    Alpine.data('initReviewTab', () => ({
        reviewsTabLoading: false,
        init() {

        }
    }));

    Alpine.store('actions', {
        // Loading states for different actions
        loading: {},

        // Set loading state for a specific button
        setLoading(buttonId, state) {
            this.loading[buttonId] = state;
        },

        // Check if a specific action is loading
        isLoading(buttonId) {
            return this.loading[buttonId] || false;
        },

        // Main action handler - similar to WordPress AJAX
        async handleAction(action, module, modelId, additionalData = {}, buttonElement = null) {
            const buttonId = buttonElement?.getAttribute('data-button-id') || `${action}-${module}-${modelId}`;

            try {
                // Set loading state
                this.setLoading(buttonId, true);

                // Update button text if element provided
                if (buttonElement) {
                    const loadingSpinner = buttonElement.querySelector('.action-loading-spinner');
                    if (loadingSpinner) {
                        loadingSpinner.classList.remove('hidden');
                    }
                }

                // Prepare request data
                const requestData = {
                    action: action,
                    module: module,
                    model_id: modelId,
                    ...additionalData
                };

                // Make AJAX request
                const response = await fetch('/actions/handle', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify(requestData)
                });

                const result = await response.json();

                if (result.success) {
                    // Handle successful response
                    this.handleSuccess(action, result, buttonElement);

                    // Dispatch custom event for other components to listen
                    window.dispatchEvent(new CustomEvent('action-completed', {
                        detail: { action, module, modelId, result }
                    }));

                } else {
                    // Handle error response
                    this.handleError(result.message || 'An error occurred', buttonElement);
                }

            } catch (error) {
                console.error('Action error:', error);
                this.handleError('Network error occurred', buttonElement);
            } finally {
                // Reset loading state
                this.setLoading(buttonId, false);

                // Reset button state
                if (buttonElement) {
                    const loadingSpinner = buttonElement.querySelector('.action-loading-spinner');
                    if (loadingSpinner) {
                        loadingSpinner.classList.add('hidden');
                    }
                    buttonElement.disabled = false;
                }
            }
        },

        // Handle successful action response
        handleSuccess(action, result, buttonElement) {
            switch (action) {
                case 'add_to_favourite':
                    this.updateFavoriteButton(buttonElement, true);
                    // this.showNotification(result.message, 'success');
                    break;

                case 'remove_from_favourite':
                    this.updateFavoriteButton(buttonElement, false);
                    // this.showNotification(result.message, 'success');
                    break;
                default:
                // this.showNotification(result.message, 'success');
            }
        },

        // Handle error response
        handleError(message, buttonElement) {
            // this.showNotification(message, 'error');
        },

        // Update favorite button appearance
        updateFavoriteButton(buttonElement, isFavorite) {
            if (!buttonElement) return;

            const icon = buttonElement.querySelector('i') || buttonElement.querySelector('svg');
            const text = buttonElement.querySelector('.btn-text');

            if (isFavorite) {
                buttonElement.classList.add('text-red-500');
                buttonElement.setAttribute('data-action', 'remove_from_favourite');
            } else {
                buttonElement.classList.remove('text-red-500');
                buttonElement.setAttribute('data-action', 'add_to_favourite');
            }
        },
    });
});

/**
 * Share Modal Functionality
 */

// Store modal instance globally
let shareModalInstance = null;

/**
 * Initialize the share modal with Flowbite
 */
function initShareModal() {
    const modalElement = document.getElementById('share-modal');

    // Only initialize if modal exists on the page
    if (!modalElement) {
        return;
    }

    // Create modal instance with options
    const modalOptions = {
        placement: 'center',
        backdrop: 'dynamic',
        backdropClasses: 'bg-gray-900/50 dark:bg-gray-900/80 fixed inset-0 z-40',
        closable: true,
        onHide: () => {
            // Optional: cleanup when modal is hidden
        },
        onShow: () => {
            // Optional: actions when modal is shown
        }
    };

    // Initialize the modal
    shareModalInstance = new Modal(modalElement, modalOptions);
}

/**
 * Update share modal content with the provided type and URL
 * @param {string} type - The type of content being shared (e.g., 'profile', 'listing')
 * @param {string} url - The URL to be shared
 */
function updateShareModal(type, url) {
    // Update the modal title based on type
    const modalTitle = document.querySelector('#share-modal h2');
    const modalDescription = document.querySelector('#share-modal p');
    if (modalTitle) {
        switch (type) {
            case 'profile':
                modalTitle.textContent = 'Share this Profile';
                modalDescription.textContent = 'Share this profile with your network';
                break;
            case 'listing':
                modalTitle.textContent = 'Share this Listing';
                modalDescription.textContent = 'Share this listing with your network';
                break;
            case 'article':
                modalTitle.textContent = 'Share this Article';
                modalDescription.textContent = 'Share this article with your network';
                break;
            case 'job':
                modalTitle.textContent = 'Share this Job';
                modalDescription.textContent = 'Share this job with your network';
                break;
            case 'classified':
                modalTitle.textContent = 'Share this Classified';
                modalDescription.textContent = 'Share this classified with your network';
                break;
            case 'equipment':
                modalTitle.textContent = 'Share this Equipment';
                modalDescription.textContent = 'Share this equipment with your network';
                break;
            case 'project':
                modalTitle.textContent = 'Share this Project';
                modalDescription.textContent = 'Share this project with your network';
                break;
            default:
                modalTitle.textContent = 'Share this Page';
                modalDescription.textContent = 'Share this page with your network';
                break;
        }
    }

    // Update the URL in the copy text field
    const textElement = document.getElementById('text');
    if (textElement) {
        textElement.textContent = url;
    }

    // Update social sharing links
    updateSocialShareLinks(url);
}

/**
 * Update social media sharing links with the provided URL
 * @param {string} url - The URL to be shared
 */
function updateSocialShareLinks(url) {
    const encodedUrl = encodeURIComponent(url);
    const encodedTitle = encodeURIComponent(document.title || '');

    // Facebook
    const facebookBtn = document.querySelector('.social-share-btn[data-platform="facebook"]');
    if (facebookBtn) {
        facebookBtn.href = `https://www.facebook.com/sharer/sharer.php?u=${encodedUrl}`;
    }

    // Twitter
    const twitterBtn = document.querySelector('.social-share-btn[data-platform="twitter"]');
    if (twitterBtn) {
        twitterBtn.href = `https://twitter.com/intent/tweet?url=${encodedUrl}&text=${encodedTitle}`;
    }

    // LinkedIn
    const linkedinBtn = document.querySelector('.social-share-btn[data-platform="linkedin"]');
    if (linkedinBtn) {
        linkedinBtn.href = `https://www.linkedin.com/sharing/share-offsite/?url=${encodedUrl}`;
    }

    // Email
    const emailBtn = document.querySelector('.social-share-btn[data-platform="email"]');
    if (emailBtn) {
        const subject = encodeURIComponent(`Check out: ${document.title || ''}`);
        const body = encodeURIComponent(`I thought you might be interested in this: ${url}`);
        emailBtn.href = `mailto:?subject=${subject}&body=${body}`;
    }
}

/**
 * Handle social media sharing
 * @param {string} platform - The social media platform
 * @param {string} url - The URL to be shared
 */
function handleSocialShare(platform, url) {
    const encodedUrl = encodeURIComponent(url);
    const encodedTitle = encodeURIComponent(document.title || '');
    let shareUrl = '';

    switch (platform) {
        case 'facebook':
            shareUrl = `https://www.facebook.com/sharer/sharer.php?u=${encodedUrl}`;
            break;
        case 'twitter':
            shareUrl = `https://twitter.com/intent/tweet?url=${encodedUrl}&text=${encodedTitle}`;
            break;
        case 'linkedin':
            shareUrl = `https://www.linkedin.com/sharing/share-offsite/?url=${encodedUrl}`;
            break;
        case 'instagram':
            // Instagram doesn't support direct URL sharing, show a message
            showInstagramMessage(url);
            return;
        case 'email':
            const subject = encodeURIComponent(`Check out: ${document.title || ''}`);
            const body = encodeURIComponent(`I thought you might be interested in this: ${url}`);
            shareUrl = `mailto:?subject=${subject}&body=${body}`;
            break;
        default:
            // Unknown social platform
            return;
    }

    if (shareUrl) {
        // Open in a new window/tab
        window.open(shareUrl, '_blank', 'width=600,height=400,scrollbars=yes,resizable=yes');
    }
}

/**
 * Show Instagram sharing message (since Instagram doesn't support direct URL sharing)
 * @param {string} url - The URL to be shared
 */
function showInstagramMessage(url) {
    // Copy URL to clipboard and show message
    copyTextToClipboard(url);

    // Show a temporary message
    const message = document.createElement('div');
    message.className = 'fixed top-4 right-4 bg-primary-blue text-white px-4 py-2 rounded-lg shadow-lg z-[1100]';
    message.textContent = 'URL copied! Please paste it in your Instagram post or story.';

    document.body.appendChild(message);

    // Remove message after 3 seconds
    setTimeout(() => {
        if (message.parentNode) {
            message.parentNode.removeChild(message);
        }
    }, 3000);
}

/**
 * Copy the URL to clipboard from the share modal
 */
function copyToClipboard() {
    const textElement = document.getElementById('text');
    if (!textElement) {
        // Text element not found
        return;
    }

    const url = textElement.textContent.trim();
    copyTextToClipboard(url);
}

/**
 * Copy text to clipboard and show feedback
 * @param {string} text - The text to copy
 */
function copyTextToClipboard(text) {
    // Try using the modern clipboard API first
    if (navigator.clipboard && navigator.clipboard.writeText) {
        navigator.clipboard.writeText(text).then(() => {
            showCopyFeedback();
        }).catch(err => {
            fallbackCopyText(text);
        });
    } else {
        // Fallback for older browsers
        fallbackCopyText(text);
    }
}

/**
 * Fallback copy method for older browsers
 * @param {string} text - The text to copy
 */
function fallbackCopyText(text) {
    const textArea = document.createElement('textarea');
    textArea.value = text;
    textArea.style.position = 'fixed';
    textArea.style.left = '-999999px';
    textArea.style.top = '-999999px';
    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();

    try {
        // @ts-ignore - Using deprecated execCommand as fallback for older browsers
        const successful = document.execCommand('copy');
        if (successful) {
            showCopyFeedback();
        } else {
            showCopyError();
        }
    } catch (err) {
        showCopyError();
    }

    document.body.removeChild(textArea);
}

/**
 * Show feedback when URL is successfully copied
 */
function showCopyFeedback() {
    const copyBtn = document.getElementById('copy');
    if (copyBtn) {
        const originalText = copyBtn.textContent;
        copyBtn.textContent = 'Copied!';
        copyBtn.classList.add('bg-green-500');
        copyBtn.classList.remove('bg-red-1300');

        setTimeout(() => {
            copyBtn.textContent = originalText;
            copyBtn.classList.remove('bg-green-500');
            copyBtn.classList.add('bg-red-1300');
        }, 2000);
    }
}

/**
 * Show error feedback when copy fails
 */
function showCopyError() {
    const copyBtn = document.getElementById('copy');
    if (copyBtn) {
        const originalText = copyBtn.textContent;
        copyBtn.textContent = 'Error!';

        setTimeout(() => {
            copyBtn.textContent = originalText;
        }, 2000);
    }
}

function updateSearchResultText(ephemeral) {
    const perPage = ephemeral.perPage;
    const totalRecords = ephemeral.totalRecords;
    const currentPage = ephemeral.currentPage;
    const sortText = ephemeral.sortText;

    const offset = (currentPage - 1) * perPage;
    let to = offset + perPage;
    if (to > totalRecords) {
        to = totalRecords;
    }
    const paginationElement = document.querySelector('.member-list-results-summery');
    if (paginationElement) {
        paginationElement.innerHTML = `Showing ${offset + 1}–${to} of ${totalRecords} results ${searchForTxt}`;
    }

    // Upates sort order selected text
    const sortElement = document.querySelector('.sort-order-selected');
    if (sortElement) {
        sortElement.innerHTML = sortText;
    }
}

// // Initialize Quill editor with Snow theme
// const initQuillEditor = (editorClass, toolbarClass) => {
//     return new Quill(editorClass, {
//         modules: {
//             toolbar: toolbarClass,
//             history: {
//                 delay: 1000,
//                 maxStack: 500,
//                 userOnly: true
//             }
//         },
//         theme: 'snow',
//         placeholder: 'Type your answer here...',
//         formats: ['bold', 'italic', 'underline', 'strike', 'list', 'bullet', 'indent'],
//     });
// };

// // Initialize editors
// const editor1 = initQuillEditor('.editor-1', '.toolbar-1');
// const editor2 = initQuillEditor('.editor-2', '.toolbar-2');
// const editor3 = initQuillEditor('.editor-3', '.toolbar-3');

function scrollToSection(sectionId, event) {
    event.preventDefault();
    const section = document.getElementById(sectionId);
    const offset = 100; // Adjust this value to control the scroll position

    // Add active state to clicked link
    const links = document.querySelectorAll('.scroll-container a');
    links.forEach(link => link.classList.remove('bg-black-1200', 'text-white'));
    event.target.classList.add('bg-black-1200', 'text-white');

    // Animate scroll with smooth behavior
    window.scrollTo({
        top: section.offsetTop - offset,
        behavior: 'smooth'
    });

    // Add fade-in animation to section heading
    const heading = section.querySelector('h6');
    if (heading) {
        heading.style.opacity = '0';
        heading.style.transform = 'translateY(-20px)';
        setTimeout(() => {
            heading.style.transition = 'all 0.5s ease';
            heading.style.opacity = '1';
            heading.style.transform = 'translateY(0)';
        }, 300);
    }
}

// Initialize first item as active
document.addEventListener('DOMContentLoaded', () => {

    const firstLink = document.querySelector('.scroll-container a');
    if (firstLink) {
        firstLink.classList.add('bg-black-1200', 'text-white');
    }
});
window.scrollToSection = scrollToSection;

// Utility function to bind action handlers to elements
function initActionHandlers() {
    // Handle action buttons with data attributes
    document.addEventListener('click', async (e) => {
        const button = e.target.closest('[data-action]');
        if (!button) return;

        e.preventDefault();

        const action = button.getAttribute('data-action');
        const module = button.getAttribute('data-module');
        const modelId = button.getAttribute('data-model-id');

        if (!action || !module || !modelId) {
            console.error('Missing required data attributes for action button');
            return;
        }

        // Get additional data from data attributes
        const additionalData = {};
        Array.from(button.attributes).forEach(attr => {
            if (attr.name.startsWith('data-extra-')) {
                const key = attr.name.replace('data-extra-', '');
                additionalData[key] = attr.value;
            }
        });

        // Handle the action
        await Alpine.store('actions').handleAction(action, module, modelId, additionalData, button);
    });
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', initActionHandlers);

// Submit blog category
function submitBlogSearch() {
    const form = document.getElementById("jorunalSearch");
    const searchField = document.getElementById("querySearch");

    // Only initialize if form and search field exist on the page
    if (!form || !searchField) {
        return;
    }

    const searchButton = form.querySelector('.btn_search_posts');
    const formAction = form.getAttribute('action');

    // Single function to handle form submission
    function handleSearchSubmission() {
        const query = searchField.value.trim();
        const params = new URLSearchParams(new FormData(form)).toString();
        window.location.href = `${formAction}?q=${encodeURIComponent(query)}&${params}`;
    }

    // Event listener for form submit
    // form.addEventListener('submit', (e) => {
    //     e.preventDefault();
    //     handleSearchSubmission();
    // });

    // // Event listener for search button click
    // if (searchButton) {

    //     searchButton.addEventListener('click', (e) => {
    //         e.preventDefault();
    //         handleSearchSubmission();
    //     });
    // }

    // Event listener for search field enter key
    searchField.addEventListener('keypress', (e) => {
        if (e.key === "Enter") {
            e.preventDefault();
            handleSearchSubmission();
        }
    });
}

document.addEventListener('DOMContentLoaded', submitBlogSearch);

window.heroSearchSubmit = function heroSearchSubmit() {
    const form = document.getElementById("heroSearchForm");

    // Only initialize if form exists on the page
    if (!form) {
        return;
    }

    const qInput = form.querySelector('input[name="q"]');
    if (!qInput) {
        return;
    }

    const q = qInput.value;
    // window.location.href = `?q=${q}`;
}

document.addEventListener('DOMContentLoaded', heroSearchSubmit);

document.addEventListener("DOMContentLoaded", function () {
    const steps = document.querySelectorAll(".lead-wizard-form .step");
    const stepLinks = document.querySelectorAll(".step-link");
    const nextBtns = document.querySelectorAll(".next-btn");
    let currentStep = 0;

    function updateSidebar(index) {
        stepLinks.forEach((link, i) => {
            link.classList.remove("active", "complete");

            if (i === index) {
                link.classList.add("active");
            } else if (i < index) {
                link.classList.add("complete");
            }
        });
    }

    function showStep(index) {
        steps.forEach((step, i) => {
            step.style.display = i === index ? "block" : "none";
        });

        updateSidebar(index);
    }

    // Initialize
    showStep(currentStep);

    // Continue Button (Livewire gated: do not auto-advance here)
    nextBtns.forEach((btn) => {
        btn.addEventListener("click", function (e) {
            // Livewire will handle and emit events to advance
        });
    });

    // Sidebar Step Click (disabled: Livewire controls progression)
    stepLinks.forEach((link, i) => {
        link.addEventListener("click", function (e) {
            e.preventDefault();
        });
    });

    // Sync UI with Livewire step events
    window.addEventListener('leadform-go-next', () => {
        if (currentStep < steps.length - 1) {
            currentStep++;
            showStep(currentStep);
        }
    });
    window.addEventListener('leadform-go-prev', () => {
        if (currentStep > 0) {
            currentStep--;
            showStep(currentStep);
        }
    });

    // Also refresh wizard if Livewire triggers a generic refresh event
    window.addEventListener('leadform-refresh', () => {
        try { window.initLeadWizard && window.initLeadWizard(); } catch (e) { }
    });
});

document.querySelectorAll(".start-btn").forEach(function (btn) {
    btn.addEventListener("click", function () {
        const reviewMain = this.closest(".review-main");
        if (reviewMain) reviewMain.style.display = "none";

        const wizardForm = document.querySelector(".lead-wizard-form");
        if (wizardForm) wizardForm.style.display = "block";
    });
});
document.querySelectorAll(".filter-map-btn").forEach(function (btn) {
    btn.addEventListener("click", function () {
        document.querySelector(".map-filter").classList.add("active");
        document.querySelector(".overly").classList.add("active");
    });
});

document.querySelectorAll(".filter-close, .overly").forEach(function (btn) {
    btn.addEventListener("click", function () {
        document.querySelector(".map-filter").classList.remove("active");
        document.querySelector(".overly").classList.remove("active");
    });
});