/**
 * ESP Directory Google Maps Utilities
 * Collection of reusable Google Maps functions for the ESP Directory application
 */

const ESPGoogleMaps = {
    // Current map instance
    currentMap: null,
    // Current markers on the map
    currentMarkers: [],
    // Default marker icon URL (string)
    defaultMarkerIcon: null,
    // Active marker icon URL (string)
    activeMarkerIcon: null,
    // InfoWindow instance
    currentInfoWindow: null,
    // Flag to track if info window was opened by click vs hover
    _infoWindowOpenedByClick: false,

    /**
     * Helper function to get a valid URL string from an icon input.
     * Icon input might be a string URL or an object (e.g., { url: 'path/to/icon.png' }).
     * @param {*} iconInput - The icon input (string or object).
     * @param {string} [iconName='icon'] - Name of the icon for logging.
     * @returns {string|null} The URL string or null if invalid.
     */
    getIconUrl: function (iconInput, iconName = 'icon') {
        if (typeof iconInput === 'string') {
            return iconInput;
        }
        if (typeof iconInput === 'object' && iconInput !== null && typeof iconInput.url === 'string') {
            return iconInput.url;
        }
        if (iconInput) {
        }
        return null;
    },

    /**
     * Initialize Google Maps Autocomplete on an input field
     *
     * @param {Object} options - Configuration options for the autocomplete
     * @param {string|HTMLElement} options.inputField - The input field selector or element for autocomplete
     * @param {string|HTMLElement} [options.latitudeField] - The input field selector or element for latitude
     * @param {string|HTMLElement} [options.longitudeField] - The input field selector or element for longitude
     * @param {string|HTMLElement} [options.cityField] - The input field selector or element for city
     * @param {string|HTMLElement} [options.stateField] - The input field selector or element for state/province
     * @param {string|HTMLElement} [options.stateCodeField] - The input field selector or element for state/province code
     * @param {string|HTMLElement} [options.countryField] - The input field selector or element for country
     * @param {string|HTMLElement} [options.countryCodeField] - The input field selector or element for country code
     * @param {string} [options.country] - Country code to restrict results (e.g., 'us', 'ca')
     * @param {Array} [options.types=['geocode']] - Types of predictions to return
     * @param {Function} [options.onPlaceChanged] - Callback function when a place is selected
     * @returns {google.maps.places.Autocomplete|null} - The autocomplete instance or null if initialization failed
     */
    initAutocomplete: function (options) {
        // Ensure options is an object
        options = options || {};

        // Get input element
        const inputElement = typeof options.inputField === 'string'
            ? document.querySelector(options.inputField)
            : options.inputField;

        // Validate input element
        if (!inputElement) {
            console.error('initAutocomplete: Input element not found:', options.inputField);
            return null;
        }

        // Check if Google Maps Places API is available
        if (!window.google || !window.google.maps || !window.google.maps.places) {
            console.error('initAutocomplete: Google Maps Places API not loaded');
            return null;
        }

        // Get latitude and longitude elements if provided
        const latElement = options.latitudeField ? (
            typeof options.latitudeField === 'string'
                ? document.querySelector(options.latitudeField)
                : options.latitudeField
        ) : null;

        const lngElement = options.longitudeField ? (
            typeof options.longitudeField === 'string'
                ? document.querySelector(options.longitudeField)
                : options.longitudeField
        ) : null;

        // Get city, state, and country elements if provided
        const cityElement = options.cityField ? (
            typeof options.cityField === 'string'
                ? document.querySelector(options.cityField)
                : options.cityField
        ) : null;

        const stateElement = options.stateField ? (
            typeof options.stateField === 'string'
                ? document.querySelector(options.stateField)
                : options.stateField
        ) : null;

        const stateCodeElement = options.stateCodeField ? (
            typeof options.stateCodeField === 'string'
                ? document.querySelector(options.stateCodeField)
                : options.stateCodeField
        ) : null;

        const countryElement = options.countryField ? (
            typeof options.countryField === 'string'
                ? document.querySelector(options.countryField)
                : options.countryField
        ) : null;

        const countryCodeElement = options.countryCodeField ? (
            typeof options.countryCodeField === 'string'
                ? document.querySelector(options.countryCodeField)
                : options.countryCodeField
        ) : null;

        // Set up autocomplete options
        const autocompleteOptions = {
            types: options.types || ['geocode'],
            fields: ['address_components', 'geometry', 'name', 'formatted_address']
        };

        // Add country restriction if specified
        if (options.country && options.country.length === 2) {
            autocompleteOptions.componentRestrictions = {
                country: options.country
            };
        }

        // Before creating a new autocomplete instance, remove any existing pac-containers
        // This helps prevent multiple dropdowns from appearing
        document.querySelectorAll('.pac-container').forEach(container => {
            container.remove();
        });

        // Initialize the autocomplete
        const autocomplete = new google.maps.places.Autocomplete(
            inputElement,
            autocompleteOptions
        );

        // Handle place selection
        autocomplete.addListener('place_changed', function () {
            const place = autocomplete.getPlace();

            // Ensure we have valid place data
            if (!place || !place.geometry) {
                return;
            }

            // Update latitude and longitude fields if provided
            if (latElement && place.geometry.location) {
                latElement.value = place.geometry.location.lat();
            }

            if (lngElement && place.geometry.location) {
                lngElement.value = place.geometry.location.lng();
            }

            // Extract city, state, and country from address_components
            if (place.address_components && (cityElement || stateElement || stateCodeElement || countryElement || countryCodeElement)) {
                let city = '';
                let state = '';
                let stateCode = '';
                let country = '';
                let countryCode = '';

                place.address_components.forEach(component => {
                    const types = component.types;

                    // Extract city (locality or sublocality)
                    if (types.includes('locality')) {
                        city = component.long_name;
                    } else if (!city && types.includes('sublocality')) {
                        city = component.long_name;
                    } else if (!city && types.includes('administrative_area_level_2')) {
                        city = component.long_name;
                    }

                    // Extract state/province
                    if (types.includes('administrative_area_level_1')) {
                        state = component.long_name;
                        // Only set state code if short_name is not too long (max 10 chars)
                        stateCode = component.short_name.length <= 10 ? component.short_name : '';
                    }

                    // Extract country
                    if (types.includes('country')) {
                        country = component.long_name;
                        countryCode = component.short_name;
                    }
                });

                // Update the fields with extracted values
                if (cityElement) {
                    cityElement.value = city;
                }

                if (stateElement) {
                    stateElement.value = state;
                }

                if (stateCodeElement) {
                    stateCodeElement.value = stateCode;
                }

                if (countryElement) {
                    countryElement.value = country;
                }

                if (countryCodeElement) {
                    countryCodeElement.value = countryCode;
                }
            }

            // Call custom callback if provided
            if (typeof options.onPlaceChanged === 'function') {
                options.onPlaceChanged(place);
            }
        });

        return autocomplete;
    },

    /**
     * Initialize a Google Map in a container
     * 
     * @param {Object} options - Configuration options for the map
     * @param {string|HTMLElement} options.mapContainer - The container selector or element for the map
     * @param {number} [options.latitude=0] - Initial latitude for map center
     * @param {number} [options.longitude=0] - Initial longitude for map center
     * @param {number} [options.zoom=12] - Initial zoom level
     * @param {boolean} [options.scrollwheel=false] - Whether to enable scroll wheel zoom
     * @param {Array} [options.markers=[]] - Array of marker objects with lat, lng properties
     * @param {Array} [options.styles=[]] - Custom map styles to apply
     * @param {Function} [options.onMapLoad] - Callback function when map is loaded
     * @returns {google.maps.Map|null} - The map instance or null if initialization failed
     */
    initMap: function (options) {
        // Default options
        const defaults = {
            latitude: 0,
            longitude: 0,
            zoom: 12,
            scrollwheel: false,
            markers: [],
            styles: [],
            minZoom: null,
            maxZoom: null,
            mapId: null // Added mapId to defaults
        };

        // Merge defaults with provided options
        const config = { ...defaults, ...options };

        // Get map container element
        const mapElement = typeof config.mapContainer === 'string'
            ? document.querySelector(config.mapContainer)
            : config.mapContainer;

        // Validate map container
        if (!mapElement) {
            return null;
        }

        // Create map instance
        const mapOptions = {
            center: { lat: parseFloat(config.latitude), lng: parseFloat(config.longitude) },
            zoom: config.zoom,
            scrollwheel: config.scrollwheel,
            mapTypeControl: true,
            streetViewControl: true,
            fullscreenControl: true,
            minZoom: config.minZoom || null,
            maxZoom: config.maxZoom || null,
            styles: config.styles || [], // Apply custom styles if provided
            mapId: config.mapId // Use mapId from config
        };

        // IMPORTANT: Check if mapId is provided when Advanced Markers might be used.
        if (!mapOptions.mapId) {
            // A mapId is required for Advanced Markers.
            // Using a DEMO_MAP_ID for now, replace with your actual Map ID for production.
            mapOptions.mapId = 'DEMO_MAP_ID';
        }

        const map = new google.maps.Map(mapElement, mapOptions);

        // Add markers if provided (legacy marker handling, Advanced Markers are handled separately)
        const markers = [];
        if (config.markers && config.markers.length) {
            config.markers.forEach(markerData => {
                if (markerData.lat && markerData.lng) {
                    const marker = new google.maps.Marker({
                        position: {
                            lat: parseFloat(markerData.lat),
                            lng: parseFloat(markerData.lng)
                        },
                        map: map,
                        title: markerData.title || '',
                        icon: markerData.icon || null
                    });

                    // Add info window if content is provided
                    if (markerData.infoWindow) {
                        const infoWindow = new google.maps.InfoWindow({
                            content: markerData.infoWindow
                        });

                        marker.addListener('click', function () {
                            infoWindow.open(map, marker);
                        });

                        // Auto-open info window if specified
                        if (markerData.openInfoWindow) {
                            infoWindow.open(map, marker);
                        }
                    }

                    markers.push(marker);
                }
            });

            // Fit bounds to markers if we have multiple markers
            if (markers.length > 1) {
                const bounds = new google.maps.LatLngBounds();
                markers.forEach(marker => {
                    bounds.extend(marker.getPosition());
                });
                map.fitBounds(bounds);
            }
        }

        // Call custom callback if provided
        if (typeof config.onMapLoad === 'function') {
            config.onMapLoad(map, markers);
        }

        return map;
    },

    /**
     * Calculate distance between two points
     * 
     * @param {number} lat1 - Latitude of point 1
     * @param {number} lng1 - Longitude of point 1
     * @param {number} lat2 - Latitude of point 2
     * @param {number} lng2 - Longitude of point 2
     * @param {string} [unit='mi'] - Unit of measurement ('mi' for miles, 'km' for kilometers)
     * @returns {number} - Distance between the points in specified unit
     */
    calculateDistance: function (lat1, lng1, lat2, lng2, unit = 'mi') {
        if ((lat1 === lat2) && (lng1 === lng2)) {
            return 0;
        }

        const radlat1 = Math.PI * lat1 / 180;
        const radlat2 = Math.PI * lat2 / 180;
        const theta = lng1 - lng2;
        const radtheta = Math.PI * theta / 180;
        let dist = Math.sin(radlat1) * Math.sin(radlat2) + Math.cos(radlat1) * Math.cos(radlat2) * Math.cos(radtheta);
        if (dist > 1) {
            dist = 1;
        }
        dist = Math.acos(dist);
        dist = dist * 180 / Math.PI;
        dist = dist * 60 * 1.1515; // Distance in miles

        if (unit === 'km') {
            dist = dist * 1.609344; // Convert to kilometers
        }

        return parseFloat(dist.toFixed(2));
    },

    /**
     * Initialize Google Maps functionality for city input in the search sidebar
     * @param {string} componentId - Livewire component ID for proper element selection
     * @param {string} country - Country code to restrict search results
     * @returns {google.maps.places.Autocomplete|null} - The autocomplete instance or null
     */
    initSearchSidebarAutocomplete: function (componentId, country) {
        if (!componentId) {
            return null;
        }

        // Get the input and coordinate fields
        const inputField = `#city-input-${componentId}`;
        const latField = `#lat-${componentId}`;
        const lngField = `#lng-${componentId}`;

        // Create options object for initAutocomplete
        const options = {
            inputField: inputField,
            latitudeField: latField,
            longitudeField: lngField,
            types: ['geocode'],
            country: country,
            onPlaceChanged: function (place) {
                if (!place) return;

                // Use the formatted address directly instead of just the city
                let locationName = place.formatted_address || place.name || '';

                // Get latitude and longitude values
                const lat = place.geometry && place.geometry.location ? place.geometry.location.lat() : null;
                const lng = place.geometry && place.geometry.location ? place.geometry.location.lng() : null;

                // Update the input field with the formatted address
                const inputElement = document.querySelector(inputField);
                if (inputElement) {
                    inputElement.value = locationName;
                }

                // Update lat/lng fields directly
                const latElement = document.querySelector(latField);
                const lngElement = document.querySelector(lngField);

                if (latElement && lat !== null) {
                    latElement.value = lat;
                }

                if (lngElement && lng !== null) {
                    lngElement.value = lng;
                }

                // Use a small delay to ensure the DOM is updated before Livewire processes the changes
                setTimeout(() => {
                    // Update the Livewire component directly
                    try {
                        const livewireComponent = window.Livewire.find(componentId);
                        if (livewireComponent) {
                            // Update city value
                            livewireComponent.set('city', locationName);

                            // Update lat/lng values in Livewire
                            if (lat !== null && lng !== null) {
                                livewireComponent.set('lat', lat);
                                livewireComponent.set('lng', lng);

                                // Dispatch a custom event after setting values to trigger any listeners
                                document.dispatchEvent(new CustomEvent('location-selected', {
                                    detail: { componentId, lat, lng, city: locationName }
                                }));
                            }
                        }
                    } catch (e) {
                        // Handle errors silently
                    }

                    // Now trigger input events after Livewire updates
                    if (inputElement) {
                        inputElement.dispatchEvent(new Event('input', { bubbles: true }));
                    }

                    if (latElement && lat !== null) {
                        latElement.dispatchEvent(new Event('input', { bubbles: true }));
                    }

                    if (lngElement && lng !== null) {
                        lngElement.dispatchEvent(new Event('input', { bubbles: true }));
                    }
                }, 50);
            }
        };

        // Initialize autocomplete
        return this.initAutocomplete(options);
    },

    /**
     * Initialize the member directory map
     * 
     * @param {Object} options - Configuration options for the member map
     * @param {string} options.mapContainer - Selector for the map container
     * @param {number} [options.defaultLat=37.7749] - Default latitude for initial center (San Francisco)
     * @param {number} [options.defaultLng=-122.4194] - Default longitude for initial center (San Francisco)
     * @param {number} [options.defaultZoom=4] - Default zoom level
     * @param {string} [options.defaultMarkerIcon] - URL to the default marker icon
     * @param {string} [options.activeMarkerIcon] - URL to the active marker icon
     * @param {Array} [options.mapStyles] - Custom map styles to apply
     * @param {Object} [options.mapOptions] - Additional Google Maps options
     * @param {function} [options.onMarkerClick] - Callback for marker click events
     * @returns {google.maps.Map|null} - The map instance or null if initialization failed
     */
    initMemberDirectoryMap: function (options) {
        // Default options
        const defaults = {
            mapContainer: '#esp-directory-map',
            defaultLat: 37.7749,
            defaultLng: -122.4194,
            defaultZoom: 4,
            defaultMarkerIcon: null,
            activeMarkerIcon: null,
            mapStyles: [],
            mapOptions: { mapId: 'DEMO_MAP_ID' }, // Ensure mapId is here. Replace DEMO_MAP_ID with your actual Map ID.
            onMarkerClick: null
        };

        // Merge provided options with defaults
        const config = { ...defaults, ...options };

        // Store marker icons for later use
        this.defaultMarkerIcon = config.defaultMarkerIcon;
        this.activeMarkerIcon = config.activeMarkerIcon;

        // Initialize a basic map
        // IMPORTANT: For AdvancedMarkerElement, the map must be initialized with a mapId.
        // Example: config.mapOptions.mapId = 'YOUR_MAP_ID';
        // Ensure this.initMap forwards or includes mapId in its options.
        const mapOptionsToPass = {
            mapContainer: config.mapContainer,
            latitude: config.defaultLat,
            longitude: config.defaultLng,
            zoom: config.defaultZoom,
            scrollwheel: config.mapOptions.scrollwheel !== undefined ? config.mapOptions.scrollwheel : true,
            styles: config.mapStyles || [],
            minZoom: config.mapOptions.minZoom,
            maxZoom: config.mapOptions.maxZoom,
            mapId: config.mapOptions.mapId, // Pass mapId to initMap
            onMapLoad: (map) => {
                // Store the map instance for later use
                this.currentMap = map;

                // Add additional map controls or settings here
                map.setOptions({
                    mapTypeControl: config.mapOptions.mapTypeControl !== undefined ? config.mapOptions.mapTypeControl : true,
                    streetViewControl: config.mapOptions.streetViewControl !== undefined ? config.mapOptions.streetViewControl : true,
                    fullscreenControl: config.mapOptions.fullscreenControl !== undefined ? config.mapOptions.fullscreenControl : true,
                    zoomControl: config.mapOptions.zoomControl !== undefined ? config.mapOptions.zoomControl : true,
                    minZoom: config.mapOptions.minZoom !== undefined ? config.mapOptions.minZoom : null,
                    maxZoom: config.mapOptions.maxZoom !== undefined ? config.mapOptions.maxZoom : null,
                    styles: config.mapStyles || [] // Apply custom styles if provided
                });

                // Handle any custom onMapLoad callback
                if (typeof config.onMapLoad === 'function') {
                    config.onMapLoad(map);
                }
            }
        };

        return this.initMap(mapOptionsToPass);
    },

    /**
     * Update member markers on the map based on provided member data
     * 
     * IMPORTANT: This function now uses google.maps.marker.AdvancedMarkerElement.
     * Ensure the Google Maps API is loaded with the "marker" library.
     * e.g., <script src="https://maps.googleapis.com/maps/api/js?key=YOUR_API_KEY&libraries=marker&callback=initMap"></script>
     * Also, the map must be initialized with a `mapId`.
     * 
     * @param {Array} members - Array of member objects with location data
     * @param {boolean} [fitBounds=true] - Whether to fit the map bounds to include all markers
     * @param {boolean} [clearExisting=true] - Whether to clear existing markers
     */
    updateMemberMarkers: function (members, fitBounds = true, clearExisting = true) {
        // Ensure we have a map instance
        if (!this.currentMap) {
            return;
        }

        if (!members || !Array.isArray(members)) {
            return;
        }

        // Clear existing markers if requested
        if (clearExisting) {
            this.clearMarkers();
        }

        // Create a bounds object to fit all markers
        const bounds = new google.maps.LatLngBounds();
        let validMarkerCount = 0;

        // Process each member and add a marker if they have location data
        members.forEach(member => {
            // Check for valid location data
            if (!member.lat || !member.lng) {
                return;
            }

            const lat = parseFloat(member.lat);
            const lng = parseFloat(member.lng);

            // Skip invalid coordinates
            if (isNaN(lat) || isNaN(lng)) {
                return;
            }

            // Create marker position
            const position = { lat, lng };

            // Extend bounds with this position
            bounds.extend(position);
            validMarkerCount++;

            // Create content for the AdvancedMarkerElement (image icon)
            const markerContent = document.createElement('img');
            const defaultIconUrl = this.getIconUrl(this.defaultMarkerIcon, 'defaultMarkerIcon');
            if (defaultIconUrl) {
                markerContent.src = defaultIconUrl;
            }

            // Ensure the member has an ID and UUID for consistent lookup
            if (!member.uuid && member.id) {
                member.uuid = member.id;
            } else if (!member.id && member.uuid) {
                member.id = member.uuid;
            }

            // Prepare marker options for AdvancedMarkerElement
            const markerOptions = {
                position: position,
                map: this.currentMap,
                title: member.name || '',
                content: markerContent
            };

            // Create the AdvancedMarkerElement
            const marker = new google.maps.marker.AdvancedMarkerElement(markerOptions);

            // Explicitly attach member data to marker
            marker.member = { ...member }; // Create a copy to avoid reference issues

            this.currentMarkers.push(marker);

            // Create info window content from member data
            let infoWindowContent = this.createMemberInfoWindowContent(member);

            // Add click event to the marker
            marker.addListener('click', () => {
                const memberData = marker.member; // memberData is attached to the marker

                if (memberData) {
                    // Call highlightMemberMarker with isClick = true.
                    // This function will handle:
                    // - Setting the active icon for the clicked marker (and resetting others).
                    // - Closing any previous info window.
                    // - Creating and opening a new info window for this marker.
                    // - Setting _infoWindowOpenedByClick = true.
                    // - Panning the map to center the marker (because isClick = true).
                    // - Dispatching the 'map-marker-clicked' event.
                    this.highlightMemberMarker(memberData.id || memberData.uuid, true);

                    // Call the custom onMarkerClick callback if it was provided during map initialization.
                    // This is preserved from the original listener as highlightMemberMarker does not call it.
                    if (typeof this.onMarkerClick === 'function') {
                        this.onMarkerClick(memberData, marker);
                    }
                }
            });
        });

        // Fit bounds if we have valid markers and fitBounds is true
        if (validMarkerCount > 0 && fitBounds) {
            if (validMarkerCount === 1) {
                // For a single marker, set center and a specific zoom level
                this.currentMap.setCenter(bounds.getCenter());
                this.currentMap.setZoom(15); // Zoom level for a single marker
            } else {
                // For multiple markers, fit bounds with padding
                const padding = 50; // Padding in pixels
                try {
                    this.currentMap.fitBounds(bounds, padding);
                } catch (e) {
                    // Handle error silently if fitBounds fails
                }
            }
        } else if (validMarkerCount === 0) {
            // Center the map on a default location
            this.currentMap.setCenter({ lat: 37.7749, lng: -122.4194 }); // Default to San Francisco
            this.currentMap.setZoom(3); // Default zoom level for continental view
        }

        // Force a resize event on the map to ensure proper rendering
        try {
            google.maps.event.trigger(this.currentMap, 'resize');
        } catch (e) {
            // Handle error silently
        }
    },

    /**
     * Clear all markers from the map
     */
    clearMarkers: function () {
        if (this.currentMarkers.length > 0) {
            // Remove each marker from the map
            this.currentMarkers.forEach(marker => {
                // For AdvancedMarkerElement, set marker.map = null
                // For legacy google.maps.Marker, it was marker.setMap(null)
                marker.map = null;
            });

            // Clear the markers array
            this.currentMarkers = [];
        }

        // Close any open info window
        if (this.currentInfoWindow) {
            this.currentInfoWindow.close();
            this.currentInfoWindow = null;
        }
    },

    /**
     * Create HTML content for the member info window
     * 
     * @param {Object} member - Member data object
     * @returns {string} HTML content for the info window
     */
    createMemberInfoWindowContent: function (member) {
        // If pre-rendered HTML is available from the server, use it
        if (member.infoWindowHtml) {
            return member.infoWindowHtml;
        }
        return 'no data';
    },

    /**
     * Generate HTML for star ratings
     * 
     * @param {number} rating - Rating value (0-5)
     * @returns {string} HTML for star rating display
     */
    generateStars: function (rating) {
        const fullStars = Math.floor(rating);
        const halfStar = rating % 1 >= 0.5;
        const emptyStars = 5 - fullStars - (halfStar ? 1 : 0);

        let starsHTML = '';

        // Full stars
        for (let i = 0; i < fullStars; i++) {
            starsHTML += '<svg class="w-3 h-3 text-yellow-500" fill="currentColor" viewBox="0 0 20 20"><path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"></path></svg>';
        }

        // Half star
        if (halfStar) {
            starsHTML += '<svg class="w-3 h-3 text-yellow-500" fill="currentColor" viewBox="0 0 20 20"><path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" clip-path="inset(0 50% 0 0)"></path></svg>';
        }

        // Empty stars
        for (let i = 0; i < emptyStars; i++) {
            starsHTML += '<svg class="w-3 h-3 text-gray-300" fill="currentColor" viewBox="0 0 20 20"><path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"></path></svg>';
        }

        return starsHTML;
    },

    /**
     * Handle when a member is selected in the listing view to highlight on map
     * 
     * @param {string} memberId - The ID of the member to highlight
     * @param {boolean} isClick - Whether this was triggered by a click (vs hover)
     */
    highlightMemberMarker: function (memberId, isClick = false) {
        if (!this.currentMap || !this.currentMarkers.length) {
            return;
        }

        // Track if this was a click or hover event
        this._infoWindowOpenedByClick = isClick;

        // Find the marker for this member - enhanced with debugging
        let marker = null;

        // First try exact match on uuid
        marker = this.currentMarkers.find(m =>
            m.member && m.member.uuid === memberId
        );

        // If not found, try with id
        if (!marker) {
            marker = this.currentMarkers.find(m =>
                m.member && m.member.id === memberId
            );
        }

        // Last resort, try string comparison (in case of type differences)
        if (!marker) {
            marker = this.currentMarkers.find(m =>
                m.member && (String(m.member.uuid) === String(memberId) || String(m.member.id) === String(memberId))
            );
        }

        if (marker) {
            // Change icon to active if available (for AdvancedMarkerElement, change content.src)
            const activeIconUrl = this.getIconUrl(this.activeMarkerIcon, 'activeMarkerIcon for highlight');
            if (activeIconUrl && marker.content && marker.content.tagName === 'IMG') {
                // Reset all other markers to default icon first
                this.currentMarkers.forEach(m => {
                    if (m !== marker && m.content && m.content.tagName === 'IMG') {
                        const dIconUrl = this.getIconUrl(this.defaultMarkerIcon, 'defaultMarkerIcon for highlight reset');
                        if (dIconUrl) m.content.src = dIconUrl;
                    }
                });
                // Set this marker to active icon
                marker.content.src = activeIconUrl;
            }

            // Close any existing info window first
            if (this.currentInfoWindow) {
                this.currentInfoWindow.close();
            }

            // Create and open the info window
            if (marker.member) {
                const infoWindowContent = this.createMemberInfoWindowContent(marker.member);

                // Configure the info window with explicit pixel offset
                const infoWindow = new google.maps.InfoWindow({
                    content: infoWindowContent,
                    disableAutoPan: true, // We'll handle panning manually for better UX
                    pixelOffset: new google.maps.Size(0, -10) // Small offset to position above marker
                });

                // Get current map bounds and projection
                const bounds = this.currentMap.getBounds();
                const projection = this.currentMap.getProjection();

                if (bounds && projection) {
                    // Convert marker position to pixel coordinates
                    const markerPosition = marker.position;
                    const scale = Math.pow(2, this.currentMap.getZoom());
                    const nw = projection.fromLatLngToPoint(
                        new google.maps.LatLng(
                            bounds.getNorthEast().lat(),
                            bounds.getSouthWest().lng()
                        )
                    );
                    const worldPoint = projection.fromLatLngToPoint(markerPosition);
                    const markerPixel = new google.maps.Point(
                        Math.floor((worldPoint.x - nw.x) * scale),
                        Math.floor((worldPoint.y - nw.y) * scale)
                    );

                    // Get map container dimensions
                    const mapDiv = this.currentMap.getDiv();
                    const mapWidth = mapDiv.offsetWidth;
                    const mapHeight = mapDiv.offsetHeight;

                    // Estimated info window dimensions based on our CSS - responsive
                    const infoWindowHeight = 220; // Adjusted height in pixels
                    let infoWindowWidth = 320;  // Default width in pixels
                    let viewportPadding = 20; // Added padding in pixels

                    // Adjust dimensions for mobile
                    if (mapWidth <= 768) {
                        infoWindowWidth = Math.min(320, mapWidth - 40);
                        viewportPadding = 20;
                    }
                    if (mapWidth <= 375) {
                        infoWindowWidth = Math.min(280, mapWidth - 20);
                        viewportPadding = 10;
                    }

                    // Determine adjustments needed
                    let needsPanning = false;
                    let panX = 0;
                    let panY = 0;

                    // Check top edge - info window extends above map
                    // Ensure space for infoWindowHeight + viewportPadding from the top
                    if (markerPixel.y < infoWindowHeight + viewportPadding) {
                        needsPanning = true;
                        panY = (infoWindowHeight + viewportPadding) - markerPixel.y; // Amount to pan down
                    }

                    // Check right edge - info window extends beyond right of map
                    // Ensure space from the right edge (mapWidth - viewportPadding)
                    if (markerPixel.x + (infoWindowWidth / 2) > mapWidth - viewportPadding) {
                        needsPanning = true;
                        // Amount to pan left (negative)
                        panX = (mapWidth - viewportPadding) - (markerPixel.x + (infoWindowWidth / 2));
                    }

                    // Check left edge - info window extends beyond left of map
                    // Ensure space from the left edge (viewportPadding)
                    if (markerPixel.x - (infoWindowWidth / 2) < viewportPadding) {
                        needsPanning = true;
                        // Amount to pan right (positive)
                        panX = viewportPadding - (markerPixel.x - (infoWindowWidth / 2));
                    }

                    // Pan the map if adjustments are needed or this was a click
                    if (needsPanning || isClick) {
                        const center = this.currentMap.getCenter();
                        const worldCenter = projection.fromLatLngToPoint(center);
                        const newCenterPoint = new google.maps.Point(
                            worldCenter.x - (panX / scale),
                            worldCenter.y - (panY / scale)
                        );

                        const newCenter = projection.fromPointToLatLng(newCenterPoint);

                        // Different behavior for click vs hover
                        if (isClick) {
                            // For clicks, center on marker with smooth animation
                            this.currentMap.panTo(markerPosition);
                        } else if (needsPanning) {
                            // For hover, only adjust as needed with a gentler animation
                            this.currentMap.panTo(newCenter);
                        }
                    }
                } else {
                    // Fallback if bounds/projection not available: just center on marker for clicks
                    if (isClick) {
                        this.currentMap.panTo(marker.position);
                    }
                }

                // Open the info window
                infoWindow.open(this.currentMap, marker);
                this.currentInfoWindow = infoWindow;

                // Add a listener for when the InfoWindow's DOM is ready
                google.maps.event.addListener(infoWindow, 'domready', () => {
                    // Hide the Google Maps close button
                    const googleMapsCloseButton = document.querySelector('.gm-ui-hover-effect');
                    if (googleMapsCloseButton) {
                        googleMapsCloseButton.style.display = 'none';
                    }

                    // Alternative selectors for different versions of Google Maps
                    const alternativeCloseButtons = document.querySelectorAll('[title="Close"], .gm-style-iw-chr, .gm-style-iw-c button');
                    alternativeCloseButtons.forEach(btn => {
                        if (btn && btn.getAttribute && (btn.getAttribute('title') === 'Close' || btn.classList.contains('gm-ui-hover-effect'))) {
                            btn.style.display = 'none';
                        }
                    });

                    // Wait a bit for the DOM to be fully rendered
                    setTimeout(() => {
                        const closeButton = document.querySelector('.infowindow-close-button');
                        if (closeButton) {
                            // Remove any existing listeners to prevent duplicates
                            const newCloseButton = closeButton.cloneNode(true);
                            closeButton.parentNode.replaceChild(newCloseButton, closeButton);

                            // Add click listener
                            newCloseButton.addEventListener('click', (e) => {
                                e.stopPropagation();
                                e.preventDefault();
                                infoWindow.close(); // This will trigger the 'closeclick' event below
                            });
                        }
                    }, 50);
                });

                // Add a listener for when the InfoWindow is closed (by any means)
                google.maps.event.addListener(infoWindow, 'closeclick', () => {
                    this._infoWindowOpenedByClick = false; // Reset click flag
                    if (this.currentInfoWindow === infoWindow) { // Ensure it's the current one
                        this.currentInfoWindow = null;
                    }
                    // Reset the icon of the marker associated with this closed info window
                    if (marker.content && marker.content.tagName === 'IMG') {
                        const defaultIconUrl = this.getIconUrl(this.defaultMarkerIcon, 'defaultMarkerIcon for close');
                        if (defaultIconUrl) marker.content.src = defaultIconUrl;
                    }
                });

                // Make the info window stay open with a slight delay to ensure proper positioning
                setTimeout(() => {
                    if (this.currentInfoWindow) {
                        this.currentInfoWindow.open(this.currentMap, marker);
                    }
                }, 100);
            }

            // Dispatch a custom event for external components to react - only on click, not hover
            if (isClick && marker.member) {
                document.dispatchEvent(new CustomEvent('map-marker-clicked', {
                    detail: { member: marker.member, markerId: marker.member.uuid || marker.member.id }
                }));
            }
        }
    },

    /**
     * Check if marker is too close to the edge of the viewport
     * 
     * @param {google.maps.LatLng} position Marker position
     * @param {google.maps.LatLngBounds} bounds Current map bounds
     * @returns {boolean} True if marker is too close to edge
     */
    isMarkerTooCloseToEdge: function (position, bounds) {
        // Get the bounds of the viewport
        const ne = bounds.getNorthEast();
        const sw = bounds.getSouthWest();

        // Calculate the viewport dimensions in degrees
        const latSpan = ne.lat() - sw.lat();
        const lngSpan = ne.lng() - sw.lng();

        // Define the buffer (20% of the viewport)
        const buffer = 0.2;

        // Check if the marker is within the buffer zone
        const tooFarNorth = position.lat() > (ne.lat() - buffer * latSpan);
        const tooFarSouth = position.lat() < (sw.lat() + buffer * latSpan);
        const tooFarEast = position.lng() > (ne.lng() - buffer * lngSpan);
        const tooFarWest = position.lng() < (sw.lng() + buffer * lngSpan);

        return tooFarNorth || tooFarSouth || tooFarEast || tooFarWest;
    },

    /**
     * Calculate the optimal position to pan to
     * 
     * @param {google.maps.LatLng} currentCenter Current map center
     * @param {google.maps.LatLng} markerPosition Marker position
     * @param {google.maps.LatLngBounds} bounds Current map bounds
     * @returns {google.maps.LatLng} Position to pan to
     */
    calculateOptimalPanPosition: function (currentCenter, markerPosition, bounds) {
        // Get the bounds of the viewport
        const ne = bounds.getNorthEast();
        const sw = bounds.getSouthWest();

        // Calculate the viewport dimensions in degrees
        const latSpan = ne.lat() - sw.lat();
        const lngSpan = ne.lng() - sw.lng();

        // Define the target position (30% of the way from current center to marker)
        const targetLat = currentCenter.lat() + (markerPosition.lat() - currentCenter.lat()) * 0.3;
        const targetLng = currentCenter.lng() + (markerPosition.lng() - currentCenter.lng()) * 0.3;

        return new google.maps.LatLng(targetLat, targetLng);
    },
};

// Make the object globally available
window.ESPGoogleMaps = ESPGoogleMaps;

// Export for ES modules
export default ESPGoogleMaps; 