import os
import re
from pathlib import Path

def get_identifier(filename):
    """Extract identifier from filename (text after second '-' from right)"""
    parts = filename.split(' - ')
    if len(parts) >= 3:
        return parts[-2]
    return None

def extract_parameters_content(xml_content):
    """Extract content between Parameters tags preserving formatting"""
    start_tag = "<Parameters>"
    end_tag = "</Parameters>"
    start_idx = xml_content.find(start_tag)
    end_idx = xml_content.find(end_tag) + len(end_tag)
    if start_idx == -1 or end_idx == -1:
        return None
    return xml_content[start_idx:end_idx]

def replace_parameters_content(xml_content, new_parameters):
    """Replace content between Parameters tags"""
    pattern = r'<Parameters>.*?</Parameters>'
    return re.sub(pattern, new_parameters, xml_content, flags=re.DOTALL)

def copy_parameters(source_path, target_path):
    """Copy Parameters section from source to target preserving formatting"""
    try:
        # Read source file
        with open(source_path, 'r', encoding='utf-8') as f:
            source_content = f.read()
            
        # Extract Parameters content
        parameters_content = extract_parameters_content(source_content)
        if parameters_content is None:
            print(f"No Parameters tag found in source file: {source_path}")
            return False
            
        # Read target file
        with open(target_path, 'r', encoding='utf-8') as f:
            target_content = f.read()
            
        # Replace Parameters content
        new_content = replace_parameters_content(target_content, parameters_content)
            
        # Write updated content back to target file
        with open(target_path, 'w', encoding='utf-8', newline='') as f:
            f.write(new_content)
            
        return True
            
    except Exception as e:
        print(f"Error processing files: {e}")
        return False

def main():
    # Define paths
    base_dir = Path('.')
    source_dir = base_dir / '27' / 'AddPars'
    target_dir = base_dir / '28' / 'AddPars'
    
    # Process each XML file in source directory
    for source_file in source_dir.glob('*.xml'):
        source_identifier = get_identifier(source_file.name)
        if not source_identifier:
            continue
            
        # Find matching files in target directory
        matching_targets = list(target_dir.glob(f'*- {source_identifier} -*.xml'))
        
        if len(matching_targets) == 0:
            print(f"No matching target file found for: {source_file.name}")
            continue
            
        if len(matching_targets) > 1:
            print(f"\nMultiple matching files found for: {source_file.name}")
            print("Available target files:")
            for i, target in enumerate(matching_targets, 1):
                print(f"{i}. {target.name}")
            
            choice = input("Enter the number of the target file (or press Enter to skip): ")
            if not choice.strip():
                continue
                
            try:
                target_path = matching_targets[int(choice) - 1]
            except (ValueError, IndexError):
                print("Invalid choice, skipping...")
                continue
        else:
            target_path = matching_targets[0]
        
        # Copy parameters
        if copy_parameters(source_file, target_path):
            print(f"Successfully copied parameters from {source_file.name} to {target_path.name}")
        else:
            print(f"Failed to copy parameters for {source_file.name}")

if __name__ == '__main__':
    main()