//
//  AddBookIntent.swift
//  BooksShelfControlActionStarter
//
//  Created by Giugliano Antonella on 16/04/25.
//

import AppIntents
import SwiftUI

struct AddBookIntent: AppIntent {
    
    static var title: LocalizedStringResource = "Add a Book to your Shelf"
    
    @Parameter(title: "Title", requestValueDialog: IntentDialog("What's the title?"))
    var title: String
    
    @Parameter(title: "Author", requestValueDialog: IntentDialog("Who's the author?"))
    var author: String
    
    @Parameter(title: "Genre", requestValueDialog: IntentDialog("What is the genre?"))
    var genre: Genre
    
    @Parameter(title: "Type of book", requestValueDialog: IntentDialog("Is it an e-book or a physical one?"))
    var type: BookContentType?
    
    @Parameter(title: "Include Type", requestValueDialog: IntentDialog("Do you want to include the type of book?"))
    var isTypeIncluded: IsTypeIncluded
    
    @Dependency
    private var navigationManager: NavigationManager
    
    
    @MainActor
    func perform() async throws -> some IntentResult & ShowsSnippetView & ProvidesDialog {
        
        var chosenType: BookContentType?
        
        if isTypeIncluded == IsTypeIncluded.yes {
            
            chosenType = try await $type.requestDisambiguation(among: BookContentType.allCases, dialog: "Which type?")
        }
        let newBook = Book(title: title, author: author, genre: genre, contentType: type)
        do{
            try await DataManager.createNewBook(book: newBook)
        }catch{
            fatalError("Unable to save the book")
        }
        
        let dialog = IntentDialog("New book added on your shelf.")
        return .result(dialog: dialog){
            
            IntentSnippetResultView(book: newBook)
        }
    }
}

enum IsTypeIncluded: String, CaseIterable, Identifiable, AppEnum {
    static var typeDisplayRepresentation = TypeDisplayRepresentation(name: "Type Included")
    
    static var caseDisplayRepresentations: [Self: DisplayRepresentation] = [
        .yes: DisplayRepresentation(title: "Yes"),
        .no: DisplayRepresentation(title: "No"),
        
    ]
    case yes = "Yes"
    case no = "No"
    var id: Self { self }
}

struct IntentSnippetResultView: View{
    var book: Book
    var body: some View{
        HStack{
            Image(systemName: "books.vertical.fill")
                .imageScale(.large)
                .foregroundStyle(.tint)
            VStack(alignment: .leading){
                Text("You have a new book on your shelf!")
                Divider()
                Section {
                    Text("Title: \(book.title)")
                    Text("Author: \(book.author!)")
                }
                Section {
                    Text("Genre: \(book.genre!.rawValue)")
                    Text("Type: \(book.contentType!.rawValue)")
                }
            }
        }.padding()
    }
}
