//
//  BooksListView.swift
//  BooksShelfControlActionStarter
//
//  Created by Giugliano Antonella on 04/03/25.
//

import SwiftUI
import SwiftData

struct BooksListView: View {
    
    @Query var books: [Book]
    
    @Environment(\.modelContext) private var modelContext
    @Environment(NavigationManager.self) private var navigation
    
    @State var openAddBook: Bool = false
    
    var filteredBooks: [Book] {
            if navigation.searchText.isEmpty {
                return books
            } else {
                let searchText = navigation.searchText
                return books.filter {book in book.title.contains(searchText) || book.author?.contains(searchText) ?? false}
            }
        }
    
    var body: some View {
        
        @Bindable var navigation = navigation
        
        NavigationStack(path: $navigation.navigationPath) {
            List {
                ForEach(filteredBooks) { book in
                    NavigationLink(value: book) {
                        HStack {
                            Image(systemName: "book")
                                .imageScale(.large)
                                .foregroundStyle(.tint)
                            Text(book.title)
                        }
                    }
                }
                .onDelete(perform: deleteBook(at:))
            }
            // Empty state of the list
            .overlay {
                if books.isEmpty {
                    ContentUnavailableView {
                        Label("It is empty!", systemImage: "book")
                    } description: {
                        Text("You don't have any books registered on your bookshelf yet.")
                    } actions: {
                        Button("Add new book") {
                            openAddBook.toggle()
                        }
                        .buttonStyle(BorderedButtonStyle())
                    }
                }
            }
            
            .navigationTitle("Books Shelf")
            .toolbar {
                ToolbarItem(placement: .topBarTrailing) {
                    Button("Add new book", systemImage: "plus") {
                        openAddBook.toggle()
                    }
                    .labelStyle(.iconOnly)
                }
            }
            // Add a new book
            .sheet(isPresented: $openAddBook) {
                AddBookView().environment(navigation)
            }
            
            // Show book details
            .navigationDestination(for: Book.self) { book in
                BookDetailView(book: book)
            }
            
            .searchable(text: $navigation.searchText)
        }
    }
    
    private func deleteBook(at offsets: IndexSet) {
        for index in offsets {
            let book = books[index]
            Task {
                try await DataManager.deleteBook(book: book)
            }
        }
    }
}

#Preview("Empty State") {
    BooksListView()
        .environment(NavigationManager())
}
